/*
   LPR.C -  a utility to allow you to pipe into the DOS PRINT command
													R. Brittain  May 3, 1989
	Spool files are created in directory $SPOOLDIR  (default is C:/SPOOL)
	File creation template is LPR???.$$$ where ??? are filled in with digits
	000-999

	This program is
					Copyright  1989  by Richard Brittain
	Permission is granted to freely use and distribute this program so long as
	this copyright notice is maintained intact.  This software may not be sold.

	Revision history:
	1.0 	6/18/89 Released to usenet as version 1.0   6/18/89
	1.1 	9/19/89	changed exit code for "print.com not loaded" to 2 for use
			in batch files. Fixed bug in prepending directory to given
			filename in root directory, or if drive is specified.  Changed
			wildcard expansion in setargv() to handle "/" in paths, and also
			to treat as wildcards filespecs which have [] but not * or ?, for
			better use with the "wildunix" TSR.
			File specs containing combinations of "." and ".." are now handled
			correctly
*/

#include <stdio.h>
#include <dir.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include <alloc.h>
#include <ctype.h>
#include <dos.h>
#include <errno.h>
#include <time.h>

#define SPOOLDIR  "C:\\SPOOL"    /* where we will put the temporary print files */
#define SPOOLMASK "LPR???.$$$"   /* template for the print file names */
#define CONTROLZ '\032'
#define FF		 '\014'
#define	DOSPRINT 	1
#define MULTIPLEX	0x2F
#define QELEN		64			 /* length in bytes of a print queue entry */
#define MAGIC		'*'

/* function prototypes */

char 		*getfilename(char*);
int	  		index(char*, char*);
int   		delete_files(char*);
char 		*mkformat(char*, int*);
int  		strcount(char*, char);
int			check(char*, char*);
int			size(char*);
int			print_loaded(void);
int			print_file(char*);
char far 	*print_queue(void);
char		*localcopy(char far *);
char		*localcopy2(char far *);
void		release_print_queue(void);
int			lp_stdin(int, int);
int 		add_time_to_queue(void);
char far 	*farcopy2(char far *, char *);
char far 	*strcpyfar2(char far *, char far *);
int			print_job_rush(int);
int			resolve_path(char *,char *);
char		*parent(char *);
char		*endchar(char *);
char 		*catdir(char *, char*);
char 		*translit(char *, char, char);

main(argc,argv)
int 	argc;
char	**argv;
{
/*	Main program processes command line arguments and options only       */
/*  The exit code is 0 is no errors are encountered, 1 if any argument   */
/*  is in error.  All arguments are processed even if some give an error */
/*  Exit code is 2 if PRINT.COM is not loaded                            */

	char	*p, pathname[64] ;
	int		i, files=0, exitcode=0, verbose=0, badargs=0, rush=0, job, result;
	char    version[]="1.1";
	char	copyright[]="Copyright  1989  by  Richard Brittain";
	char	usage[]="Usage: command | lpr [-v] [-h] [-p] [file [-] ....]";
	char	helptext[]="\
Spool files are created in C:\SPOOL or in the directory given in SPOOLDIR\n\
Filename arguments (wildcards may be used) are added to print queue.\n\
Argument \"-\" means stdin (if redirected or piped). Stdin is assumed if no\n\
arguments are given.\n\
Options:\n\
	-h display help message\n\
	-v verbose mode\n\
	-p priority mode - add job(s) to head of print queue\n\n\
PRINT should be loaded prior to using lpr";
	char	permiss[]="Permission is granted to freely use and distribute "
					  "this program";

/*  check that the print spooler is loaded */
	if (!print_loaded()) {
		printf("lpr: DOS print spooler is not loaded\n");
		exit(2);
	}

/* process the command line arguments */
	for (i=1; i<argc; i++) {
		p = argv[i];
		if ( (*p == '-') && (strlen(p) > 1)) {
			do {
				p++;
				switch(*p) {
					case '\0':
						break;
					case 'h':  /* help */
						printf("LPR Version %s of %s\n",version,__DATE__);
						printf("                 %s\n\n",copyright);
						printf("%s\n\n",usage);
						printf("%s\n\n",helptext);
						printf("%s\n",permiss);
						exit(1);
					case 'v':  /* verbose mode */
						verbose = 1;
						break;
					case 'p':  /* priority */
						rush = 1;
						break;
					default:
						printf("lpr: unrecognised option %c\n",*p);
						badargs = 1;
				}
			} while (*p);
		} else {
			files++;
			if (strcmp(p,"-") == 0) {
				if (isatty(fileno(stdin))) {
					printf("lpr: cannot take input from a terminal\n");
					exitcode = 1;
					badargs  = 1;
				} else {
					exitcode = lp_stdin(verbose,rush);
				}
			} else {
				/* turn the specified file name into a full path */
				result = resolve_path(pathname,p);
				if (result == -1) {
					printf("lpr: invalid drive specified\n");
					continue;
				}

				if (print_file(pathname) < 0) {
					switch (_doserrno) {
						case ENOMEM:
							printf("lpr: spool queue is full: failed to add %s\n",
							pathname);
							break;
						case EACCES:
							printf("lpr: cannot print a directory: %s\n",
							pathname);
							break;
						default:
							printf("lpr: %s ",pathname);
							perror("");
							break;
					}
					exitcode = 1;
				} else {
					job = add_time_to_queue();
					if (rush) print_job_rush(job);
					if (verbose) {
						if (rush)
							printf("lpr: adding %s to head of print queue\n",pathname);
						else
							printf("lpr: adding %s to print queue\n",pathname);
					}
				}
			}
		}
	}

/* 	If we have not done any files, do stdin, if not a tty */
/* 	check that our input is not a terminal - this isn't really an error but */
/*  it isn't useful to allow it and may be confusing */
	if (files == 0) {
    	if (isatty(fileno(stdin))) {
			printf("lpr: cannot take input from a terminal\n");
			badargs  = 1;
			exitcode = 1;
		} else {
			exitcode = lp_stdin(verbose,rush);
		}
	}
	release_print_queue();
	if (badargs) {
		printf("%s\n",usage);
		if (verbose) printf("\n%s\n",helptext);
	}
	exit(exitcode);
}

int lp_stdin(int verbose, int rush)
{
/*  Routine to take stdin and print it via the DOS PRINT program        */
/*  Makes a file in a spool directory and then passes the name to PRINT */
	FILE	*tfp;
	char	spoolname[13],  spooldir[64]=SPOOLDIR;
	char	cwd[MAXDIR], *sp, sdrive[MAXDRIVE];
	int		c, i, lastc, fileok=1, cdisk, sdisk, job;
	struct	dfree  dtable;

/*  find out our cwd and default drive */
	cdisk = getdisk();
	getcwd(cwd, MAXDIR);

/*  now change to the spool directory, possibly on another drive */
	if ((sp=getenv("SPOOLDIR")) != NULL) strcpy(spooldir,sp);
	fnsplit(spooldir,sdrive,NULL,NULL,NULL);
	sdisk = toupper(sdrive[0])-'A';
	if (setdisk(sdisk) < sdisk+1) perror("lpr");
	if (chdir(spooldir) != 0) {
		printf("lpr: cannot access spool directory %s \n",spooldir);
		setdisk(cdisk);
		return(1);
	}

/*  get a unique filename for the spool file */
	strcpy(spoolname, getfilename(SPOOLMASK));

	if ((tfp=fopen(spoolname,"w"))==NULL) {
		printf("lpr: cannot create spool file %s \n",spoolname );
		setdisk(cdisk);
		chdir(cwd);
		return(1);
	}

/* copy stdin to the new spool file */
	c = fgetchar();
	if (feof(stdin)) {
		fileok = 0;
		if (verbose) printf("lpr: input file is empty\n") ;
	}
	rewind(stdin);
	for (;;) {
		lastc =  c;
		c = fgetchar();
		if (feof(stdin)) {
			break ;
		} else if (ferror(stdin)) {
			printf("lpr: cannot read stdin\n") ;
			fileok = 0;
			break ;
		}
		if (putc(c,tfp) == EOF) {
			printf("lpr: write error on spool file: ") ;
			getdfree( 0, &dtable);
			if (dtable.df_avail == 0)
				printf("disc is full\n");
			else
				perror("");
			fileok = 0;
			break ;
		}
	}

	if (!fileok) {
		fclose(tfp);
	} else {
/*  	we don't want to copy a trailing formfeed because PRINT will add one */
		if (lastc == FF ) {
			fseek( tfp, ftell(tfp)-1, SEEK_SET );
			putc( CONTROLZ, tfp);
		}
		fclose(tfp);

/*  	now pass this name to print for insertion into the queue */
		i = strlen(spooldir)-1;
		if (spooldir[i] != '\\') strcat(spooldir,"\\");
		strcat(spooldir,spoolname);
		if (print_file(spooldir) < 0) {
			if (_doserrno == ENOMEM)
				printf("lpr: spool queue is full: failed to add %s\n",spooldir);
			else
				perror("lpr");
		} else {
			job = add_time_to_queue();
			if (rush) print_job_rush(job);
			if (verbose) {
				if (rush)
					printf("lpr: adding %s (stdin) to head of print queue\n",spooldir);
				else
					printf("lpr: adding %s (stdin) to print queue\n",spooldir);
			}
		}
	}

/*  we are done, but first clean up the spool directory by deleting any */
/*  files matching SPOOLMASK that do NOT appear in the print queue  */
	delete_files(SPOOLMASK);

/*  cd back to where we came from and quit */
	setdisk(cdisk);
	chdir(cwd);
	return(0);
}

char *getfilename(pattern)
char *pattern;
/* generate a unique filename in the current directory,                 */
/* based on the given pattern.  Patterns look like bbbb???.eee          */
/* where bbbb is a "base name" of 1-7 characters, ??? will be replaced  */
/* by sequential numbers, up to all 9's, and eee is an optional         */
/* constant extension.  Thus "LPR???.$$$" will generate the lowest      */
/* file in the range LPR000.$$$ to LPR999.$$$ not currently existing    */
/* Turning the pattern into a format for sprintf is done by mkformat()  */
/* The maximum allowable number of "?" is 4, any more are ignored		*/
{
	int 	i, maxtries;
	char	format[30];
	static	char	filename[13];

	if (strcpy(format,mkformat(pattern, &maxtries))==NULL) return(NULL);
	i = 0;
	do {
		sprintf( filename, format, i);
		i++;
		if ( i > maxtries) return( NULL ) ;
	} while (access(filename,0) == 0) ;

	return ((char *)filename);
}

int index(str,sub)
char	*str, *sub;
{
/*  return index into <str> of first occurrence of <sub>, if any */
	char *i;
	if ((i=strstr(str,sub)) != NULL)
		return(i-str+1);
	else
		return(0);
}

int delete_files( mask )
char *mask;
{
/*  The current print queue is obtained by a call to print_queue, and this */
/*  is checked against the files matching the filename mask, so that we do */
/*  not delete files in the queue.  This routine returns the number of     */
/*  files deleted.														   */

	int 	i=0, n=0;
	char	*m, far *ps, far *p;
	struct	ffblk fblock;

	if (*(ps=print_queue()) == NULL) return(0) ;
	/* the print queue contains fixed length (64 bytes) entries */

	/*  make an upper case copy of the mask for comparing  */
	m = strupr(strdup(mask)) ;
	if (findfirst(m, &fblock, 0) == 0) {
		p = ps;
		i = 0;
		while (*p != NULL) {  /* cycle through all entries in queue */
			if (strstr(localcopy(p),fblock.ff_name) != NULL) {
				/* this file is in the queue - don't search further */
				i++;
				break;
			}
			p += QELEN;
		}
		if ((i == 0) && check(m,fblock.ff_name)) {
			if (remove(fblock.ff_name) == 0) n++ ;
		}
	}
	while (findnext(&fblock) == 0) {
		p = ps;
		i = 0;
		while (*p != NULL) {  /* cycle through all entries in queue */
			if (strstr(localcopy(p),fblock.ff_name) != NULL) {
				/* this file is in the queue - don't search further */
				i++;
				break;
			}
			p += QELEN;
		}
		if ((i == 0) && check(m,fblock.ff_name)) {
			if (remove(fblock.ff_name) == 0) n++ ;
		}
	}
	release_print_queue();
	return(n);
}

char *localcopy(char far *s)
/* Make a copy of a string pointed to by a far pointer.  The space is mallocd */
{
	char far *p, *l, *r ;
	int		 i=0 ;
	p = s;
	while (*p++ != NULL) { i++; }
	r = l = (char *)malloc(i+1);
	p = s;
	while (*p != NULL) { *l++ = *p++ ;}
	*l = '\0';
	return(r);
}

char *localcopy2(char far *s)
/* Make a copy of a string pointed to by a far pointer             */
/* Look for magic character and optionally copy second string also */
/* Never copy more than QELEN total characters					   */
{
	char far *p, *l, *r ;
	int		 i=0 ;
	p = s;
	while (*p++ != NULL) { i++; }
	if (*(++p) == MAGIC) {
		i++;
		while (*p++ != NULL) { if (i++ >= QELEN-1) break; }
	}
	r = l = (char *)malloc(i+1);
	p = s;
	i = 0;
	while (*p != NULL) { *l++ = *p++; i++;}
	*l = '\0';
	if (*(++p) == MAGIC) {
		l++;
		i++;
		while (*p != NULL) { *l++ = *p++;  if (i++ >= QELEN) break; }
		*l = '\0';
	}
	return(r);
}

int strcount( s, c)
char *s, c;
{
	int i,j=0;
	for (i=0;i<strlen(s);i++) if (*(s+i) == c) j++ ;
	return(j);
}

char  *mkformat(pattern,maxtries)
char  *pattern;
int   *maxtries;
{
	int 	ln, lp, lf, lb, i;
	char    fmtspec[12], ext[4];
	static 	char format[20];

	lp = strlen(pattern);
	lf = index(pattern,".")-1 ;
	lb = index(pattern,"?")-1 ;
	lf = (lf > 0) ? lf : lp ;
	ln = min(4,strcount(pattern,'?'));
	if (ln < 1) return(NULL) ;
	strncpy(format,pattern,lb);
	format[lb] = '\0';
	sprintf(fmtspec,"%s%d%s","%0",ln,"d.") ;
	strcat (format, fmtspec);
	if (lf+1 < lp) {
		strncpy(ext,&pattern[lf+1],3);
		ext[3] = '\0';
	} else {
		strcpy( ext,"   ");
	}
	strcat (format, ext);

	*maxtries = 10;
	for (i=2;i<=ln;i++) *maxtries = *maxtries*10;
	*maxtries--;

	return(format);
}

int size(s)
char *s;
{
/*  return the length of string <s> after stripping trailing blanks */
	int l;
	l = strlen(s);
	while ((*(s+l-1) == ' ') && (l>0)) l-- ;
	return(l);
}

int check(mask, str)
char *mask, *str;
{
/*  verify that <mask> and <str> are the same, except that for */
/*  each "?" in <mask> there is a corresponding DIGIT in <str>	*/

	int i,len;
	char c;

	if ((len=size(mask)) != size(str)) 	return(0);

	for  (i=0; i<len; i++) {
		if ((c=*(mask+i)) == '?') {
			if (!isdigit(*(str+i))) return(0) ;
		} else {
			if (*(str+i) != c) return(0) ;
		}
	}
	return(1) ;
}

int print_loaded()
{
/* find out if the resident portion of the DOS print spooler is loaded */
	union REGS  regs;
	regs.h.ah = DOSPRINT;
	regs.h.al = 0;
	int86(MULTIPLEX,&regs,&regs);
	if (regs.h.al == 0xFF)
		return(1);		/* print is installed */
	else
		return(0);		/* print is not installed */
}

int print_file(filename)
char *filename;
{
/* submit a filename to PRINT for addition to the queue */
/* The returned value is the value returned in AL by PRINT */
/*	AL = 158 - filename accepted and now printing */
/*	AL = 1	 - filename accepted and added to queue */
/*  if the carry flag indicates an error, -1 is returned */

	static struct submit_packet {
		unsigned char level;
		char far 	  *p;
	} submit ;
	union 	REGS  	regs;

	regs.h.ah = DOSPRINT;
	regs.h.al = 1;                      /* AL=1 means we are submitting a file          */
	submit.level = 0;					/* don't know what this is for but it is needed */
	submit.p = MK_FP(_DS, filename);	/* make a far pointer to the file name          */
	regs.x.dx = (char near *)&submit;   /* put the segment offset into dx               */
	/* NOTE: the above line gives a pointer warning */
	int86(MULTIPLEX,&regs,&regs);       /* the ds register is set already               */
	if (regs.x.cflag != 0)
		return(-1);						/* error */
	else
		return (regs.h.al);
}

char far *print_queue()
{
/* Get a far pointer to the start of the list of files in the print queue    */
/* This also freezes the queue (and suspends operation of the print spooler) */
/* until another call is made to any PRINT function */
	union 	REGS 	regs;
	struct	SREGS 	sregs;
	regs.h.ah = DOSPRINT;
	regs.h.al = 4;
	int86x(MULTIPLEX,&regs,&regs,&sregs);
	return (MK_FP(sregs.ds, regs.x.si));
	/* DS:SI is a far pointer to the queue */
}

void release_print_queue()
{
/*  This is a do-nothing print function that serves only to release the print queue */
	union 	REGS 	regs;
	regs.h.ah = DOSPRINT;
	regs.h.al = 5;
	int86(MULTIPLEX,&regs,&regs);
}

int add_time_to_queue()
{
/*  Find the last entry in the print queue, and add the current time in */
/*  hh:mm behind the filename, if there is room.  Return the job number */
	char	queuentry[QELEN], far *fp;
	int		i=0, l;
	time_t 	now;

	if (*(fp=print_queue()) == NULL) return(i);
	/* find last entry in queue */
	while (*fp) { i++; fp += QELEN; };
	fp -= QELEN;

	strcpy(queuentry, localcopy(fp));
	l = strlen(queuentry);
	if (QELEN - l >= 11) {		/* put the current time after the filename, if room */
		time(&now);
		queuentry[l+1] = MAGIC;
		strncpy(&queuentry[l+2], ctime(&now)+11, 8);
		queuentry[l+10] = '\0';
		farcopy2(fp, queuentry);
	}
	return(i);
}

char far *farcopy2(char far *dest, char *src)
/* Copy a string from a near pointer into a far pointer (already allocated) */
/* If the magic character follows, copy a second string 					*/
/* Never copy more than QELEN total characters								*/
{
	char far *p, *s ;
	int	 i=0;
	p = dest;
	s = src;
	while (*s != NULL) { *p++ = *s++; if (i++ >= QELEN) break; }
	*p = '\0';
	if (*(++s) == MAGIC) {
		p++;
		while (*s != NULL) { *p++ = *s++; if (i++ >= QELEN) break; }
		*p = '\0';
	}
	return(dest);
}

char far *strcpyfar2(char far *dest, char far *src)
/* Copy a string from one far pointer into another      */
/* If the magic character follows, copy a second string */
/* Never copy more than QELEN total characters			*/
{
	char far *p, far *s ;
	int	 i=0;
	p = dest;
	s = src;
	while (*s != NULL) { *p++ = *s++; if (i++ >= QELEN) break; }
	*p = '\0';
	if (*(++s) == MAGIC) {
		p++;
		while (*s != NULL) { *p++ = *s++; if (i++ >= QELEN) break; }
		*p = '\0';
	}
	return(dest);
}

int print_job_rush(job)
int job;
{
/*  Get the print queue, look for entry <job>, and place it at the head */
/*  of the queue */
/*  Return -1 if <job> is invalid else return 0 */
	char far *ps, *l;
	int		i;

	if (*(ps=print_queue()) != NULL) {
		i = 1;
		while (*ps != NULL) {
			if (i == job) break;
			ps += QELEN;
			i++;
		}
		if (*ps == NULL) return(-1) ;  /* there were not that many jobs in queue */
		if (i <= 2) return(0) ;        /* job is already at head */
		l = localcopy2(ps);			   /* make a copy of entry <job> */
		for (i=1; i<=job-2; i++) {     /* copy each job into the following slot */
			strcpyfar2(ps, ps-QELEN);
			ps -= QELEN;
		}
		farcopy2(ps,l);				   /* put the priority job into slot 2 */
		return(0);
	} else {
		return(-1);                    /* no jobs in queue */
	}
}

int resolve_path(pathname,p)
char *pathname, *p;
{
/*
 * Given a filespec p, which may contain any combination of drive, directory,
 * filename, "." or "..", we must derive the complete path name to pass to
 * print.
 */
	int  flags, disk;
	static char newdrive[MAXDRIVE], filname[MAXFILE], filext[MAXEXT];
	static char newdir[MAXDIR], tempdir[MAXDIR], currdir[MAXDIR];
	char *s;

	flags = fnsplit(p,newdrive,newdir,filname,filext);
	translit(newdir,'/','\\');

	if (flags & DRIVE) {
		/* drive specified, check it out, return(-1) if bad */
		/* getcurdir() does not put a leading "\" so we glue one on        */
		currdir[0] = '\\';
		if (getcurdir(toupper(newdrive[0])-'A'+1,&currdir[1]) != 0)	return(-1);
	} else {
		/* get our default drive, and the current directory */
		disk = getdisk();
		strcpy(newdrive," :");
		newdrive[0] = (char) (disk+'A');
		currdir[0] = '\\';
		getcurdir(disk+1,&currdir[1]);
	}

	/* currdir now contains the current directory for the desired drive */
	/* currdir has a leading \ and no trailing \ */

	if (flags & DIRECTORY) {
		/* newdir directory specified, but it may be in terms of . and .. */
		/* build a new directory in tempdir, then copy it into newdir     */
		s = newdir;

		if (newdir[0] == '\\') {
			/* directory begins at root */
			tempdir[0] = '\0';
		} else {
			/* directory begins with current directory */
			strcpy(tempdir,currdir);
			strcat(tempdir,"\\");
		}

		while (*s) {
			if (strncmp(s,".\\",2) == 0) {
				s += 2;
			} else if (strncmp(s,"..\\",3) == 0) {
				parent(tempdir);
				s += 3;
			} else {
				s = catdir(tempdir,s);
			}
		}
		strcpy(newdir,tempdir);
	} else {
		/* no directory specified, use the default one for the given drive */
		strcpy(newdir,currdir);
	}
	/* all done, merge drive and directory into a full path */
	fnmerge(pathname,newdrive,newdir,filname,filext);
	return(0);
}

char *parent(dir)
char *dir;
{
/*  Modify dir by truncating the last directory, thereby giving the parent */
/*  The string will always end in "\" */

	char *s ;
	if ((s = strrchr(dir,'\\')) == NULL) {
		/* no directory separators at all */
		strcpy(dir,"\\");
	} else {
		if (*endchar(dir) == '\\') *(endchar(dir)) = '\0';
		if ((s = strrchr(dir,'\\')) == NULL)
			strcpy(dir,"\\");
		else
			*(++s) = '\0';
	}
	return(dir);
}

char *catdir(d,s)
char *d, *s;
{
/*
 * Copy a directory from s onto the end of d, include any leading \ and
 * exclude any trailing \
 * Returns a pointer to first character of next directory (or null)
 */
	while (*d) d++;
	if (*s == '\\') *d++ = *s++;
	while (*s != '\\' && *s != '\0') *d++ = *s++;
	*d = '\0';
	return(s);
}

char  *endchar(s)
char *s;
{
/* return a pointer to the last character in s */
	if (*s == NULL) return(s);
	while (*s) s++;
	return(--s);
}

char *translit(s,c1,c2)
char *s, c1, c2;
{
/*  turn all c1 into c2 in string s, in situ. Return pointer to start of s */
	char *p;
	p = s;
	while (*s) {
		if (*s == c1) *s = c2;
		s++;
	}
	return(p);
}