/*
//
// msg.c
//
// Processing messages (informational, warning, and error)
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

/*
// Because the Linux Gnu compiler doesnt support stdarg.h (at least, the
// copy I have doesnt), htp uses varargs.h, which doesnt support full
// prototyping of variable-argument functions ... this causes tons o' warning
// messages on DOS (and I suspect most ANSI) compilers.  So, to alleviate
// the problem, if this macro is defined, the K&R prototype is used,
// otherwise all other .c modules get the ANSI prototype
*/
#define DUMB_MSG_PROTOTYPE

#include "htp.h"

/* symbols used to express severity of message */
const char *severitySymbol[3] =
{
    "[-]",
    "[*]",
    "[!]",
};

/* the current severity level (any message of this level or higher is printed */
uint severityLevel = MSG_INFO;

/* set severity level */
void SetMessageSeverityLevel(uint level)
{
    if(level > MSG_ERROR)
    {
        severityLevel = MSG_ERROR;
    }
    else
    {
        severityLevel = level;
    }
}

void HtpMsg(va_alist) va_dcl
{
    va_list argptr;
    char *str;
    uint level;
    TEXTFILE *textFile;
    const char *format;

    /* allocate room for string ... 1K should be large enough, but a more */
    /* deterministic method would be better */
    if((str = AllocMemory(1024)) == NULL)
    {
        return;
    }

    /* convert variable arguments into single string */
    va_start(argptr);
    level = va_arg(argptr, uint);
    textFile = va_arg(argptr, TEXTFILE *);
    format = va_arg(argptr, const char *);
    vsprintf(str, format, argptr);
    va_end(argptr);

    /* check severity level */
    if(level < severityLevel)
    {
        FreeMemory(str);
        return;
    }

    /* print the standard message header followed by formatted message */
    printf("%s ", severitySymbol[level]);
    if(textFile != NULL)
    {
        printf("%s line %d: ", textFile->name, textFile->lineNumber);
    }
    printf("%s\n", str);

    /* free the string and exit */
    FreeMemory(str);
}

FILE *debugMsgFile = NULL;

void DebugInit(const char *debugMsgFilename)
{
    remove(debugMsgFilename);
    debugMsgFile = fopen(debugMsgFilename, "at");

    if(debugMsgFile == NULL)
    {
        printf("htp: unable to open debug file \"%s\", aborting\n",
            debugMsgFilename);
        exit(1);
    }
}

void DebugTerminate(void)
{
    if(debugMsgFile != NULL)
    {
        fclose(debugMsgFile);
        debugMsgFile = NULL;
    }
}

/*
// DebugMsg
//
// DebugMsg is a helper function to (a) log a formatted string to disk and
// (b) display the same string to the screen.  This function is called by
// DEBUG_PRINT, which is normally removed from the code in a final release
// build.
//
// Because this debug information is really most useful when the program
// is coughing up blood, this function will flush contents every time,
// doing whatever it can to get the debug string to disk.  Slow, but that's
// what a debug build is for.
//
// Important: this file can't use the abstraction present in textfile.c
// because *that* module uses DEBUG_PRINT as well ... (same for suballoc.c
// functions)
//
*/
void DebugMsg(va_alist) va_dcl
{
    va_list argptr;
    char *str;
    const char *format;

    /* 1K should be enough, but no guarantees */
    if((str = malloc(1024)) == NULL)
    {
        /* !! gaaak */
        return;
    }

    /* convert variable arguments into single string */
    va_start(argptr);
    format = va_arg(argptr, const char *);
    vsprintf(str, format, argptr);
    va_end(argptr);

    /* write the string to disk */
    fprintf(debugMsgFile, str);

    /* flush it out to disk */
    fflush(debugMsgFile);

#if 0
    /* write it to screen */
    printf(str);
#endif

    free(str);
}

