/*
 *  win32lock.c
 *
 *  This contains locking code for WAIS, called from irfiles.c.
 *
 *  (C) Copyright 1994 The University Court of the University of Edinburgh
 *
 *  Author:  Chris Adie <C.J.Adie@ed.ac.uk> 
 *
 */

/******************************************************************************/
/* INCLUDE FILES                                                              */
/******************************************************************************/

#include <windows.h>
#include "cutil.h"

/******************************************************************************/
/* CONSTANT DEFINITIONS                                                       */
/******************************************************************************/

/* This section deliberately left blank */

/******************************************************************************/
/* MACRO FUNCTION DEFINITIONS                                                 */
/******************************************************************************/

#define MAXQUERIES  5       /* Max no of simultaneour queries on a database */

/******************************************************************************/
/* TYPE DEFINITIONS                                                           */
/******************************************************************************/

/* This section deliberately left blank */

/******************************************************************************/
/* GLOBAL VARIABLES AND FUNCTIONS IMPORTED                                    */
/******************************************************************************/

/* This section deliberately left blank */
/* Consider whether these should appear in a header file */

/******************************************************************************/
/* GLOBAL VARIABLES EXPORTED                                                  */
/******************************************************************************/

/* This section deliberately left blank */

/******************************************************************************/
/* VARIABLES PRIVATE TO THIS FILE                                             */
/******************************************************************************/

static int LockCount = 0;   /* How many times the Query sem has been acquired */
static HANDLE hIndex = INVALID_HANDLE_VALUE;   /* Handle of Index mutex */ 
static HANDLE hQuery = INVALID_HANDLE_VALUE;   /* Handle of Query sem */ 

/******************************************************************************/
/* FUNCTIONS PRIVATE TO THIS FILE                                             */
/******************************************************************************/

/* This section deliberately left blank */

/******************************************************************************/
/* GLOBAL FUNCTIONS EXPORTED                                                  */
/******************************************************************************/

void ResetInterlock(void) {
    if (hIndex!=INVALID_HANDLE_VALUE) {
        /* We own the Index mutex */
        CloseHandle(hIndex);
        hIndex = INVALID_HANDLE_VALUE;
    }
    if (hQuery!=INVALID_HANDLE_VALUE) {
        if (LockCount>0) {
            ReleaseSemaphore(hQuery,LockCount,NULL);
            LockCount = 0;
        }
        CloseHandle(hQuery);
        hQuery = INVALID_HANDLE_VALUE;
    }
}

boolean SetInterlock(char *name,boolean initialize,boolean for_search) {
char *p;
char IndexName[MAX_PATH];
char QueryName[MAX_PATH];
int i;
boolean bResult;
DWORD dwResult;

    /* Get the last component of the database name to construct the semaphore names */
    i = strlen(name);
    while (i>0) {
        p = name + i - 1;
        if (*p=='\\' || *p=='/') break;
        i--;
    }
    p = name + i;
    
    /* Put the name into upper case and copy to both semaphore names */
    strncpy(IndexName,p,MAX_PATH-1);
    IndexName[MAX_PATH-1] = '\0';
    p = IndexName;
    while (*p) {
        toupper(*p);
        p++;
    }
    strncpy(QueryName,IndexName,MAX_PATH-1);
    QueryName[MAX_PATH-1] = '\0';

    /* Distinguish the strings */
    strncat(IndexName,"-WAIS-Index-mutex",MAX_PATH-1);
    IndexName[MAX_PATH-1] = '\0';
    strncat(QueryName,"-WAIS-Query-semaphore",MAX_PATH-1);
    IndexName[MAX_PATH-1] = '\0';
  
    /* Open or create the Query semaphore */
    hQuery = CreateSemaphore(NULL,MAXQUERIES,MAXQUERIES,QueryName);
    if (hQuery==0 || hQuery==INVALID_HANDLE_VALUE) {
        hQuery = INVALID_HANDLE_VALUE;
        waislog(WLOG_HIGH, WLOG_ERROR,"can't create Query semaphore");
        return(false);
    }

    bResult = true;
    
    if (for_search == true) {
  
        dwResult = WaitForSingleObject(hQuery,0);
        if(dwResult==WAIT_OBJECT_0) {
            /* Got the query semaphore */
            LockCount = 1;
        } else {
            /* Either an update is running, or there are too many queries */ 
            waislog(WLOG_HIGH, WLOG_ERROR,
                "can't search the database as an update is currently running");
            bResult = false;
        }

    } else {

        /* Can we get ownership of the Index mutex? */
        hIndex = CreateMutex(NULL,FALSE,IndexName);
        dwResult = WaitForSingleObject(hIndex,0);
        if (dwResult==WAIT_ABANDONED) {
            /* Clear the abandoned flag and try again */
            ReleaseMutex(hIndex);
            dwResult = WaitForSingleObject(hIndex,0);
        }
        if(dwResult==WAIT_TIMEOUT || dwResult==WAIT_FAILED) {
            waislog(WLOG_HIGH, WLOG_ERROR,
                "an indexing is currently running on the database. Try again later.");
            CloseHandle(hIndex);
            hIndex = INVALID_HANDLE_VALUE;
            bResult = false;
        }
    
        if ( bResult==true && initialize==true ) {
            /* Wait for all queries to finish */
            for (i=0;i<MAXQUERIES;i++) {
                /* Get query semaphore */
                dwResult = WaitForSingleObject(hQuery,30000); /* Thirty secs */
                if (dwResult==WAIT_OBJECT_0) {
                    LockCount++;
                } else {
                    /* timed out */
                    break;
                }
            }
            if ( i<MAXQUERIES ) {
                waislog(WLOG_HIGH, WLOG_ERROR,
                    "timed out in waiting for queries to finish. Try again later.");
                if (i>0) ReleaseSemaphore(hQuery,i,NULL);
                bResult = false;
            }
        }

    }

    if (bResult==false) {
        if (hQuery!=INVALID_HANDLE_VALUE) {
            CloseHandle(hQuery);
            hQuery = INVALID_HANDLE_VALUE;
        }
        if (hIndex!=INVALID_HANDLE_VALUE) {
            ReleaseMutex(hIndex);
            CloseHandle(hIndex);
            hIndex = INVALID_HANDLE_VALUE;
        }
    }
    return(bResult);
}
