///////////////////////////////////////////////////////////////////////////////
//
// rsh client for Windows 95
// (c) 1996 Silviu Marghescu - Cornerstone Technologies, Inc.
//
//
// This program is free software; you can redistribute and/or modify
// it.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
//
// rsh - Remote Shell Client for Windows 95
// Author: Silviu C. Marghescu (http://www.cs.umd.edu/~silviu)
// Date:   Oct 1, 1996
//
///////////////////////////////////////////////////////////////////////////////

#define COPYRIGHT "Copyright 1996  Silviu C. Marghescu, Cornerstone Technologies, Inc."
#define VERSION_MAJOR 1
#define VERSION_MINOR 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <winsock.h>

struct sockaddr_in anaddr; // local socket address structure 
struct sockaddr_in saddr; // server socket address structure
u_short rshPort; // the local rsh port; determined dynamically
u_short rshErrPort; // the local rsh port for client stderr output
u_short rshSPort; // the remote rsh port; basically, the 'shell' port from services
u_short rshProto; // the rsh protocol ("tcp") 
SOCKET rshClient=INVALID_SOCKET; // the rsh client socket for outgoing connections 
SOCKET rshClientErr=INVALID_SOCKET; // the rsh client socket for stderr input

char userName[64]; // user name; given with the '-l' option in the command line
char cmd[4096];

int debugFlag=0; // debug messages on/off flag
int stderrFlag=0; // set when a connection has been detected on the stderr channel

// socket options variables 
int on=1;
struct linger linger;


// debug //////////////////////////////////////////////////////////////////////
//
// debugging functions
//
void
	debug (const char* message)
{
	if(debugFlag)
		fprintf(stderr, "%s\n", message);
}

void
	debug (const char* message, int val)
{
	if(debugFlag)
		fprintf(stderr, "%s %d\n", message, val);
}


// winsockError ///////////////////////////////////////////////////////////////
//
// displays the current Winsock error in text format
//
void 
	winsockError ()
{
    fprintf(stderr, "Winsock error: ");
    int errCode=WSAGetLastError();
    switch(errCode)
    {
        case WSAENETDOWN:
            fprintf(stderr, "The network subsystem has failed.\n");
            break;
        case WSAEINTR:
            fprintf(stderr, "A blocking call was cancelled.  This can be caused by\n1) a short response time, or\n2) User interrupts the process.\n");
            break;
        case WSAEINPROGRESS:
            fprintf(stderr, "A blocking call is in progress.\n");
            break;
        case WSAENOBUFS:
            fprintf(stderr, "No buffer space is available.\n");
            break;
        case WSAENOTSOCK:
            fprintf(stderr, "Invalid socket descriptor.\n");
            break;
        case WSAEADDRINUSE:
            fprintf(stderr, "The specified address is already in use.\n");
            break;
        case WSAEADDRNOTAVAIL:
            fprintf(stderr, "The specified address is not available\nfrom the local machine.\n");
            break;
        case WSAECONNREFUSED:
            fprintf(stderr, "The connection attempt was refused.\n");
            break;
        case WSAEINVAL:
            fprintf(stderr, "The socket is not bound to an address.\n");
            break;
        case WSAEISCONN:
            fprintf(stderr, "The socket is already connected.\n");
            break;
        case WSAEMFILE:
            fprintf(stderr, "The maximum number of sockets has exceeded.\n");
            break;
        case WSAENETUNREACH:
            fprintf(stderr, "Network cannot be reached from this host at this time.\n");
            break;
        case WSAETIMEDOUT:
            fprintf(stderr, "Attempt to connect timed out without establishing a connection.\n");
            break;
        case WSAENOTCONN:
            fprintf(stderr, "The socket is not connected.\n");
            break;
        case WSAESHUTDOWN:
            fprintf(stderr, "The socket has been shut down.\n");
            break;
        case WSAECONNABORTED:
            fprintf(stderr, "The virtual circuit was aborted due to timeout or other failure.\n");
            break;
        case WSAECONNRESET:
            fprintf(stderr, "The virtual circuit was reset by the remote side.\n");
            break;
        case WSAEACCES:
            fprintf(stderr, "The requested address is a broadcast address.\n");
            break;
        case WSAENETRESET:
            fprintf(stderr, "The connection must be reset.\n");
            break;
        case WSAHOST_NOT_FOUND:
            fprintf(stderr, "Authoritative Answer Host is not found.\n");
            break;
        default:
			fprintf(stderr, "%d.\n", errCode);
            break;
    }
}

// error //////////////////////////////////////////////////////////////////////
//
// display an error message and possibly the last Winsock error
//
void
	error (const char* message, int ex=1)
{
    fprintf(stderr, "*** ERROR: %s\n*** ", message);
    winsockError();
    if(ex)
    {
        WSACleanup();
        exit(1);
    }
}


// rresvport //////////////////////////////////////////////////////////////////
//
// the windows hack of rresvport; bind a socket to a reserved port using the
// given protocol, if any
//
int
    rresvport (u_short& alport, int sProto=0)
{
    struct sockaddr_in sin;
    int s;

    sin.sin_family=AF_INET;
    sin.sin_addr.s_addr=INADDR_ANY;
    s=socket(AF_INET, SOCK_STREAM, sProto);
    if(s<0)
        return -1;

    for(alport=IPPORT_RESERVED-1; alport>IPPORT_RESERVED/2; alport--)
    {
        sin.sin_port=htons((u_short)alport);
        if(bind(s, (struct sockaddr*)&sin, sizeof(sin))==0)
            return s;

        if(WSAGetLastError()!=WSAEADDRINUSE)
            break;
    }
    // ran out of available ports or weird error; shouldn't happen too often...
    closesocket(s);
    return -1;
}


// send //////////////////////////////////////////////////////////////////////
//
// send to the server the assembled command buffer
//
void
    send (const char* buff, int bufflen, SOCKET rshClient)
{
    if(send(rshClient, buff, bufflen, 0) < bufflen)
        error("Error sending command.", 0);
}


// receive ////////////////////////////////////////////////////////////////////
//
// receive a string from the given socket
//
int
    receive (SOCKET rshClient, char* buff, int blen)
{
    int bufflen;
    int totallen=0;
    debug("Receiving...");
    do
    {
        bufflen=recv(rshClient, buff+totallen, blen-totallen, 0);
        if(bufflen==SOCKET_ERROR)
            return bufflen;
        if(debugFlag)
            fprintf(stderr, "...got %d chars.\n", bufflen);
        totallen+=bufflen;
    } while(bufflen && totallen<blen && buff[bufflen-1]);
    if(!totallen)
        buff[0]=0;
    buff[totallen]=0;
    return totallen;
}


// winsockCheck ///////////////////////////////////////////////////////////////
//
// make sure we have the right winsock.dll version; 1.1 required 
//
void
    winsockCheck ()
{
    debug("Checking winsock.dll version...");
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    wVersionRequested = MAKEWORD( 1, 1 );
    err = WSAStartup( wVersionRequested, &wsaData );
    if ( err != 0 )
        error("Unsupported version of winsock.dll!\n");

    // Confirm that the Windows Sockets DLL supports 1.1.
    // Note that if the DLL supports versions greater 
    // than 1.1 in addition to 1.1, it will still return 
    // 1.1 in wVersion since that is the version we 
    // requested.     
    if ( LOBYTE( wsaData.wVersion ) != 1 || HIBYTE( wsaData.wVersion ) != 1 )
        error("Unsupported version of winsock.dll!\n");

    // The Windows Sockets DLL is acceptable. Proceed. 
}


// hostCheck //////////////////////////////////////////////////////////////////
//
// check the remote host name and fill the server address structure
//
void
	hostCheck (const char* hostname)
{
	debug("Checking hostname...");
	memset(&saddr, 0, sizeof(saddr));
	saddr.sin_addr.s_addr=inet_addr(hostname);
	if(saddr.sin_addr.s_addr==(u_long)INADDR_NONE)
	{
		// we must have gotten a host name instead of an IP address; resolve!
		struct hostent* hostInfo=gethostbyname(hostname);
		if(!hostInfo)
			error("Invalid hostname!");
		memcpy((void*)&saddr.sin_addr.s_addr, hostInfo->h_addr, hostInfo->h_length);
	}
}


// initSocket /////////////////////////////////////////////////////////////////
//
// standard socket initialization procedure 
//
void
    initSocket ()
{
    // get port number for rsh
    struct servent FAR* sp=getservbyname("shell", "tcp");
    if(sp==NULL)
        error("Cannot determine port number for the rsh client.");
    rshSPort=htons(sp->s_port);

    // get protocol number for tcp 
    LPPROTOENT lpProto=getprotobyname("tcp");
    if(!lpProto)
    {
        debug("Cannot obtain the protocol number; using default...");
        rshProto=IPPROTO_TCP;
    }
    else
        rshProto=lpProto->p_proto;

    debug("Creating and binding socket...");
    // create socket 
    rshClient=rresvport(rshPort, rshProto);
    if(rshClient==INVALID_SOCKET)
        error("Cannot allocate socket for the rsh client.");
	debug("Socket bound to", rshPort);
    debug("Setting options on the main socket...");
    if(setsockopt(rshClient, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on))<0)
        error("Cannot set SO_KEEPALIVE!\n", 0);
    linger.l_onoff=1;
    linger.l_linger=60;
    if(setsockopt(rshClient, SOL_SOCKET, SO_LINGER, (char*)&linger, sizeof(linger))<0)
        error("Cannot set SO_LINGER!\n", 0);
}


// openErrSocket //////////////////////////////////////////////////////////////
//
// an additional socket is created for stderr output
//
int
    initErrSocket ()
{
    // create the new socket and bind it to the client stderr port
    rshErrPort=IPPORT_RESERVED-1;
    rshClientErr=rresvport(rshErrPort);
    if(rshClientErr==INVALID_SOCKET)
    {
        error("Cannot create stderr socket!", 0);
        return 1;
    }

	debug("Error socket bound to", rshErrPort);
    debug("Setting options on the stderr socket...");
    if(setsockopt(rshClientErr, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on))<0)
        error("Cannot set SO_KEEPALIVE!", 0);
    linger.l_onoff=0;
    linger.l_linger=60;
    if(setsockopt(rshClientErr, SOL_SOCKET, SO_LINGER, (char*)&linger, sizeof(linger))<0)
        error("Cannot set SO_LINGER!", 0);
	// now listen...
	if(listen(rshClientErr, 5))
		error("Cannot listen!");
    return 1;
}


// command ////////////////////////////////////////////////////////////////////
//
// pass the command string to the rsh server and retrieve the results
//
void
	command (const char* cmd, const char* userName)
{
	char cmdbuff[2048];
	int cmdbufflen=0;
	sprintf(cmdbuff+cmdbufflen, "%d", rshErrPort); // local stderr port
	cmdbufflen=strlen(cmdbuff)+1;
	strcpy(cmdbuff+cmdbufflen, userName); // remuser
	cmdbufflen+=strlen(userName)+1;
	strcpy(cmdbuff+cmdbufflen, userName); // locuser
	cmdbufflen+=strlen(userName)+1;
	strcpy(cmdbuff+cmdbufflen, cmd); // command
	cmdbufflen+=strlen(cmd)+1;
	send(cmdbuff, cmdbufflen, rshClient);

	debug("Receiving stdout output...");
	char buff[2048];
	while(receive(rshClient, buff, 2047)>0)
		printf(buff);
}


// clientThread ///////////////////////////////////////////////////////////////
//
// this is the stderr client thread; it is started before sending the command
// string to the server; its purpose is to accept connections from the server
// and receive the stderr output
//
long
    clientThread (LPVOID)
{
    debug("stderr thread started...");
	struct sockaddr anaddr;
    int len=sizeof(anaddr);
    SOCKET rshServer=accept(rshClientErr, (struct sockaddr FAR*)&anaddr, &len);
	if(rshServer==INVALID_SOCKET)
	{
		error("Error connecting to the stderr server port!", 0);
		return 0;
	}

	stderrFlag=1; // mark connection made

	debug("Receiving stderr output...");
	char buff[2048];
	while(receive(rshServer, buff, 2047)>0)
		fprintf(stderr, buff);
    
	debug("Closing stderr socket...");
    shutdown(rshClientErr, 2);
    closesocket(rshClientErr);

	return 0;
}


// process ////////////////////////////////////////////////////////////////////
//
// main processing routine; connect to server, pass command line and wait for
// results
//
void
    process (const char* cmd, const char* userName)
{
	debug("Creating main socket...");
    initSocket();
	debug("Creating error socket...");
	if(!initErrSocket())
		error("Cannot create error socket!", 0);
	debug("Connecting...");
	saddr.sin_family=AF_INET;
	saddr.sin_port=rshSPort;
	if(connect(rshClient, (struct sockaddr FAR*)&saddr, sizeof(saddr)))
		error("Cannot connect to RSH port!\n");
	
	HANDLE threadHnd;
	if(rshClientErr!=INVALID_SOCKET)
	{
		debug("Firing stderr client thread...");
		DWORD threadID;
		threadHnd=CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)clientThread,
			(LPVOID)0, 0, (LPDWORD)&threadID);
		if(!threadHnd)
			error("Cannot start client thread...");
	}

    debug("Processing client data...");
    command(cmd, userName);
    debug("Closing main socket...");
    shutdown(rshClient, 2);
    closesocket(rshClient);
	if(threadHnd)
	{
		DWORD exitCode=0;
		GetExitCodeThread(threadHnd, &exitCode);
		while(exitCode==STILL_ACTIVE && stderrFlag)
		{
			Sleep(50);
			GetExitCodeThread(threadHnd, &exitCode);
		}
		CloseHandle(threadHnd);
	}
}


// parseCommandLine ///////////////////////////////////////////////////////////
//
// check for command line parameters
//
int
    parseCommandLine (int argc, char* argv[])
{
	// reset the user name
	strcpy(userName, "nobody");
    for(int i=1; i<argc && argv[i][0]=='-'; i++)
		if(!strcmpi(argv[i], "-d"))
			debugFlag=1;
		else
		if(!strcmpi(argv[i], "-l"))
			strcpy(userName, argv[++i]); // user name
		else
		if(!strcmpi(argv[i], "-v"))
		{
			fprintf(stderr, "\nrsh - remote shell client for Windows 95, version %d.%d\n%s\n",
				VERSION_MAJOR, VERSION_MINOR, COPYRIGHT);
			exit(0);
		}
		else
		if(!strcmp(argv[i], "-h"))
		{
			fprintf(stderr,  "\nrsh - remote shell client for Windows 95, version %d.%d\n%s\n\n\
Usage:\n\trsh [ -dhv ] [ -l username ] hostname command\n\nCommand line options:\n\
\t-d\tdebug output\n\
\t-v\tdisplay rsh version\n\
\t-h\tthis help screen\n", VERSION_MAJOR, VERSION_MINOR, COPYRIGHT);
			exit(0);
		}
		else
			fprintf(stderr, "Ignoring unknown option '%s'...\n", argv[i]);

	return i; // return the index of the first not processed parameter (hopefully the hostname)
}


// main ///////////////////////////////////////////////////////////////////////
void
    main (int argc, char* argv[])
{
	int hostndx=parseCommandLine(argc, argv);
	if(hostndx>argc-2)
	{
		error("Invalid hostname or command parameter!", 0);
		exit(2);
	}
    winsockCheck();
	hostCheck(argv[hostndx]);
	// assemble the command from the remaining command line parameters
	for(cmd[0]=0, hostndx++; hostndx<argc; hostndx++)
	{
		strcat(cmd, argv[hostndx]);
		if(hostndx<argc-1)
			strcat(cmd, " ");
	}
    process(cmd, userName);
    WSACleanup();
    exit(0);
}
