/* Ln: create a hard link to a file (os9/68k)
 * by Robert A. Larson (blarson@skat.usc.edu)
 * This is placed in the public domain for all to enjoy.
 *
 ****** USE AT YOUR OWN RISK *****
 *
 * 01/14/89	Version 1.0
 *
 * Warning: this does potentialy dangerous writing to directories
 * and to the raw disk.	 Although some attempt is made to lock out
 * other users from interfearing, I'm not sure the locking works.
 * Moving or links to directories should be used with EXTREME caution
 * if at all.  Note that moving a directory can make a directory tree
 * inaccessable, and 'deldir'ing a linked directory will delete its
 * contents and change the remaining links to normal files.
 */

/*

$Header: /h0/LIBS/OS9LIB/RCS/ln.c_v 2.1 92/08/04 00:14:37 jl Exp $
$Revision: 2.1 $
$Author: jl $
$Date: 92/08/04 00:14:37 $
$Source: /h0/LIBS/OS9LIB/RCS/ln.c_v $

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <modes.h>
#include <direct.h>
#include <errno.h>

static int dncmp(char *, char *);		/* directory name comparison	*/
static int do_link(char *, char *, int );	/* does all the real work	*/

/* keys for the do_link subroutine */

#define DO_LINK		0
#define DO_RENAME	1
#define TO_DIR		128			/* orred with one of the above	*/

#define problem(x)				/* fprintf x */

int link(char *old, char *new)
{
    return do_link(old, new, DO_LINK);
}

int rename(char *old, char *new)
{
    return do_link(old, new, DO_RENAME);
}

int do_link(char *old, char *new, int action)
{
    char *olddir, *oldname, *newdir, *newname;
    int olddirp, newdirp;
    char olddevn[32], newdevn[32+2];
    struct dirent de;
    struct ddsect dd;
    unsigned char fdl;
    int disk = 0;
    long empty_slot;
    long newdir_addr;
    long entrydir_addr;
    int same;
    int i;
    long old_slot = 0;
    char namebuf[256];
    char nn[128];

    if ((oldname = rindex(old, '/')) == NULL) {
	olddir = (char *) ".";
	oldname = old;
    } else {
	olddir = namebuf;
	strncpy(namebuf, old, oldname - old);
	namebuf[oldname - old] = '\0';
	oldname++;
    }
    
    if (action & TO_DIR) {
	newdir = new;
	newname = oldname;
	action &= ~TO_DIR;
    } else {
	if ((newname = rindex(new, '/')) == NULL) {
	    newdir = (char *) ".";
	    newname = new;
	} else {
	    newdir = namebuf;
	    if (olddir==namebuf) newdir += oldname - old;
	    strncpy(newdir, new, newname - new);
	    newdir[newname - new] = '\0';
	    newname++;
	}
    }
    
    if (_prsnam(newname) <= 0) {
	problem((stderr, "%s: Invalid file name \"%s\"\n", _prgname(), newname));
	return ERROR;
    }
    
    if ((olddirp = open(olddir, S_IREAD | S_IWRITE | S_IFDIR)) < 0) {
	problem((stderr, "%s: Can't open old directory \"%s\"\n", _prgname(), olddir));
	return ERROR;
    }

    if ((newdirp = open(newdir, S_IREAD | S_IWRITE | S_IFDIR)) < 0) {
	problem((stderr, "%s: Can't open new directory \"%s\"\n", _prgname(), newdir));
	close(olddirp);
	return ERROR;
    }

    _gs_devn(olddirp, olddevn);
    _gs_devn(newdirp, newdevn);

    if (strcmp(olddevn, newdevn) != 0) {
	problem((stderr, "%s: Not on same device \"%s\" \"%s\"\n", _prgname(), olddir, newdir));
	errno = E_DIFFER;
	close(olddirp);
	close(newdirp);
	return ERROR;
    }
    
    if ((lseek(newdirp, (long)sizeof de, 0) < 0)				/* skip .. */
	    || (read(newdirp, (char *)&de, sizeof de) != sizeof de)) {	/* read . */
	problem((stderr, "%s: Problem reading \"%s\"\n", _prgname(), newdir));
	close(olddirp);
	close(newdirp);
	return ERROR;
    }

    newdir_addr = de.dir_addr;
    
    if ((lseek(olddirp, (long)sizeof de, 0) < 0)				/* skip .. */
	    || (read(olddirp, (char *)&de, sizeof de) != sizeof de)) {	/* read . */
	problem((stderr, "%s: Problem reading \"%s\"\n", _prgname(), olddir));
	close(olddirp);
	close(newdirp);
	return ERROR;
    }

    if ((same = (de.dir_addr == newdir_addr)) != 0) {
	/* link to same directory, scan only once */
	close(olddirp);
	olddirp = newdirp;
	/* rename to same name allowed, but not link */
	if (action == DO_LINK && dncmp(oldname, newname) == 0) {
	    problem((stderr, "%s: Can't link to same name \"%s\" \"%s\"\n", _prgname(), oldname, newname));
	    errno = E_CEF;
	    close (newdirp);
	    return ERROR;
	}
	if (_ss_lock(newdirp, -1) < 0) {
	    problem((stderr, "%s: Could not lock directory \"%s\"\n", _prgname(), newdir));
	    close(newdirp);
	    return ERROR;
	}
    }
    
    empty_slot = 0;

    for(;;) {
	if (read(olddirp, (char *)&de, sizeof de) != sizeof de) {
	    problem((stderr, "%s: Could not find \"%s\" in \"%s\"\n", _prgname(), oldname, olddir));
	    errno = E_PNNF;
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	if (dncmp(de.dir_name, oldname) == 0)
		break;	/* found it */
	if (same) {
	    if (dncmp(de.dir_name, newname) == 0) {
		problem((stderr, "%s: Already exists \"%s/%s\"\n", _prgname(), newdir, newname));
		errno = E_CEF;
		close(newdirp);
		return ERROR;
	    }
	    if (empty_slot == 0 && de.dir_name[0] == '\0')
		empty_slot = _gs_pos(olddirp);
	}
    }

    if (action != DO_LINK)
		old_slot = _gs_pos(olddirp);

    entrydir_addr = de.dir_addr;

    if (!same) {
	if (_ss_lock(newdirp, 0xffffffff) < 0) {
	    problem((stderr, "%s: Can't lock directory \"%s\"\n", _prgname(), newdir));
	    close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
    }
    
    if (action != DO_LINK && same)
		empty_slot = old_slot;
    while (read(newdirp, (char *)&de, sizeof de) == sizeof de) {
	if (dncmp(de.dir_name, newname) == 0) {
	    problem((stderr, "%s: Already exists \"%s/%s\"\n", _prgname(), newdir, newname));
	    errno = E_CEF;
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	if (empty_slot == 0 && de.dir_name[0] == '\0') 
	    empty_slot = _gs_pos(newdirp);
    }

    strcpy(de.dir_name, newname);

    i = strlen(de.dir_name);
    de.dir_name[i-1] |= 0x80;

    while (i < sizeof de.dir_name) de.dir_name[i++] = '\0';

    de.dir_addr = entrydir_addr;

    if (empty_slot != 0) {
	if (lseek(newdirp, empty_slot - sizeof de, 0) < 0) {
	    problem((stderr, "%s: Can't position in \"%s\"\n", _prgname(), newdir));
/*	    close(disk);    austesten was das soll, da disk nicht initializiert ist */

	    if (!same)
		close(olddirp);

	    close(newdirp);

	    return ERROR;
	}
    }

    if (action == DO_LINK) {		/* link: increase the link count */
	newdevn[0] = '/';			/* form /<device>@ string */
	strcat(strcpy(newdevn+1, olddevn), "@");
	if ((disk = open(newdevn, S_IREAD)) < 0) {
	    problem((stderr, "%s: Can't open \"%s\"\n", _prgname(), newdevn));
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	
	read(disk, (char *) &dd, sizeof dd);
	close(disk);
	if ((disk = open(newdevn, S_IREAD | S_IWRITE)) < 0) {
	    problem((stderr, "%s: Can't open \"%s\"\n", _prgname(), newdevn));
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}

	if (lseek(disk, (entrydir_addr*dd.dd_lsnsize) +
		((char *)(&(((struct fildes *)0)->fd_link)) - (char *)0), 0) < 0) {
	    problem((stderr, "%s: Can't seek to sector 0x%x on \"%s\"\n", _prgname(), entrydir_addr, newdevn));
	    close(disk);
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}

	if (read(disk, (char *)&fdl, sizeof fdl) != sizeof fdl) {
	    problem((stderr, "%s: Can't read \"%s\"\n", _prgname(), newdevn));
	    close(disk);
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
 	}

	if (fdl == 0) {
	    problem((stderr, "%s: Bad file descriptor \"%s/%s\"\n", _prgname(), olddir, oldname));
	    errno = E_BMHP;		/* no appropriate error number */
	    close(disk);
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	
	if (++fdl == 0) {
	    problem((stderr, "%s: Too many links \"%s/%s\"\n", _prgname(), olddir, oldname));
	    errno = E_BMHP;		/* no appropriate error number */
	    close(disk);
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	
	if (lseek(disk, -(long)sizeof fdl, 1) < 0) {
	    problem((stderr, "%s: Can't lseek on \"%s\"\n", _prgname(), newdevn));
	    close(disk);
	    if (!same )
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}

	if (write(disk, (char *)&fdl, sizeof fdl) < 0) {
	    problem((stderr, "%s: Can't write to \"%s\"\n", _prgname(), newdevn));
	    close(disk);
	    if (!same)
		close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	
    } else {
	strcpy(nn, olddir);
	strcat(nn, "/");
	strcat(nn, oldname);

	/* Don't let non-superuser move directories, it is dangerous!
	 * (artificial restriction)
	 */
	 
	if (!same && access(nn, 0) != 0 && getuid() != 0) { 
	    problem((stderr, "%s: Directory moving can only be done by superuser\n", _prgname()));
	    errno = E_FNA;
	    close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
	
	strcpy(nn, newdir);
	strcat(nn, "/");
	strcat(nn, newname);
    }

    if (write(newdirp, (char *)&de, sizeof de) < 0) {
	problem((stderr, "%s: Can't write to \"%s\"\n", _prgname(), newdir));
	if (action==DO_LINK)
		close(disk);
	if (!same)
		close(olddirp);
	close(newdirp);
	return ERROR;
    }
    
    if (action != DO_LINK && !same) {		/* rename: delete old name */
	if ((disk = open(nn, S_IREAD | S_IWRITE | S_IFDIR)) >= 0) {
	    /* change .. when moving directory */
	    if (lseek(disk, (long)sizeof de.dir_name, 0) < 0) {
		problem((stderr, "%s: can't position in \"%s\"\n", _prgname(), nn));
		close(disk);
		close(olddirp);
		close(newdirp);
		return ERROR;
	    }
	    
	    if (write(disk, (char *)&newdir_addr, sizeof newdir_addr) < 0) {
		problem((stderr, "%s: Can't write to \"%s\"\n",
			_prgname(), nn));
		close(disk);
		close(olddirp);
		close(newdirp);
		return ERROR;
	    }
	    
	    close(disk);
	}
	fdl = '\0';

	if (lseek(olddirp, old_slot - sizeof de, 0) < 0) {
	    problem((stderr, "%s: can't position in \"%s\"\n", _prgname(), olddir));
	    close(olddirp);
	    close(newdirp);
	    return ERROR;
	}

	if (write(olddirp, (char *)&fdl, sizeof fdl) < 0) {
	    problem((stderr, "%s: can't write to \"%s\"\n", _prgname(), olddir));
	    close(olddirp);
	    close(newdirp);
	    return ERROR;
	}
    }
    
    if (action == DO_LINK)
	close(disk);
    if (!same)
	close(olddirp);
    close(newdirp);

    return OK;
}

/* Compares two os9 file names, case independant.
 * The first argument may be terminated by setting the msb of
 * the last character or NUL terminated, the second must be NUL
 * termintated.	 Returns zero on same, non-zero on different.
 */
 
static int dncmp(char *cp1, char *cp2)
{
    int c1, c2;

    while ((c1 = *cp1++) != 0) {
	if (c1 & 0x80) {
	    c1 &= 0x7f;
	    return ((c1 != (c2 = *cp2++)) &&
		    (isupper(c1) ? (tolower(c1) != c2) :
		    (!isupper(c2) || (tolower(c2) != c1)))) || *cp2;
	}
	if ((c1 != (c2 = *cp2++)) && (isupper(c1) ? (tolower(c1) != c2) :
		(!isupper(c2) || (tolower(c2) != c1)))) return 1;
    }
    return *cp2;
}
