/************************************************************************
 *									*
 *  Copyright (c) 1987							*
 *  by CompuServe Incorporated, Columbus, Ohio				*
 *									*
 *  The information in ths software is subject to change without	*
 *  notice and should not be construed as a commitment by CompuServe	*
 *  Incorporated							*
 *									*
 ************************************************************************/

/**
 * File:  BPXPORT.C
 *
 * Facility:  B protocol Transport layer facility
 *
 * Abstract:
 *	'C' header for B protocol transport layer facilit
 *
 * Environment:  None specified
 *
 * Author:  John Pampuch, January 29, 1988
 *
 * Modified by:
 *
 * 1	John Pampuch, January 29, 1988
 *	- Original
 *
 * 2	John Pampuch, February 23, 1988
 * 	- match COMDEF.H requirements
 *
 * 3	John Pampuch, March 17, 1988
 * 	- changed to reduce return value overloading
 *
 **/

#include "comdef.h"
#include <string.h>
#include "bpxport.h"
#include "bplus.h"

private WORD Queue_Packet (TR_PDB *);
private WORD Queue_Packets (TR_PDB *);

#define pdb xpdb->Packet
#define data_packet_max_size (xpdb->Packet->Packet_Size-1)

#define More_Packet 'M'
#define Last_Packet 'L'
#define Plus_Packet '+'
#define Error_Packet 'F'

/*$page*/
public WORD BP_Transport_Init(xpdb)
/*
* setup transport layer, initializing pointers, etc.
*
* Parameters
*    xpdb - Transport protocol descriptor block
*         
* Returns
*    Failure if there is a memory allocation or protocol failure, otherwise
*    Success 
*/
    TR_PDB *xpdb;
    {
    xpdb->First_Packet = NULL;

    xpdb->Msg_Count = 0;
    pdb->Next_Packet = 0;
    pdb->Use_Transport = 1;

    /* at this point, we need to wait for, and respond to */
    /* the host's 'plus' packet. */
    while (not pdb->Actual_Xport)
	if (Queue_Packets (xpdb) == Failure) return Failure;

    return Success;
    }

/*$page*/
public void BP_Transport_Close (xpdb)
/*
* terminate transport layer, freeing pointers, etc.
* This will discard any pending data.
*
* Parameters
*    xpdb - Transport protocol descriptor block
*         
* Returns
*    nothing
*/
    TR_PDB *xpdb;
    {
    struct TRpacketStruct *old;

    /*
     * Generally, this while loop should drop through, unless
     * the transport layer is closed prematurely... Then, it
     * it is necessary to discard received (but not handled)
     * packets.  After that, the final pointer must be freed.
     */
    while (xpdb->First_Packet != NULL)
	{
	old = xpdb->First_Packet;
	xpdb->First_Packet = xpdb->First_Packet->next;
	(*pdb->s_free) ((BYTE *) old);
	}

    do
	BP_Flush_Pending (pdb);
    while (pdb->Error_Code == Packet_Arrived);

    }

/*$page*/
public UWORD BP_Send_Data (Message, Length, xpdb)
/* 
 * Send message in to remote.
 * Queue and acknowledge incoming packets and complete
 * requested transmission using as many packets as necessary.
 * Verify outgoing data by awaiting for acknowledgement.
 *
 * Parameters
 *    Message - BYTE pointer to data to transmit
 *    Length - Length of block to send
 *
 * Returns
 *    Success indicates data sent or
 *    Failure indicates that it was not possible to send data.
 */
    BYTE *Message;
    UWORD Length;
    TR_PDB *xpdb;
    {
    UWORD Size;
    WORD Status = Success;

    while (Status == Success and Length > 0)
	{
	Queue_Packets (xpdb);

	/*
	 * build each physical packet for sending
	 */
	if (Length > data_packet_max_size)
	then
	    {
	    Size = data_packet_max_size;
	    pdb->Send_Buffer[0] = More_Packet;
	    }
	else
	    {
	    Size = Length;
	    pdb->Send_Buffer[0] = Last_Packet;
	    }

	memcpy (pdb->Send_Buffer+1, Message, Size);

	/*
	 * Send this portion of the message as a packet.
	 * if there is an incoming packet, queue it, and continue
	 * waiting til the packet is sent.  Repeat until a
	 * the packet is sent, or a failure is determined.
	 */
	Status = BP_Send_Packet (Size+1, pdb);

	if (pdb->Error_Code == Packet_Arrived)
	then
	    do
		if (Queue_Packet (xpdb) == Failure)
		then
		    return Failure;
		else
		    Status = BP_Send_Packet (Size+1, pdb);

	    while (pdb->Error_Code == Packet_Arrived);

	if ( (*xpdb->Wants_to_Abort) () )
	then
	    return Failure;

	if (Status == Success)
	    {
	    Length -= Size;
	    Message += Size;
	    }
	}
    return Status;
    }

/*$page*/
public WORD BP_Msgs_Available (xpdb)
/*
 * Determine how many whole messages are available in the queue.
 * Queue and acknowledge incoming packets.  Return count of
 * pending messages.
 *
 * Parameters
 *    none
 * 
 * Returns
 *	Count of messages available.
 *	0  indicates success, but no messages available
 * 	-1 indicates protocol failure
 */
    TR_PDB *xpdb;
    {
    if (Queue_Packets (xpdb) == Failure)
    then
	return -1;

    return xpdb->Msg_Count;
    }

/*$page*/
public WORD BP_Msg_Size (xpdb)
/*
* Determine if a whole message is available in the queue.
* Queue and acknowledge incoming packets.  Return length of
* next complete message if available.
*
* Parameters
*    none
* 
* Returns
*	Size of next message available.
*	0  indicates success, but no messages available
* 	-1 indicates protocol failure
*/
    TR_PDB *xpdb;
    {
    struct TRpacketStruct *Packet_Ptr;
    BOOL end_of_list;
    UWORD size = 0;

    /*
     * first, check and see if any new packets have arrived.
     * Then, if we don't have a whole message, return anyway.
     * Otherwise, walk through that list and find out how large
     * the next message is.
     */

    if (Queue_Packets (xpdb) == Failure)
    then
	return -1;

    if (xpdb->Msg_Count <= 0)
    then
	return 0;

    Packet_Ptr = xpdb->First_Packet;
    do
	{
	end_of_list = (Packet_Ptr->data[0] == Last_Packet);
	size += (Packet_Ptr->len - 1);
	Packet_Ptr = Packet_Ptr->next;
	}
    while (not end_of_list);

    return size;
    }

/*$page*/
public WORD BP_Read_Data (Message, Max_Len, xpdb)
/*
* Await data from the host, receiving all packets in a
* message.  
*
* Parameters
*    Message - BYTE pointer to block large enough to accept next message
*    Max_Len - Size of block for message; data after size limit is discarded
*    xpdb    - Transport protocol description block
*         
* Returns
*	Length of message.
*	0  indicates success, but no data available
* 	-1 indicates protocol failure
*/
    BYTE *Message;
    WORD Max_Len;
    TR_PDB *xpdb;
    {
    BOOL end_of_list = FALSE;
    WORD Size = 0;
    WORD Packet_Limit;
    struct TRpacketStruct *Old;

    /*
     * if there is a protocol failure, return immediately.
     * if there are no messages, return 0
     * Otherwise, walk through list of packets and build the next message.
     */

    if (Queue_Packets (xpdb) == Failure)
    then
	return -1;

    if (xpdb->Msg_Count <= 0)
    then
	return 0;

    do
	{
	end_of_list = (xpdb->First_Packet->data[0] == Last_Packet);

	if (Size < Max_Len)
	then
	    {
	    if (Size + xpdb->First_Packet->len - 1 > Max_Len)
	    then
		Packet_Limit = Max_Len - Size;
	    else
		Packet_Limit = xpdb->First_Packet->len - 1;

	    memcpy (&Message[Size], &xpdb->First_Packet->data[1],
		Packet_Limit);

	    Size += Packet_Limit;
	    }

	Old = xpdb->First_Packet;
	xpdb->First_Packet = xpdb->First_Packet->next;
	(*pdb->s_free) ((BYTE *) Old);
	}
    while (not end_of_list);

    xpdb->Msg_Count--;
    return Size;
    }

/*$page*/
private WORD Queue_Packets (xpdb)	/* read packets and queue them */
    TR_PDB *xpdb;
    {
    /*
     * Accept (and acknowledge) any packets that have arrived since
     * the last call and queue them.  Don't wait for a packet if
     * it has not begun to arrive.
     */

    if ( (*xpdb->Wants_to_Abort) () )
    then
        return Failure;

    if (BP_Read_Packet (FALSE, FALSE, TRUE, pdb) == Success)
	     {
        if (Queue_Packet (xpdb) == Failure)
        then
            return Failure;
		  };

    return Success;
    }

/*$page*/
private WORD Queue_Packet (xpdb)	
/* take one packet from incoming remote and queue it */
    TR_PDB *xpdb;
    {
    UWORD size;
    struct TRpacketStruct *new_next;

    /*
     * if a plus negotiation packet has arrived, respond to it.
     * otherwise, queue transport layer packets for later
     * processing.  Other packets are handled as errors.
     */

    if ( (*xpdb->Wants_to_Abort) () )
    then
	return Failure;

    switch (pdb->Read_Buffer[0])
	{
	case Plus_Packet:
	    return BP_Plus_Respond(pdb);

	case Last_Packet:
	    xpdb->Msg_Count++;
	    /* this is intended to drop through */

	case More_Packet:
	    size = pdb->R_Buffer_Len;

	    new_next = (struct TRpacketStruct *) (*pdb->s_alloc)
		(sizeof (struct TRpacketStruct) + size);

	    if (new_next == NULL)
	    then
		return Failure;

	    new_next->next = NULL;

	    if (xpdb->First_Packet == NULL)
	    then
		xpdb->Next_Packet = xpdb->First_Packet = new_next;
	    else
		{
		xpdb->Next_Packet->next = new_next;
		xpdb->Next_Packet = new_next;
		}

	    memcpy (new_next->data, pdb->Read_Buffer, size);

	    new_next->len = size;

	    return Success;

	case Error_Packet:
	    /* we should probaby pass the error back to the application */
	    (*pdb->Read_Error) ();
	    return Failure;

	default:
	    /*
	     * unknown packet type... probably should respond with a
	     * failure packet.
	     */
	    (*pdb->Read_Error) ();
	    BP_Send_Failure ('E', "Unexpected packet type", pdb);
	    return Failure;
	}
    }
