/*
 *	bpdte.c
 *
 * This program emulates a dumb terminal with file transfer support using
 * CompuServe's B-Protocol.  This program is just a sample of how to interface
 * the BP module (BPSLAVE.C) with the rest of the terminal emulator.
 */

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#undef max
#undef min
#include <comdef.h>      /* standard CompuServe definitions */

#include "keyboard.h"    /* these are headers for routines you must provide */
#include "commport.h"
#include "timer.h"
#include "bpslave.h"
#include "fileio.h"

/*
 * Contents:
 */

static   void Setup_FILTRN (void);
static  void Print_File_Line (void);
static  void DLE_B_Seen (struct APDBstruct *);
static  void Print_Msg (BYTE *);
static  void D_Sent (UWORD);
static  void D_Read (UWORD);
static  void Print_File (BYTE *);
static  void Print_Size (ULONG);
static  void Print_Mode (WORD, WORD, WORD);
static  void S_Error (void);
static  void R_Error (void);
static  void BPSent (UWORD);
static  void BPRead (UWORD);
static  WORD Remote_Get_Timed (WORD);
static  WORD Remote_Put (BYTE);
static  BOOL Wants_To_Abort (void);
static   BYTE *s_malloc (UWORD);
static   void s_free (BYTE *);
static   void prints (BYTE *);

extern  void Delay (UWORD);
extern  void main (WORD, BYTE * *);

#define PDB ((Protocol_Desc_Block *) apdb)

#define Exit_Key  0x012d
#define Break_Key 0x0130 
#define Is_Function_Key(C)       FALSE /* is key a local function key? */

#define Normal_Mode      0
#define Escape_Seq_Mode  1

#define packet_size 520

#define ENQ  0x05
#define DLE  0x10
#define ESC  0x1B
#define ETX  0x03

static BYTE VIDTEX_Response[] = "#IBX,PB,DT\015";
static BOOL FILTRN_Status=0;
static LONG total_sent;
static LONG total_read;
static LONG data_sent;
static LONG data_read;
static BYTE tdir[40];
static BYTE filename[40];
static BYTE ttype[40];
static ULONG fsize;
static WORD Read_Err_Count;
static WORD Send_Err_Count;

static WORD BR_Baud_Rate[] = 
    { 110, 300, 450, 1200, 1800, 2400, 4800, 9600 };
static WORD BR_Update_Freq[] = 
    {   5,  17,  21,   89,  129,  160,  234,  346 };
static WORD BR_Char_Time[] = 
    { 100,  33,  22,    8,    6,    4,    2,    1 };

WORD Baud_Rate_Code;

#ifdef CLK_TCK
#define clk 1
static clock_t start;
#else
#define clk 0
#define CLK_TCK 1
static WORD start;

private WORD clock ()
    {
    return 0;
    }
#endif

static BYTE *s_malloc(size)
UWORD size;
    {
    /* allocate space for the protocol library */
    return malloc (size);
    }

static void s_free(ptr)
BYTE *ptr;
    {
    /* release space requested by the protocol library */
    free (ptr);
    return;
    }

static void DLE_B_Seen (apdb)
Application_PDB *apdb;
    {
    /* upon seeing a <DLE> <B> sequence, begin processing B protocol */
    WORD Status;

    Comm_Set_Mode(1); 

    Status = FTR_Handle_Slave_BP(apdb);

    Comm_Set_Mode(3);

    if (FILTRN_Status)
    then
         {
         /* if a file transfer began, then reinitialize the PDB */
         FILTRN_Status = FALSE;
         BP_Init_PDB (BP_B_Plus, BR_Update_Freq [Baud_Rate_Code], 
             BP_Quote_Not_NULL, PDB);
         }

    prints ("\033[16;36H   0:00   \033[23;1H");

    if (!Status)
    then
         prints ("File Transfer Failed\n");
    }

static void Setup_FILTRN ()
    {
    /* set up for a new file transfer */
    if (FILTRN_Status != 0) return;
    FILTRN_Status = 1;

    strcpy (tdir, "");
    strcpy (filename, "");
    strcpy (ttype, "");

    total_sent = 0l;
    total_read = 0l;
    data_sent = 0l;
    data_read = 0l;
    fsize = 0l;

    Read_Err_Count = 0;
    Send_Err_Count = 0;

    prints ("\033[2J");

    prints ("\033[11;20HSent                  ");
    prints ("\033[12;20HRead                  ");
    prints ("\033[13;20HData sent             ");
    prints ("\033[14;20HData received         ");
    prints ("\033[16;20HTime remaining        ");
    start = clock();
    }

static void Print_Msg (s)
BYTE *s;
    {
    /* display a message from the protocol */
    Setup_FILTRN ();
    prints ("\033[16;20H");
    prints (s);
    prints ("\033[0K");
    }

static void Print_File_Line ()
    {
    /* display file information on screen */
    Setup_FILTRN ();
    if (fsize == 0)
         {
         prints ("\033[10;20H");
         prints (tdir);
         prints (filename);
         prints (" as ");
         prints (ttype);
         prints ("\033[0K");
         }
    else
         printf ("\033[10;20H%s %s (%ld) as %s\033[0K", tdir, filename,
              fsize, ttype);
    }

static void D_Sent (count)
UWORD count;
    {
    /* display percentage of file transmitted */
    UWORD data_percent;
    ULONG tmp;
    LONG elapsed;
#ifdef clk
    clock_t now;
#else
    WORD now;
#endif
    LONG est_data;
    LONG rem;
    WORD minutes;
    WORD sec;
    
    Setup_FILTRN ();
    data_sent += (ULONG) count;
    tmp = data_sent * 100;
    data_percent = (WORD) (tmp / fsize);

    now = clock ();
    elapsed = (now - start) / CLK_TCK;
    if (elapsed < 0) printf ("\033[16;39HBad Time: %d", elapsed);

    /* give 5 seconds to allow estimates to settle */
    if (elapsed < 10) 
    then
         prints ("\033[16;36Hestimating");
    else
         {
         /* compensate for send ahead */
         est_data = data_sent - packet_size;
         rem = (fsize - est_data) / (est_data / elapsed);
         minutes = (WORD) (rem / 60);
         sec = (WORD) (rem - minutes * 60);
         printf ("\033[16;36H   %d:%2.2d     ", minutes, sec);
         }

    printf ("\033[13;39H%3.1u %%", data_percent);

    }

static void D_Read (count)
UWORD count;
    {
    /* record count of actual data (Not protocol chars) read */
    Setup_FILTRN ();
    data_read += (ULONG) count;
    printf ("\033[14;34H%8.1lu", data_read);
    }

static void Print_File (name)
BYTE *name;
    {
    /* set up the file name */
    Setup_FILTRN ();
    strcpy (filename, name);
    Print_File_Line();
    }

static void Print_Size (size)
ULONG size;
    {
    /* record the file size (if it is available) */
    fsize = size;
    Setup_FILTRN ();
    Print_File_Line();
    }

static void Print_Mode (type, direction, check_method)
WORD type;                           /* 0 = ascii, 1 = binary */
WORD direction;                      /* 0 = receive, 1 = send */
WORD check_method;                   /* 0 = B-protocol, 1 = CRC */
    {
    /* setup the file transfer mode information */
    Setup_FILTRN ();

    if (type == 0)
         strcpy (ttype, "ASCII");
    else
         strcpy (ttype, "BINARY");

    if (check_method == BP_Check_CRC)
         strcat (ttype, " (CRC)");

    if (direction == 0)
         strcpy (tdir, "Receiving ");
    else
         strcpy (tdir, "Transmitting");
    
    Print_File_Line();
    }

static void S_Error ()
    {
    /* register an error on the display */
    Setup_FILTRN ();
    printf ("\033[7m\033[15;20HSend Errors: %2.1d \033[0m", ++Send_Err_Count);
    }

static void R_Error ()
    {
    /* register an error on the display */
    Setup_FILTRN ();
    printf ("\033[7m\033[15;40HRead Errors: %2.1d \033[0m", ++Read_Err_Count);
    }

static void BPSent (count)
UWORD count;
    {
    /* if we are in a file transfer, display protocol level char counts */
    if (FILTRN_Status != 1) return;

    total_sent+= (ULONG) count;
    Setup_FILTRN ();
    printf ("\033[11;34H%8.1lu", total_sent);
    }

static void BPRead (count)
UWORD count;
    {
    /* if we are in a file transfer, display protocol level char counts */
    if (FILTRN_Status != 1) return;
    
    total_read+= (ULONG) count;
    printf ("\033[12;34H%8.1lu", total_read);
    }

static WORD Remote_Get_Timed(time)
WORD time;
    {
    /* waits for specified time for char to arrive at serial port */
    WORD ch;
    LONG packet_time;

    if (time == 0) return Comm_Get_Char ();

    packet_time = ((long) packet_size * 
         (long) BR_Char_Time [Baud_Rate_Code]);

    Start_Timer (time + (WORD) (packet_time / 1000l) );

    while (((ch = Comm_Get_Char ()) < 0) && (!Timer_Expired ()) );

    return ch;
    }

static WORD Remote_Put(ch)
BYTE ch;
    {
    /* sends a character to the serial port */
    return Comm_Put_Char(ch);
    }

static BOOL Wants_To_Abort()
    {
    /* check keyboard to determine if user wants to abort */
    /* also empties the keyboard buffer */
    WORD Ch;

    while ((Ch = Read_Keyboard()) != -1) 
         if (Ch == ESC or Ch == ETX or Ch == DLE or Ch == Exit_Key) then
             return TRUE;
         else
             Print_Msg("Press ESC to cancel transfer");

    return FALSE;
    }

extern void Delay (time)
UWORD time;
    {
    /* delay for time seconds */
    Start_Timer (time);

    while (!Timer_Expired ());
    return;
    }

static void prints (s)
BYTE *s;
    {
    /* routine used to replace puts, which appends a CR/LF */
    fputs (s, stdout);
    }

void main(argc, argv)
WORD argc;
BYTE **argv;
    {
    Protocol_Desc_Block 
         *pdb;

    Application_PDB 
         *apdb;

    BYTE
         *cp;

    WORD
         Ch,
         Remote_Ch,
         ESC_Seq_State;              /* Escape sequence state variable */

    BOOL
         Want_7_Bit;


    argv++;
    if ( (argc > 1) && (*argv != NULL))
    then
         Baud_Rate_Code = atoi (*argv);
    else
         Baud_Rate_Code = 4;

    printf ("[ Terminal Mode: %d baud ]\n", BR_Baud_Rate[Baud_Rate_Code]);

    apdb = (Application_PDB *) malloc (sizeof (Application_PDB));
    pdb = (Protocol_Desc_Block *) apdb;

    pdb->Send_Buffer             = NULL;
    pdb->Read_Buffer             = NULL;

    pdb->Get_Ch_Timed            = Remote_Get_Timed;
    pdb->Put_Ch                  = Remote_Put;
    pdb->Delay                   = Delay;
    pdb->s_alloc         = s_malloc;
    pdb->s_free                  = s_free;

    pdb->Send_Error              = S_Error;
    pdb->Read_Error              = R_Error;
    pdb->Ch_Sent         = BPSent;
    pdb->Ch_Read         = BPRead;

    apdb->Data_Sent              = D_Sent;
    apdb->Data_Read              = D_Read;
    apdb->Display_Msg            = Print_Msg;
    apdb->Display_File           = Print_File;
    apdb->Display_Size           = Print_Size;
    apdb->Show_Transfer_Mode     = Print_Mode;

    apdb->Create         = Create_File;
    apdb->Open                   = Open_File;
    apdb->Close                  = Close_File;
    apdb->Read                   = Read_File;
    apdb->Write                  = Write_File;
    apdb->Size                   = File_Size;

    apdb->Wants_to_Abort = Wants_To_Abort;

    Want_7_Bit = TRUE;
    ESC_Seq_State = Normal_Mode;

    Comm_Init(0, Baud_Rate_Code, 0);

    BP_Init_PDB (BP_B_Plus, BR_Update_Freq [Baud_Rate_Code], 
         BP_Quote_Not_NULL, pdb);

    BP_Alloc_Buffers (pdb);

    /* 16-bit "raw" character. If > 127 than we have a function key. */
    Ch = Read_Keyboard();

    while (Ch != Exit_Key)
        {
        if (Ch > 0)
            {
             if (Ch == Break_Key)
             then
                 Comm_Send_Break();

            if (Is_Function_Key (Ch))
                {
                 /*
                 * Here to process any local function keys. This sample
                 * program is real dumb. The only function key is the EXIT
                 * key (Alt-X), which is handled as part of the main loop.
                 */
                }
            else
                Comm_Put_Char((BYTE) (Ch & 0x7F));
            }

        Remote_Ch = Comm_Get_Char();

        if (Remote_Ch >= 0)
            {
            if (Want_7_Bit) 
             then 
                 Remote_Ch &= 0x7F;

            switch (ESC_Seq_State)
                {
                case Normal_Mode:
                    switch (Remote_Ch)
                        {
                        case ESC:
                            ESC_Seq_State = 1;
                            break;

                        case ENQ:
                            /* Enquiry -- send response */
                             BP_Seen_ENQ (pdb);
                            break;

                        case DLE:
                             Remote_Ch = (BYTE) Remote_Get_Timed(10);

                            if (Remote_Ch == 'B')
                             then
                                 DLE_B_Seen (apdb);
                             else 
                                 {
                                 /* not a <DLE> <B> sequence */
                                 putchar (DLE);
                                 putchar ((char) Remote_Ch);
                                 }
                             break;

                        default:
                             /* normally, just display the character */
                            putchar ((char) Remote_Ch);
                        }

                    break;

                case Escape_Seq_Mode:
                    /* ESC -- process any escape sequences here */

                    switch (Remote_Ch)
                        {
                        case 'I':
                            /* Reply to the VIDTEX "ESC I" identify sequence */
                            cp = VIDTEX_Response;
                            while (*cp != 0) Comm_Put_Char(*cp++);
                            ESC_Seq_State = 0;
                            break;

                        default:
                            putchar (ESC);
                            putchar ((char) Remote_Ch);
                            ESC_Seq_State = 0;
                        }

                    break;
                }
            }

        Ch = Read_Keyboard();

        }        /* end main terminal loop while */

    Comm_Term();
    BP_Free_Buffers (pdb);
    free (apdb);
    }
/*
*/

