#include <stdio.h>
#include "pvic.h"
#include "locdefs.h"


/* Translate table provides translations for escape sequences. In
is the original escape sequence. Out[0] is the replacement for this
escape sequence while in command mode. Out[1] is the replacement
while in insert mode. */
static struct {
	char *in;
	char *out[2];
} translate_table[] = {
	LOCAL_BACKSPACE_KEY,
	{
		"\010",
		"\010"
	},
	LOCAL_TAB_KEY,
	{
		"\011",
		"\011"
	},
	LOCAL_RETURN_KEY,
	{
		"\015",
		"\015"
	},
	LOCAL_ESC_KEY,
	{
		"\033",
		"\033"
	},
	LOCAL_DEL_KEY,
	{
		"\010",
		"\010"
	},
	DEFAULT_TERMCAP_KEY_F1, 
	{
		":help\n",
		"\033:help\n"
	},
	DEFAULT_TERMCAP_KEY_F2, 
	{
		":help\n",
		"\033:help\n"
	},
	DEFAULT_TERMCAP_KEY_IC, 
	{
		"i",
		"\033"
	},
	DEFAULT_TERMCAP_KEY_DC, 
	{
		"\010",
		"\010"
	},
	DEFAULT_TERMCAP_KEY_HOME, 
	{
		"0",
		"\0330"
	},
	DEFAULT_TERMCAP_KEY_EOL, 
	{
		"$",
		"\033$"
	},
	DEFAULT_TERMCAP_KEY_PPAGE, 
	{
		"\025",
		"\033\025"
	},
	DEFAULT_TERMCAP_KEY_NPAGE, 
	{
		"\004",
		"\033\004"
	},
	DEFAULT_TERMCAP_KEY_UP, 
	{
		"k",
		"\033k"
	},
	DEFAULT_TERMCAP_KEY_DOWN, 
	{
		"j",
		"\033j"
	},
	DEFAULT_TERMCAP_KEY_LEFT, 
	{
		"h",
		"\033h"
	},
	DEFAULT_TERMCAP_KEY_RIGHT, 
	{
		"l",
		"\033l"
	},
	DEFAULT_TERMCAP_KEY_BACKSPACE, 
	{
		"\010",
		"\010"
	},
	ALT_TERMCAP_KEY_F1, 
	{
		":help\n",
		"\033:help\n",
	},
	ALT_TERMCAP_KEY_F2, 
	{
		":help\n",
		"\033:help\n"
	},
	ALT_TERMCAP_KEY_IC, 
	{
		"i",
		"\033"
	},
	ALT_TERMCAP_KEY_DC, 
	{
		"\010",
		"\010"
	},
	ALT_TERMCAP_KEY_HOME, 
	{
		"0",
		"\0330"
	},
	ALT_TERMCAP_KEY_EOL, 
	{
		"$",
		"\033$"
	},
	ALT_TERMCAP_KEY_PPAGE, 
	{
		"\025",
		"\033\025"
	},
	ALT_TERMCAP_KEY_NPAGE, 
	{
		"\004",
		"\033\004"
	},
	ALT_TERMCAP_KEY_UP, 
	{
		"k",
		"\033k"
	},
	ALT_TERMCAP_KEY_DOWN, 
	{
		"j",
		"\033j"
	},
	ALT_TERMCAP_KEY_LEFT, 
	{
		"h",
		"\033h"
	},
	ALT_TERMCAP_KEY_RIGHT, 
	{
		"l",
		"\033l"
	},
	ALT_TERMCAP_KEY_BACKSPACE, 
	{
		"\010",
		"\010"
	},
	NULL,
	{
		NULL, NULL
	}
};


/* Initialize the translation table used by read_input(). */
void init_translate_table()
{
	translate_table[5].in=termcap_key_f1;
	translate_table[6].in=termcap_key_f2;
	translate_table[7].in=termcap_key_ic;
	translate_table[8].in=termcap_key_dc;
	translate_table[9].in=termcap_key_home;
	translate_table[10].in=termcap_key_eol;
	translate_table[11].in=termcap_key_ppage;
	translate_table[12].in=termcap_key_npage;
	translate_table[13].in=termcap_key_up;
	translate_table[14].in=termcap_key_down;
	translate_table[15].in=termcap_key_left;
	translate_table[16].in=termcap_key_right;
	translate_table[17].in=termcap_key_backspace;
}



/* Read characters from the input, and translate incomming escape sequences. */
void read_input()
{
	static char c_buffer[32]="";
	static char c_buffer_count=0;
	int c;
	int i,j;
        int found;
	int found_length;

	if(need_redraw)update_screen(0);
	fflush(stdout);

	/* If nothing in c_buffer, get a character and put it in c_buffer.
	If the character read is '\0' it is translated into a '\200',
	because a '\0' can not apear in a string. */
	if(c_buffer_count==0)
	{
		c=local_get_character();
		if(c=='\0')c='\200';
		c_buffer[c_buffer_count]=c;
		c_buffer_count++;
		c_buffer[c_buffer_count]='\0';
	}

	/* Set found to -1 to indicate we have not yet found a translation
	for the characters in c_buffer. */
	found=-1;

check_input:

	/* Try to find a translation for the characters in c_buffer. */
	for(i=0;translate_table[i].in!=NULL;i++)
	{
		/* Compare c_buffer with translate_table[i].in. */
		for(j=0;j<=c_buffer_count;j++)
		{
			/* If difference between c_buffer and 
			translate_table[i].in are found, we can stop 
			comparing. */
			if(c_buffer[j]!=translate_table[i].in[j])break;

			/* If all characters up to the terminating '\0' are
			equal, we found a matching entry in translate
			table. */
			if(c_buffer[j]=='\0' && 
				translate_table[i].in[j]=='\0')
			{

				/* If we haven't yet found an entry of
				this length, rember it. */
				if(found==-1 || found_length<j)
				{
					found=i;
					found_length=j;
				}

				/* Continue serching. Perhaps there is 
				an longer entry of witch the first part is
				matching. */
				goto next_entry;
			}

		}

		/* If c_buffer[j]=='\0', we have stopped comparing because we
		found an escape sequence of which the first part did match
		with the characters in c_buffer. More characters have to be
		read. */
		if(c_buffer[j]=='\0')
		{
			/* If there is nothing to be read from the terminal,
			or if we do dot know whether there is something to
			be read (local_is_input_pending()==-1), test for
			for new input during one second. */
			if(local_is_input_pending()<=0)
				test_n_seconds_for_input(1);

			/* If there is something to be read from the terminal,
			or if we do dot know whether there is something to
			be read (local_is_input_pending()==-1), read the next
			character, and put it into c_buffer. If the character 
			read is '\0' it is translated into a '\200', because 
			a '\0' can not apear in a string. */
			if(local_is_input_pending()!=0)
			{
				c=local_get_character();
				if(c=='\0')c='\200';
				c_buffer[c_buffer_count]=c;
				c_buffer_count++;
				c_buffer[c_buffer_count]='\0';
				goto check_input;
			}

			/* If a translation is found, the translation is 
			put into the input buffer, and we return to
			the calling routine. */
			else if(found!=-1)
			{
				/* Translation depends on command or insert 
				mode. */
				if(current_status==STATUS_NORMAL)
				{
					put_string_into_input_buffer(
						translate_table[found].out[0]);
				}
				else 
				{
					put_string_into_input_buffer(
						translate_table[found].out[1]);
				}

				/* Remove the original characters from 
				c_buffer. */
				c_buffer_count-=found_length;
				for(i=0;i<c_buffer_count;i++)
				{
					c_buffer[i]=c_buffer[i+found_length];
				}

				/* Return. The calling routine will process 
				the characters put in the input buffer. */
				return;

			}
			else
			{
				/* Put the first character from c_buffer into 
				the input buffer. */
				put_char_into_input_buffer(c_buffer[0]);

				/* And remove it from c_buffer. */
				c_buffer_count--;
				for(i=0;i<=c_buffer_count;i++)
				{
					c_buffer[i]=c_buffer[i+1];
				}

				/* Return. The calling routine will process 
				the characters put in the input buffer. */
				return;
			}
		}
next_entry:;
	}

	/* If a translation is found, the translation is put into
	the input buffer. */
	if(found!=-1)
	{
		/* Translation depends on command or insert mode. */
		if(current_status==STATUS_NORMAL)
		{
			put_string_into_input_buffer(
				translate_table[found].out[0]);
		}
		else 
		{
			put_string_into_input_buffer(
				translate_table[found].out[1]);
		}

		/* Remove the original characters from c_buffer. */
		c_buffer_count-=found_length;
		for(i=0;i<=c_buffer_count;i++)
		{
			c_buffer[i]=c_buffer[i+found_length];
		}

		/* Return. The calling routine will process the characters 
		put in the input buffer. */
		return;

	}

	/* Put the first character from c_buffer into the input buffer. */
	put_char_into_input_buffer(c_buffer[0]);

	/* And remove it from c_buffer. */
	c_buffer_count--;
	for(i=0;i<=c_buffer_count;i++)
	{
		c_buffer[i]=c_buffer[i+1];
	}

	/* Return. The calling routine will process the characters 
	put in the input buffer. */
	return;
}




#define	RBSIZE	1024
static char input_buffer[RBSIZE];
static char *pointer_into_input_buffer = NULL;

void put_string_into_input_buffer(s)
char *s;
{
	if (s == NULL) {		/* clear the stuff buffer */
		pointer_into_input_buffer = NULL;
		return;
	}

	if (pointer_into_input_buffer == NULL) {
		strcpy(input_buffer,s);
		pointer_into_input_buffer = input_buffer;
	} else
		strcat(input_buffer,s);
}



void put_char_into_input_buffer(c)
char c;
{
	int i;

	if (pointer_into_input_buffer == NULL) 
	{
		input_buffer[0]=c;
		input_buffer[1]='\0';
		pointer_into_input_buffer = input_buffer;
	} 
	else 
	{
		for(i=0;input_buffer[i]!='\0';i++);
		input_buffer[i]=c;
		input_buffer[i+1]='\0';
	}
}



void put_int_into_input_buffer(n)
int n;
{
	char buffer[32];

	sprintf(buffer,"%d",n);
	put_string_into_input_buffer(buffer);
}



int get_char_from_input_buffer()
{
	int c;

	for(;;)
	{
		if ( pointer_into_input_buffer != NULL ) 
		{
			c = *pointer_into_input_buffer++;
			if ( *pointer_into_input_buffer == '\0' ) 
			{
				input_buffer[0] = '\0';
				pointer_into_input_buffer = NULL;
			}
			return(c);
		}
		read_input();
	} 
}



/*
 * Return non-zero if input is pending.
 */
int is_input_pending()
{
	return (pointer_into_input_buffer != NULL) || 
		local_is_input_pending()>0;
}




/* Loop for a while testing for new input. */
void test_n_seconds_for_input(seconds)
int seconds;
{
	long t;

	fflush(stdout);

	t=time(NULL);
	while(time(NULL)<t+1+seconds && local_is_input_pending()<=0);
}



