/*      Find the NEXT/PREVIOUS:
 *      - SENTENCE      find_sent (dir)
 *      - PARAGRAPH     find_param (dir)
 *      - FUNCTION      find_function (dir)
 *
 *      I've split these off from SEARCH.C, because they're alike and
 *      SEARCH.C is a big file already.  find_function() was already there.
 *      I added find_sent() and find_param().  -  Dave Tutelman
 *  v1.1 Toad Hall Tweak, 20 Apr 90
 */

#include <stdio.h>
#include "pvic.h"
#include "locdefs.h"

/* We'll be doing some classification of input characters, into: */
#define BLANK           0       /* Whitespace */
#define DOT             1       /* Period, exclamation, q-mark */
#define END_OF_LINE     2       /* End-of-line */
#define OTHER           3       /* Any other non-blank stuff */

extern  int     operator;       /* From normal.c, is there an operator
						 * pending?
						 */

int
inclass (c)
  char c;
{
	switch (c) {
	  case ' ':
	  case '\t':
		return BLANK;
	  case '.':
	  case '!':
	  case '?':
		return DOT;
	  case CTRL_J:
	  case CTRL_M:
	  case '\0':
		return END_OF_LINE;
	  default:
		if (c<' ' || c>'~')     return END_OF_LINE;
		else                    return OTHER;
	}
}

/* We'll also need to (1) tell if a line is just blanks, and
 *                    (2) skip to the next OTHER character.
 * Here are a couple of functions to do it.
 */

int
blank_line (line)
  LPTR *line;
{
	char    *p;
	int     class;

	if (! line)     return (1);
	for (p = line->linep->s; (class=inclass(*p))!=END_OF_LINE; p++)
		if (class!=BLANK)       return (0);
	return (1);
}


LPTR *
skip_to_text (lp, dir)
  LPTR *lp;
  int  dir;
{
	LPTR *lpp;

	lpp = lp;
	while (inclass( CHAR( lpp )) != OTHER) {
		lpp = (dir==SEARCH_FORWARD) ? next_char (lpp) : previous_char (lpp);
		if (!lpp) return (lp);          /* hit the end */
	}
	return (lpp);
}


/*
 * find_sent (dir) - Find the next sentence in direction 'dir'
 *
 * Return (1) if a sentence was found.
 *
 * Algorithm: found end of a sentence if:
 *   FWD - current char is BLANK | END_OF_LINE and last is DOT.
 *   BKWD- current char is DOT and last is BLANK | END_OF_LINE.
 * In either case, we then have to skip to text at beginning of next sentence.
 *
 */
int
find_sent (dir)
int     dir;
{
	LPTR    *curr, *last;   /* LPTR for current & last characters */
	int     ccurr, clast;           /* class of curr and last characters */
	int     oldindex;                       /* need to keep in case search fails */

	curr  = cursor_char;
	oldindex = curr->index;
	/* Get INTO most recent sentence sentence. */
	if (dir==SEARCH_BACKWARD)
		curr = previous_char (curr);
	curr = skip_to_text (curr, SEARCH_BACKWARD);
	ccurr = OTHER;


	do {
		/* Take a step */
		last = curr; clast = ccurr;

		curr = (dir == SEARCH_FORWARD) ? next_char(curr) : previous_char(curr);
		ccurr = inclass (CHAR( curr ));

		/* Test halting condition */
		if (dir==SEARCH_FORWARD &&
		    (ccurr==BLANK || ccurr==END_OF_LINE) && clast==DOT) {
			set_pc_mark();
			last = skip_to_text (last, SEARCH_FORWARD);
			*cursor_char = *last;
			return (1);
		}
		else if (dir==SEARCH_BACKWARD &&
		     ccurr==DOT && (clast==BLANK || clast==END_OF_LINE)) {
			set_pc_mark();
			last = skip_to_text (last, SEARCH_FORWARD);
			*cursor_char = *last;
			return (1);
		}
	} while (curr != NULL);

	cursor_char->index = oldindex;     /* restore if search failed */
	return (0);
}


/*
 * find_param(dir) - Find the next paragraph in direction 'dir'
 *
 * Return (1) if a paragraph was found.
 *
 * Algorithm: found beginning of paragraph if:
 *   FWD - current line is non-blank and last is blank.
 *   BKWD- current line is blank and last is non-blank.
 * Then we skip to the first non-blank, non-dot text.
 *
 */
int
find_param(dir)
int     dir;
{
	LPTR    *curr, *last;   /* current & last lines */
#ifdef NEVER_USED               /* v1.1 */
	LPTR    *marker;                /* end of current para */
#endif
	int     bcurr, blast;   /* "blankness" value for lines */

	curr  = cursor_char;
	bcurr = (dir==SEARCH_FORWARD) ? (0) : (1);      /* keeps us from passing the
											 * text initially. */

	do {
		/* Take a step */
		last = curr; blast = bcurr;
		curr = (dir == SEARCH_FORWARD) ? next_line(curr) : previous_line(curr);
		bcurr = blank_line (curr);

		/* Test halting condition */
		if (dir==SEARCH_FORWARD && bcurr && !blast) {
			set_pc_mark();
			curr = skip_to_text (curr, SEARCH_FORWARD);
			*cursor_char = *curr;
			return (1);
		}
		else if (dir==SEARCH_BACKWARD && bcurr && !blast) {
			set_pc_mark();
			last = skip_to_text (last, SEARCH_FORWARD);
			*cursor_char = *last;
			return (1);
		}
	} while (curr != NULL);

	return (0);
}


/*
 * find_function(dir) - Find the next function in direction 'dir'
 *
 * Return (1) if a function was found.
 *
 * Algorithm depends on a style of C coding in which the ONLY '{'
 * in the first column occurs at the beginning of a function definition.
 * This is a good and common style, but not syntactically required by C.
 */
int
find_function(dir)
int     dir;
{
	LPTR    *curr;

	curr = cursor_char;

	do {
		curr = (dir == SEARCH_FORWARD) ? next_line(curr) : previous_line(curr);

		if (curr != NULL && curr->linep->s[0] == '{') {
			set_pc_mark();
			*cursor_char = *curr;
			return (1);
		}
	} while (curr != NULL);

	return (0);
}

