/* 
 * Code to handle user-settable parameters. This is all pretty much table-
 * driven. To add a new parameter, put it in the params array, and add a
 * macro for it in param.h. If it's a numeric parameter, add any necessary
 * bounds checks to do_set(). String parameters aren't currently supported.
 */

#include <stdio.h>
#include "pvic.h"
#include "locdefs.h"

struct  param   params[] = {

	{ "lines",      "lines",        24,     NUMERIC_PARAMETER },
	{ "report",     "report",       5,      NUMERIC_PARAMETER },
	{ "scroll",     "scroll",       12,     NUMERIC_PARAMETER },
	{ "tabstop",    "ts",           8,      NUMERIC_PARAMETER },

	{ "autoindent", "ai",           0,      BOOLEAN_PARAMETER },
	{ "backup",     "bk",           0,      BOOLEAN_PARAMETER },
	{ "errorbells", "eb",           0,      BOOLEAN_PARAMETER },
	{ "ignorecase", "ic",           0,      BOOLEAN_PARAMETER },
	{ "list",       "list",         0,      BOOLEAN_PARAMETER },
	{ "modelines",  "ml",           1,      BOOLEAN_PARAMETER },
	{ "number",     "nu",           0,      BOOLEAN_PARAMETER },
	{ "showmatch",  "sm",           0,      BOOLEAN_PARAMETER },
	{ "showmode",   "mo",           1,      BOOLEAN_PARAMETER },
	{ "vbell",      "vb",           0,      BOOLEAN_PARAMETER },
	{ "wrapscan",   "ws",           1,      BOOLEAN_PARAMETER },
	{ "",           "",             0,      0, }            /* end marker */

};

static  void    showparms();

void
do_set(arg)
char    *arg;           /* parameter string */
{
	register int    i;
	register char   *s;
	int     did_lines = (0);
	int     state = (1);            /* new state of boolean parms. */

	if (arg == NULL) {
		showparms((0));
		return;
	}
	if (strncmp(arg, "all", 3) == 0) {
		showparms((1));
		return;
	}
	if (strncmp(arg, "no", 2) == 0) {
		state = (0);
		arg += 2;
	}

	for (i=0; params[i].fullname[0] != '\0' ;i++) {
		s = params[i].fullname;
		if (strncmp(arg, s, strlen(s)) == 0)    /* matched full name */
			break;
		s = params[i].shortname;
		if (strncmp(arg, s, strlen(s)) == 0)    /* matched short name */
			break;
	}

	if (params[i].fullname[0] != '\0') {    /* found a match */
		if (params[i].flags & NUMERIC_PARAMETER) {
			did_lines = (i == PARAMETER_LINES);
			if (arg[strlen(s)] != '=' || state == (0))
				error_message("Invalid set of numeric parameter");
			else {
				params[i].value = atoi(arg+strlen(s)+1);
				params[i].flags |= PARAMETER_HAS_CHANGED;
			}
		} else /* boolean */ {
			if (arg[strlen(s)] == '=')
				error_message("Invalid set of boolean parameter");
			else {
				params[i].value = state;
				params[i].flags |= PARAMETER_HAS_CHANGED;
			}
		}
	} else
		error_message("Unrecognized 'set' option");

	/*
	 * Update the screen in case we changed something like "tabstop"
	 * or "list" that will change its appearance.
	 */
	update_screen(0);

	if (did_lines) {
		current_lines=PARAMETER_VALUE(PARAMETER_LINES);
		if(current_lines<4)current_lines=
			PARAMETER_VALUE(PARAMETER_LINES)=4;
		else if(current_lines>termcap_lines)current_lines=
			PARAMETER_VALUE(PARAMETER_LINES)=termcap_lines;
		if (alloc_screen() == -1) return;        /* allocate new screen buffers */
		clear_screen();
		update_screen(0);
	}
	/*
	 * Check the bounds for numeric parameters here
	 */
	if (PARAMETER_VALUE(PARAMETER_TABSTOP) <= 0 || PARAMETER_VALUE(PARAMETER_TABSTOP) > 32) {
		error_message("Invalid tab size specified");
		PARAMETER_VALUE(PARAMETER_TABSTOP) = 8;
		return;
	}

	if (PARAMETER_VALUE(PARAMETER_SCROLL) <= 0 || PARAMETER_VALUE(PARAMETER_SCROLL) > termcap_lines) {
		error_message("Invalid scroll size specified");
		PARAMETER_VALUE(PARAMETER_SCROLL) = 12;
		return;
	}

	/*
	 * Check for another argument, and call do_set() recursively, if
	 * found. If any argument results in an error, no further
	 * parameters are processed.
	 */
	while (*arg != ' ' && *arg != '\t') {   /* skip to next white space */
		if (*arg == '\0')
			return;                 /* end of parameter list */
		arg++;
	}
	while (*arg == ' ' || *arg == '\t')     /* skip to next non-white */
		arg++;

	if (*arg)
		do_set(arg);     /* recurse on next parameter */
}

static  void
showparms(all)
int     all;    /* show ALL parameters */
{
	register struct param   *p;
	char    buf[64];

	goto_command((1), 0);
	fprintf(stdout,"Parameters:\n");

	for (p = &params[0]; p->fullname[0] != '\0' ;p++) {
		if (!all && ((p->flags & PARAMETER_HAS_CHANGED) == 0))
			continue;
		if (p->flags & BOOLEAN_PARAMETER)
			fprintf(stdout,"\t%s%s\n",
				(p->value ? "" : "no"), p->fullname);
		else
			fprintf(stdout,"\t%s=%d\n",p->fullname, p->value);

	}
	wait_return();
}
