/* 
 *
 * Routines to save and retrieve marks.
 */

#include <stdio.h>
#include "pvic.h"
#include "locdefs.h"

#define NMARKS  10              /* max. # of marks that can be saved */

struct  mark {
	char    name;
	LPTR    pos;
};

static  struct  mark    mlist[NMARKS];
static  struct  mark    pcmark;         /* previous context mark */
static  int     pcvalid = (0);  /* true if pcmark is valid */

/*
 * set_mark(c) - set mark 'c' at current cursor position
 *
 * Returns (1) on success, (0) if no room for mark or bad name given.
 */
int
set_mark(c)
register char   c;
{
	register int    i;

	if (!is_alpha(c))
		return (0);

	/*
	 * If there is already a mark of this name, then just use the
	 * existing mark entry.
	 */
	for (i=0; i < NMARKS ;i++) {
		if (mlist[i].name == c) {
			mlist[i].pos = *cursor_char;
			return (1);
		}
	}

	/*
	 * There wasn't a mark of the given name, so find a free slot
	 */
	for (i=0; i < NMARKS ;i++) {
		if (mlist[i].name == '\0') {    /* got a free one */
			mlist[i].name = c;
			mlist[i].pos = *cursor_char;
			return (1);
		}
	}
	return (0);
}

/*
 * set_pc_mark() - set the previous context mark to the current position
 */
void
set_pc_mark()
{
	pcmark.pos = *cursor_char;
	pcvalid = (1);
}

/*
 * get_mark(c) - find mark for char 'c'
 *
 * Return pointer to LPTR or NULL if no such mark.
 */
LPTR *
get_mark(c)
register char   c;
{
	register int    i;

	if (c == '\'' || c == '`')      /* previous context mark */
		return pcvalid ? &(pcmark.pos) : (LPTR *) NULL;

	for (i=0; i < NMARKS ;i++) {
		if (mlist[i].name == c)
			return &(mlist[i].pos);
	}
	return (LPTR *) NULL;
}

/*
 * clear_all() - clear all marks
 *
 * Used mainly when trashing the entire buffer during ":e" type commands
 */
void
clear_all()
{
	register int    i;

	for (i=0; i < NMARKS ;i++)
		mlist[i].name = '\0';
	pcvalid = (0);
}

/*
 * clear_mark(line) - clear any marks for 'line'
 *
 * Used any time a line is deleted so we don't have marks pointing to
 * non-existent lines.
 */
void
clear_mark(line)
register LINE   *line;
{
	register int    i;

	for (i=0; i < NMARKS ;i++) {
		if (mlist[i].pos.linep == line)
			mlist[i].name = '\0';
	}
	if (pcvalid && (pcmark.pos.linep == line))
		pcvalid = (0);
}
