/* 
 *
 * Various allocation routines and routines returning information about
 * allocated objects.
 */

#include <stdio.h>
#include "pvic.h"
#include "locdefs.h"

char * alloc(size)
unsigned size;
{
	char    *p;             /* pointer to new storage space */

	p = malloc(size);
	if ( p == (char *)NULL ) {      /* if there is no more room... */
		error_message("alloc() is unable to find memory!");
	}
	return(p);
}

char * strsave(string)
char    *string;
{
	return(strcpy(alloc((unsigned)(strlen(string)+1)),string));
}

alloc_screen()
{
	/*
	 * If we're changing the size of the screen, free the old arrays
	 */
	if (real_screen != NULL)
		free(real_screen);
	if (next_screen != NULL)
		free(next_screen);

	real_screen = malloc((unsigned)(current_lines*current_columns));
	next_screen = malloc((unsigned)(current_lines*current_columns));
	if (!real_screen || !next_screen)
		return (-1);
	else    return (0);
}

/*
 * Allocate and initialize a new line structure with room for
 * 'nchars'+1 characters. We add one to nchars here to allow for
 * null termination because all the callers would just do it otherwise.
 */
LINE *
newline(nchars)
int     nchars;
{
	register LINE   *l;

	if ((l = (LINE *) alloc(sizeof(LINE))) == NULL)
		return (LINE *) NULL;

	l->s = alloc((unsigned) (nchars+1));    /* the line is empty */
	if (l->s == NULL)       return (LINE *) NULL;
	l->s[0] = '\0';
	l->size = nchars + 1;

	l->prev = (LINE *) NULL;        /* should be initialized by caller */
	l->next = (LINE *) NULL;

	return l;
}

/*
 * file_alloc() - construct an initial empty file buffer
 */
void
file_alloc()
{
	if ((file_memory->linep = newline(0)) == NULL) {
		fprintf(stderr,"Unable to allocate file memory!\n");
		exit(1);
	}
	if ((top_of_file->linep = newline(0)) == NULL) {
		fprintf(stderr,"Unable to allocate file memory!\n");
		exit(1);
	}
	if ((end_of_file->linep = newline(0)) == NULL) {
		fprintf(stderr,"Unable to allocate file memory!\n");
		exit(1);
	}
	file_memory->index = 0;
	top_of_file->index = 0;
	end_of_file->index = 0;

	top_of_file->linep->next = file_memory->linep;  /* connect top_of_file to file_memory */
	file_memory->linep->prev = top_of_file->linep;

	file_memory->linep->next = end_of_file->linep;  /* connect file_memory to end_of_file */
	end_of_file->linep->prev = file_memory->linep;

	*cursor_char = *file_memory;
	*top_char  = *file_memory;

	file_memory->linep->num = 0;
	end_of_file->linep->num = 0xffff;

	clear_all();               /* clear all marks */
	u_clear();              /* clear the undo buffer */
}

/*
 * freeall() - free the current buffer
 *
 * Free all lines in the current buffer.
 */
void
freeall()
{
	register LINE   *lp, *xlp;

	for (lp = top_of_file->linep; lp != NULL ;lp = xlp) {
		if (lp->s != NULL)
			free(lp->s);
		xlp = lp->next;
		free((char *)lp);
	}

	cursor_char->linep = NULL;         /* clear pointers */
	top_of_file->linep = NULL;
	file_memory->linep = NULL;
	end_of_file->linep = NULL;

	u_clear();
}

/*
 * buffer_empty() - return (1) if the buffer is empty
 */
int
buffer_empty()
{
	return (buf1line() && file_memory->linep->s[0] == '\0');
}

/*
 * buf1line() - return (1) if there is only one line
 */
int
buf1line()
{
	return (file_memory->linep->next == end_of_file->linep);
}

/*
 * line_empty() - return (1) if the current line is empty
 */
int
line_empty()
{
	return (cursor_char->linep->s[0] == '\0');
}

/*
 * end_of_line() - return (1) if the given position is at end of line
 *
 * This routine will probably never be called with a position resting
 * on the '\0' byte, but handle it correctly in case it happens.
 */
int
end_of_line(p)
register LPTR   *p;
{
	return (p->linep->s[p->index] == '\0' || p->linep->s[p->index+1] == '\0');
}
/*
 * can_increase(n) - returns (1) if the current line can be increased 'n' bytes
 *
 * This routine returns immediately if the requested space is available.
 * If not, it attempts to allocate the space and adjust the data structures
 * accordingly. If everything fails it returns (0).
 */
int
can_increase(n)
register int    n;
{
	register int    nsize;
	register char   *s;             /* pointer to new space */

	nsize = strlen(cursor_char->linep->s) + 1 + n;     /* size required */

	if (nsize <= cursor_char->linep->size)
		return (1);

	/*
	 * Need to allocate more space for the string. Allow some extra
	 * space on the assumption that we may need it soon. This avoids
	 * excessive numbers of calls to malloc while entering new text.
	 */
	if ((s = alloc((unsigned) (nsize + SLOP))) == NULL) {
		error_message("Can't add anything, file is too big!");
		current_status = STATUS_NORMAL;
		return (0);
	}

	cursor_char->linep->size = nsize + SLOP;
	strcpy(s, cursor_char->linep->s);
	free(cursor_char->linep->s);
	cursor_char->linep->s = s;
	
	return (1);
}

char * mkstr(c)
char    c;
{
	static  char    s[2];

	s[0] = c;
	s[1] = '\0';

	return s;
}
