/*
  These routines deal with OS9 type I/O to the OS9 disk.

  These are somewhat low level routines.
  
  A few routines are almost file level, but not quite.
*/


void OpenOS9Disk(void)
/*
  Open up the disk as an OS9 disk
*/
{
if (options.debug) printf("OpenOS9Disk\n");
PrepOS9Disk();
ResetDisk();
ReadLSN0();
ReadDAM();
}

void CloseOS9Disk(void)
/*
  Done with OS9 disk.  CLose it and make it PC again.
*/
{
if (options.debug) printf("CloseOS9Disk\n");
FinishedDisk();
ResetDisk();
}

void ReadLSN0(void)
/*
  read the Ident sector at LSN0 so we know what params to use
  You don't need to call this routine.
*/
{struct s_ddsect OS9_ddsect;int x;unsigned char *cp;
if (options.debug) printf("Reading LSN0\n");

ReadBytes(0L,&OS9_ddsect,sizeof(OS9_ddsect));
diskinfo=CvtDDSect(&OS9_ddsect);

if (options.debug > 2)
   {printf("LSN0 dump: (after conversion)\n");cp=(unsigned char *)&diskinfo;
    for (x=0;x<sizeof(diskinfo);x++) fprintf(stderr,"%02x  ",*cp++);
    printf("\n");
   }

diskinfo.sides=(diskinfo.dd_fmt & 1) + 1;
diskinfo.sectors=diskinfo.dd_tks;
/* Should I use dd_tks or dd_spt ?????? */
diskinfo.tracks=(diskinfo.dd_tot / diskinfo.sectors);
diskinfo.cylsectors=diskinfo.sectors * diskinfo.sides;
/* Set up a fake dir entry to point to the root dir */
strcpy(rootdirentry.dir_name,"Root_Dir");
rootdirentry.dir_addr=diskinfo.dd_dir;

if (diskinfo.dd_tks != diskinfo.dd_spt)
  {fprintf(stderr,"dd_tks is %d and dd_spt is %d\n",
           diskinfo.dd_tks,diskinfo.dd_spt);
   fprintf(stderr,"I can only handle disks where they are the same\n");
   QuitProgram(0);
  }

if (diskinfo.dd_bit != 1)
  {fprintf(stderr,"I can only handle disks with 1 sectors per cluster.\n");
   QuitProgram(0);
  }
}

void ReadDAM(void)
{
damsize=diskinfo.dd_map;
if (options.debug) printf("Reading DAMsize: %d\n",damsize);

if (damsize >8191)
  {fprintf(stderr,"This disk has a Disk Allocation Map of %u bytes\n",damsize);
   fprintf(stderr,"I can only handle up to 8192 bytes\n");
   QuitProgram(0);
  }

ReadBytes(256L,dam,damsize);
}

void WriteDAM(void)
{
if (options.debug) printf("Writing DAM\n");
WriteBytes(256L,dam,damsize);
}

void ShowSize(void)
/* Show the disk size, bytes used and free */
{unsigned long bitclear;
 unsigned int x;
 int temp;

diskinfo.sectorsused=0L;
for (x=0;x<damsize;x++)
   {temp = BitCount(dam[x]);
   diskinfo.sectorsused += temp;
   if (options.debug > 2) printf("DAM #%d = %d, %d bits\n",x,dam[x],temp);
   }

diskinfo.sectorsfree=diskinfo.dd_tot - diskinfo.sectorsused;
diskinfo.bytesused = diskinfo.sectorsused * 256L;
diskinfo.bytesfree = diskinfo.sectorsfree * 256L;
if (options.debug)
   {printf("DAM used bits: %lu  DAM free bits: %lu\n",
           diskinfo.sectorsused,diskinfo.sectorsfree);
    printf("(bytes)   Disk used: %lu  Disk free: %lu\n",
           diskinfo.bytesused,diskinfo.bytesfree);
    printf("(sectors) Disk used: %lu  Disk free: %lu\n",
           diskinfo.sectorsused,diskinfo.sectorsfree);
   }

bitclear=FirstBitClear();
if (options.debug) printf("First free LSN: %lu\n",bitclear);
}


void ReadOS9Sector(long lsn,unsigned char *buffer)
/*
  This routine deals with reading the individual OS9 logical
  sectors (LSN's).
  It also has retries in case one attempt fails.

  If you need to read a sector, you can call this one.  Or better yet,
  convert the LSN into a byte offset (*256L) and call ReadBytes()

  This is level 1 of the disk Input routines.  All it knows how to do
  is read in an individual sector (LSN) and performs retries if the
  read fails.
*/
{int tries=5;int failed=TRUE;int x;
/* ResetDisk(); */

if (lsn > diskinfo.dd_tot)
  {fprintf(stderr,"Trying to read lsn %lu and disk only has %lu\n",
           lsn,diskinfo.dd_tot);
   QuitProgram(EINVDAT);
  }

do
  {failed=!ReadTheDisk(lsn,buffer);
   if (failed) ResetDisk();
  } while ((tries--) && (failed));
if (failed)
  {fprintf(stderr,"Retry read failed.  Giving up\n");
   if (options.debug > 2)
     {for (x=0;x<256;x++) fprintf(stderr,"%02x  ",buffer[x]);
      fprintf(stderr,"\n");
     }
   QuitProgram(EFAULT);
  }
}


void WriteOS9Sector(long lsn,unsigned char *buffer)
/*
  This routine deals with writing the individual OS9 logical
  sectors (LSN's).

  It also has retries in case one attempt fails.

  If you need to write a sector, you can call this one.  Or better yet,
  convert the LSN into a byte offset (*256L) and call WriteBytes()

  This is level 1 of the disk Output routines.  All it knows how to do
  is Write out an individual sector (LSN) and performs retries if the
  Write fails.
*/
{int tries=5;int failed=TRUE;int x;
/* ResetDisk(); */

if (lsn > diskinfo.dd_tot)
  {fprintf(stderr,"Trying to write lsn %lu and disk only has %lu\n",
           lsn,diskinfo.dd_tot);
   QuitProgram(EINVDAT);
  }

do
  {failed=!WriteTheDisk(lsn,buffer);
   if (failed) ResetDisk();
  } while ((tries--) && (failed));
if (failed)
  {fprintf(stderr,"Retry write failed.  Giving up\n");
   if (options.debug > 2)
     {for (x=0;x<256;x++) fprintf(stderr,"%02x  ",buffer[x]);
      fprintf(stderr,"\n");
     }
   QuitProgram(EFAULT);
  }
}

/*
===================================================================
This is the routine that you actually call to read from
an absolute location on the disk
===================================================================
*/

void ReadBytes(long pos, void * buffer,unsigned int len)
/*
  read LEN num of bytes that can span sectors

  This is called with a byte offset rather than a LSN & offset.  It's
  just easier that way.

  When it reaches the end of a sector, it just goes to the next one.
  This should be okay because we shouldn't ever ask it for more
  than is available in consecutive segments.

  This is level 2 of the sector input routines.  It reads data
  from sequentially numbered sectors (not segments!).  You are
  responsible for not reading past the end of a segment.
*/
{long lsn;int offset;
 unsigned char *ptr=(unsigned char *)buffer;
 unsigned char sector[1024];

if (options.debug >1) printf("\nREADBYTE: Pos=%lu Len=%u\n",pos,len);

lsn=pos / 256L;offset = (int) (pos % 256L);
ReadOS9Sector(lsn,sector);

while (len--)
  {*ptr++=sector[offset++];
   if ((offset==256) && (len!=0L))
      {lsn++;ReadOS9Sector(lsn,sector);offset=0;}
  }
}

/*
===================================================================
This is the routine that you actually call to write to
an absolute location on the disk
===================================================================
*/

void WriteBytes(long pos, void * buffer,unsigned int len)
/*
  Write LEN num of bytes that can span sectors

  This is called with a byte offset rather than a LSN & offset.  It's
  just easier that way.

  When it reaches the end of a sector, it just goes to the next one.
  This should be okay because we shouldn't ever ask it for more
  than is available in consecutive segments.

  This is level 2 of the sector Output routines.  It writes data
  from sequentially numbered sectors (not segments!).  You are
  responsible for not reading past the end of a segment.
*/
{long lsn;int offset;
 unsigned char *ptr=(unsigned char *)buffer;
 unsigned char sector[1024];

if (options.debug >1) printf("\nWRITEBYTE: Pos=%lu Len=%u\n",pos,len);

lsn=pos / 256L;offset = (int) (pos % 256L);

/*
  If we aren't starting at the beginning of a sector or we are only
  doing part of it, load what is already on disk.
*/
if ((offset !=0) || (len <= 256)) ReadOS9Sector(lsn,sector);

while (len--)
  {sector[offset++]=*ptr++;
   /* is sector full or are we finished writing to the sector? */
   if ( (offset == 256) || (len == 0) )
     {
      WriteOS9Sector(lsn,sector);  /* Write it out. */
      offset=0;lsn++;              /* Next sector */
      /* If we are only writing to *part* of the next sector, */
      /* we have to read it in first. */
      if ((len > 0) && (len <= 256)) ReadOS9Sector(lsn,sector);
     }
  }
}



