/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/sched.c,v 1.4 1994/02/15 21:18:38 ms5u Exp $";
/* sched.c
 *
 * This module contains the routines necessary for the internal scheduler.
 *
 * No guarantees are made as to the accuracy of this scheduler.  The primary
 * routine simply waits one minute before the next check against the internal
 * schedule.  It is possible, due to the PC's internal clock "ticking" 18.72 
 * times per second, that there may be an increase or decrease of a whole
 * second over an extended period due to rounding.
 *
 * Routines called by the scheduler should not assume that if they are called
 * at 12:34:59 then the next call will be at 12:35:59 -- it might be at
 * 12:36:00.
 *
 * Revision History:
 *   $Log: sched.c,v $
 * Revision 1.4  1994/02/15  21:18:38  ms5u
 * Add call to DELOLD_delete_files to delete files daily.
 *
 * Revision 1.3  1994/02/15  18:25:15  ms5u
 * Change version to V1.1.
 *
 * Revision 1.2  1993/10/05  15:35:01  ms5u
 * Add copyright header.
 *
 * Revision 1.1  1993/09/13  19:36:19  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <process.h>
#include <errno.h>
#include <time.h>
#include "prototyp.h"

struct SCHED_entry {
	int minute, hour, weekday;
	void (*schedfunc)(void);
} SCHED_entry;

/* Format for the following structure is:
 *	minute, hour, day-of-week, routine-to-call
 * For the first three arguments, the values can either be:
 *	- A non-negative number indicating at which value the routine
 *	  should be called
 *	- A negative number indicating for what period the routine
 *	  should be called
 * For example, a "30" for the minute field represents something that
 * is done on the half hour.  A "-5", on the other hand, would be called
 * every five minutes (at :00, :05, :10, etc.).  The "-1" matches at every
 * number since they are all multiples of 1.
 *
 * The list should be terminated as "{ 99, 99, 99, NULL }".
 */

static struct SCHED_entry SCHEDule[] =
	{ { -1, -1, -1, LOAD_compute_period_load },
	  { -1, -1, -1, LOAD_graph_console },
	  { -1, -1, -1, LOAD_store },
	  { -5, -1, -1, LOAD_flush },
	  {  0,  0, -1, LOAD_update_filename },
	  {  0,  0, -1, DELOLD_delete_files },
	  {  0,  0,  0, REPORT_init },
	  { 99, 99, 99, NULL } };

int SCHED_match(int curval, int testval)
{
	return(((testval < 0) && (curval % (-testval) == 0)) ||
	       ((testval >= 0) && (curval == testval)));
}

void SCHED_run()
{
	int index;
	int curmin, curhour, curdow;
	time_t curtime;
	char timestring[6];
	
	while (1)
	{
		delay(60 * 1000);	/* 60 seconds */

		curtime = time(NULL);
		strftime(timestring, sizeof(timestring),
			 "%w%H%M", localtime(&curtime));
		curdow = (timestring[0] - 48);
		curhour = (timestring[1] - 48) * 10 + (timestring[2] - 48);
		curmin = (timestring[3] - 48) * 10 + (timestring[4] - 48);

		index = 0;
		while (SCHEDule[index].minute < 60)
		{
			if (sched_match(curmin, SCHEDule[index].minute) &&
			    sched_match(curhour, SCHEDule[index].hour) &&
			    sched_match(curdow, SCHEDule[index].weekday))
				SCHEDule[index].schedfunc();
			index++;
		}
	}
}

void SCHED_init()
{
	int retstat;
	 
	retstat = BeginThread(SCHED_run, NULL, 0, NULL);
	if (retstat == EFAILURE)
		perror("Error starting internal SCHEDuler");
}
