/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/report.c,v 1.5 1994/02/15 18:25:03 ms5u Exp $";
/* report.c
 *
 * This module contains the routines necessary to generate the two weekly
 * reports.  The "internal" report will contain the detailed information
 * about the previous week's usage including load graphs, downtime details,
 * and top-user and -network usage.  The "external" report, on the other
 * hand, contains only summary information such as average load over a more
 * extended period, total number of connections, etc.
 *
 * When REPORT_init() is called, it will check for the existance of
 * SYS:\NOVSTAT\IREPORT.nn, where `nn' is last week's week number (of the
 * year) in the event that NovStat was not running during the week crossover.
 * If this file is not found, then the report will be automatically generated
 * for last week.
 *
 * Revision History:
 *   $Log: report.c,v $
 * Revision 1.5  1994/02/15  18:25:03  ms5u
 * Change version to V1.1.
 *
 * Revision 1.4  1993/11/04  22:16:33  ms5u
 * Minor changes.  Check for NULL file and increase copy buffer length.
 *
 * Revision 1.3  1993/10/11  14:27:22  ms5u
 * Only generate a report if we don't have one on disk already.
 *
 * Revision 1.2  1993/10/05  15:35:00  ms5u
 * Add copyright header.
 *
 * Revision 1.1  1993/09/13  19:36:16  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <time.h>
#include <string.h>
#include "prototyp.h"

#define REPORT_WIDTH 72
#define GRAPH_WIDTH 66
#define WHICH_SLOT(min) ((min % 1440) * GRAPH_WIDTH / 1440)

#define DOWN_FUDGE 2	/* Number of minutes before downtime is noted */
#define DAY_NAMES "SUNMONTUEWEDTHUFRISAT"

int REPORT_start_month, REPORT_start_day, REPORT_start_year, REPORT_start_jul,
	REPORT_end_month, REPORT_end_day, REPORT_end_year, REPORT_end_jul,
	REPORT_weeknum;
static char REPORT_time_line[GRAPH_WIDTH + 8];

void REPORT_init_time_line()
{
	int slotpos, hourout, printhour;
	
	/* Set up the time line with a plain bar... */
	strcpy(REPORT_time_line, "     +");
	for (slotpos = 0; slotpos < GRAPH_WIDTH; slotpos++)
		REPORT_time_line[slotpos+6] = '-';
	REPORT_time_line[slotpos+6] = 0;

	for (hourout = 0; hourout < 24; hourout += 4)
	{
		slotpos = WHICH_SLOT(hourout * 60);
		printhour = hourout % 12;
		if (printhour == 0)
		{
			REPORT_time_line[slotpos+6] = '1';
			REPORT_time_line[slotpos+7] = '2';
			REPORT_time_line[slotpos+8] = (hourout == 0) ? 'A' : 'P';
			REPORT_time_line[slotpos+9] = 'M';
		} else {
			REPORT_time_line[slotpos+6] = (printhour + 48);
			REPORT_time_line[slotpos+7] = (hourout < 12) ? 'A' : 'P';
			REPORT_time_line[slotpos+8] = 'M';
		}
	}
}

char *REPORT_server_name()
{
	static char servername[48];
	
	GetFileServerName(0, servername);
	return (servername);
}

char *center_string(char *str)
{
	static char spaces[] = "                                                                                 ";
	static char outline[81];
	int centerlen;
	
	centerlen = (REPORT_WIDTH/2) - (strlen(str) / 2);
	spaces[centerlen] = 0;
	sprintf(outline, "%s%s", spaces, str);
	spaces[centerlen] = ' ';
	
	return(outline);
}

void REPORT_page_hdr(FILE *rptfile, char *subhead, int pagenum)
{
	char tmp[80];
	
	fprintf(rptfile, "\f");

	sprintf(tmp, "%s NetWare Server - %02d/%02d/%02d to %02d/%02d/%02d",
		REPORT_server_name(),
		REPORT_start_month, REPORT_start_day, REPORT_start_year,
		REPORT_end_month, REPORT_end_day, REPORT_end_year);
	fprintf(rptfile, "%s\n", center_string(tmp));

	sprintf(tmp, "Page %d", pagenum);
	fprintf(rptfile, "%s\n", center_string(tmp));
	
	fprintf(rptfile, "%s\n", center_string(subhead));
}

int *REPORT_graph_load(FILE *rptfile)
{
	FILE *loadfile;
	char loadfilename[64];
	int curjul;
	struct tm *timerec;
	struct LOAD_rec loadrec;
	
	int loadtotals[GRAPH_WIDTH], loadcounts[GRAPH_WIDTH];
	int graphcurmin, graphlastmin;
	int *downlist = NULL, numdownlist = 0;
	int loadpct, loadtime;

	REPORT_page_hdr(rptfile, "CPU Utilization Graphs", 1);

	/* While we're going through the loads, we can also check for
	 * downtimes!
	 */
	graphlastmin = 0;
	for (curjul = REPORT_start_jul; curjul <= REPORT_end_jul; curjul++)
	{
		sprintf(loadfilename, "SYS:\\NOVSTAT\\LOADFILE.%03d",
			curjul);
		loadfile = fopen(loadfilename, "rb");
		memset((void *)&loadtotals, 0, sizeof(loadtotals));
		memset((void *)&loadcounts, 0, sizeof(loadcounts));
		while ((loadfile != NULL) && !feof(loadfile))
		{
			fread(&loadrec, sizeof(loadrec), 1, loadfile);
			timerec = localtime(&loadrec.curtime);
			mktime(timerec);
			graphcurmin = (timerec->tm_wday * 24 * 60) +
				(timerec->tm_hour * 60) +
				timerec->tm_min;
			if (graphcurmin - graphlastmin > DOWN_FUDGE)
			{
				numdownlist += 2;
				downlist = (int *)realloc((void *)downlist,
					numdownlist * sizeof(*downlist));
				downlist[numdownlist - 2] = graphlastmin;
				downlist[numdownlist - 1] = graphcurmin;
			}
			graphlastmin = graphcurmin;
			
			loadtotals[WHICH_SLOT(graphcurmin)] +=
				loadrec.curload;
			loadcounts[WHICH_SLOT(graphcurmin)]++;
		}
		fclose(loadfile);
		
		for (loadtime = 0; loadtime < GRAPH_WIDTH; loadtime++)
			if (loadcounts[loadtime] > 0)
				loadtotals[loadtime] /=	loadcounts[loadtime];

		/* Now we have the information for one day... */
		for (loadpct = 70; loadpct >= 0; loadpct -= 10)
		{
			switch (loadpct)
			{
				case 70 : 
				case 10 : fprintf(rptfile, "     |");
					break;

				case 60 :
				case 20 : fprintf(rptfile, "  %02d%%|",
					loadpct);
					break;

				case 50 : fprintf(rptfile, "%c    |",
					DAY_NAMES[(curjul - REPORT_start_jul)*3]);
					break;

				case 40 : fprintf(rptfile, "%c 40%%|",
					DAY_NAMES[(curjul - REPORT_start_jul)*3+1]);
					break;

				case 30 : fprintf(rptfile, "%c    |",
					DAY_NAMES[(curjul - REPORT_start_jul)*3+2]);
					break;

				case 0  : fprintf(rptfile, REPORT_time_line);
					break;
			}
			if (loadpct > 0)
				for (loadtime = 0; loadtime < GRAPH_WIDTH;
					loadtime++)
				{
					if ((loadpct == 70) &&
						(loadtotals[loadtime] > 74))
						fprintf(rptfile, "=\010^");
					else if (loadtotals[loadtime] >
						loadpct - 5)
						fprintf(rptfile, "#");
					else if (loadpct == 40)
						fprintf(rptfile, ".");
					else
						fprintf(rptfile, " ");
					if ((loadpct == 10) &&
						(loadcounts[loadtime] > 0))
						fprintf(rptfile, "\010_");
				}
			fprintf(rptfile, "\n");
		}
	}
	downlist = (int *)realloc((void *)downlist,
		(numdownlist+2) * sizeof(*downlist));
	downlist[numdownlist] = -1;
	downlist[numdownlist+1] = -1;
	return(downlist);
}

void REPORT_generate()
{
	char ireportname[64], ereportname[64], copyiname[64], copyename[64];
	char copyline[512];
	FILE *extreport, *intreport, *copyreport;
	time_t curtime;
	struct tm *pasttime;
	int *downtime;

	curtime = time(NULL);

	sprintf(ireportname, "SYS:\\NOVSTAT\\IREPORT.%02d", REPORT_weeknum);
	sprintf(ereportname, "SYS:\\NOVSTAT\\EREPORT.%02d", REPORT_weeknum);
	strcpy(copyiname, "SYS:\\NOVSTAT\\IREPORT.CUR");
	strcpy(copyename, "SYS:\\NOVSTAT\\EREPORT.CUR");
	
	pasttime = localtime(&curtime);

	/* The day that we want to start is on the Sunday of the previous
	 * week.  Since we know how many days since this past Sunday, we use
	 * that and an additional 7 to get the date for last Sunday.
	 */
	pasttime->tm_mday -= (pasttime->tm_wday + 7);
	pasttime->tm_sec = pasttime->tm_min = pasttime->tm_hour = 0;
	mktime(pasttime);
	REPORT_start_month = pasttime->tm_mon + 1;
	REPORT_start_day = pasttime->tm_mday;
	REPORT_start_year = pasttime->tm_year;
	REPORT_start_jul = pasttime->tm_yday + 1;    /* Julian starts at 1 */

	pasttime->tm_mday += 6;
	mktime(pasttime);
	REPORT_end_month = pasttime->tm_mon + 1;
	REPORT_end_day = pasttime->tm_mday;
	REPORT_end_year = pasttime->tm_year;
	REPORT_end_jul = pasttime->tm_yday + 1;      /* Julian starts at 1 */

	intreport = fopen(ireportname, "w");
	extreport = fopen(ereportname, "w");
	
	/* This line is printed at the beginning of the report:
	 *	A) because it provides easy access to know which report
	 *	   this is
	 *	B) so it can be used to ensure that we have the right
	 *	   week's report by simply comparing the first line
	 *	   (i.e., with head -1)
	 *	C) and so that it can be used as the subject line of
	 *	   a mail message.
	 */
	fprintf(intreport,
		"%s Server Report, %02d/%02d/%02d - %02d/%02d/%02d\n\n",
		REPORT_server_name(),
		REPORT_start_month, REPORT_start_day, REPORT_start_year,
		REPORT_end_month, REPORT_end_day, REPORT_end_year);

	fprintf(intreport, "This page intentionally left blank.\n\n");
	fprintf(intreport,
		"(This permits AMS programs to begin printing this report on\n");
	fprintf(intreport, "a fresh page.)\n");

	/* A side-effect of getting the load graph is that we know when
	 * the system has been down.
	 */
	downtime = REPORT_graph_load(intreport);
	REPORT_report_acct(intreport, downtime);
	DISKS_report(intreport);

	fclose(intreport);
	fclose(extreport);
	
	intreport = fopen(ireportname, "r");
	copyreport = fopen(copyiname, "w");
	while (fgets(copyline, sizeof(copyline), intreport) != NULL)
		fputs(copyline, copyreport);
	fclose(intreport);
	fclose(copyreport);		

	extreport = fopen(ereportname, "r");
	copyreport = fopen(copyename, "w");
	while (fgets(copyline, sizeof(copyline), extreport) != NULL)
		fputs(copyline, copyreport);
	fclose(extreport);
	fclose(copyreport);		
}

void REPORT_init()
{
	char ireportname[64];
	char curweekstr[3];
	time_t curtime;
	FILE *checkfile;

	REPORT_init_time_line();

	curtime = time(NULL);
	strftime(curweekstr, sizeof(curweekstr), "%U", localtime(&curtime));
	REPORT_weeknum = (curweekstr[0] - 48) * 10 + (curweekstr[1] - 48);
	if (--REPORT_weeknum < 0)
		REPORT_weeknum = 52;
	sprintf(ireportname, "SYS:\\NOVSTAT\\IREPORT.%02d", REPORT_weeknum);

	checkfile = fopen(ireportname, "r");
	if (checkfile == NULL)
		REPORT_generate();
	else
	        fclose(checkfile);
}
