/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/load.c,v 1.5 1994/02/15 18:23:46 ms5u Exp $";
/* load.c
 *
 * This module contains all the necessary routines for computing, storing,
 * and retrieving the current load information.
 *
 * Revision History:
 *   $Log: load.c,v $
 * Revision 1.5  1994/02/15  18:23:46  ms5u
 * Change version to V1.1.
 *
 * Revision 1.4  1994/01/12  18:53:55  ms5u
 * Fix LOADFILE to be %03d for the extension.
 *
 * Revision 1.3  1993/11/04  22:15:48  ms5u
 * Check to see if LOAD_file is NULL before writing.
 *
 * Revision 1.2  1993/10/05  15:34:56  ms5u
 * Add copyright header.
 *
 * Revision 1.1  1993/09/13  19:35:59  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <time.h>
#include <process.h>
#include <nwmisc.h>
#include <errno.h>
#include "prototyp.h"

static int LOAD_base = 0;
static int LOAD_tickcount = 0, LOAD_lastticks = 0;
static FILE *LOAD_file;
static char LOAD_filename[64];

void LOAD_open()
{
	LOAD_file = fopen(LOAD_filename, "ab");
}

void LOAD_close()
{
	fclose(LOAD_file);
}

void LOAD_init()
{
	int retstat;
	
	LOAD_update_filename();	/* Automatically opens the new file */

	retstat = BeginThread(LOAD_idle_counter, NULL, 0, NULL);
	if (retstat == EFAILURE)
		perror("Error starting load thread");
}

void LOAD_update_filename()
{
	time_t curtime;
	int julianday;
	char julianstr[4];

	LOAD_close();
	curtime = time(NULL);
	strftime(julianstr, sizeof(julianstr), "%j", localtime(&curtime));
	julianday = (julianstr[0] - 48) * 100 +
		    (julianstr[1] - 48) * 10 +
		    (julianstr[2] - 48);
	sprintf(LOAD_filename, "SYS:\\NOVSTAT\\loadfile.%03d", julianday);
	LOAD_open();
}

void LOAD_idle_counter()
{
	int MyThreadID, retstat;
	
	MyThreadID = GetThreadID();
	retstat = SetPrty(2, 1, -1, MyThreadID);
	if (retstat == ESUCCESS)
		while (1) {
			LOAD_tickcount++;
			ThreadSwitch();
		}
	else
		perror("Error lowering load priority");
}

void LOAD_compute_period_load()
{
	if (LOAD_tickcount > LOAD_base)
		LOAD_base = LOAD_tickcount;
	LOAD_lastticks = LOAD_tickcount;
	LOAD_tickcount = 0;
}

int LOAD_get_load()
{
	return(100 - (LOAD_lastticks / (LOAD_base / 100)));
}

void LOAD_graph_console()
{
	static char bars[] = "########################################################################";

	int curload, barlen;
	time_t curtime;
	char timestring[6];
	
	curload = LOAD_get_load();
	curtime = time(NULL);
	strftime(timestring, sizeof(timestring), "%H:%M", localtime(&curtime));
	barlen = (curload * 65) / 100;
	bars[barlen] = 0;
	printf("%5s |%-65s| (%2d%%)\n", timestring, bars, curload);
	bars[barlen] = '#';
}

void LOAD_store()
{
	struct LOAD_rec curinfo;
	
	curinfo.curtime = time(NULL);
	curinfo.curload = LOAD_get_load();
	if (LOAD_file != NULL)
	{
		fwrite(&curinfo, sizeof(curinfo), 1, LOAD_file);
		fflush(LOAD_file);
	}
}

void LOAD_flush()
{
	LOAD_close();
	LOAD_open();
}
