/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/disks.c,v 1.4 1994/02/15 18:23:10 ms5u Exp $";
/* disks.c
 *
 * All we do here is print a small mini-status about the disks in a cute
 * bar chart.
 *
 * Revision History:
 *   $Log: disks.c,v $
 * Revision 1.4  1994/02/15  18:23:10  ms5u
 * Change version to V1.1.
 *
 * Revision 1.3  1993/11/23  16:21:05  ms5u
 * Update to correct the disk usage info.
 *
 * Revision 1.2  1993/10/05  15:34:50  ms5u
 * Add copyright header.
 *
 * Revision 1.1  1993/09/13  19:35:52  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <errno.h>
#include <nwdir.h>
#include "prototyp.h"

#define MAXDISKS 32

static char *TOPBOT = "                 +---------+---------+---------+---------+---------+\n";
static char *BREAKER = "                 |         |         |         |         |         |\n";

struct {
	char diskname[17];
	LONG total, avail;
	WORD sectorsper;
} diskstats[MAXDISKS];
int largestvol, mostMB;

void DISKS_load_info()
{
	int i, totalsize;
	VOLUME_INFO curdisk;

	mostMB = 0;
	for (i = 0; i < MAXDISKS; i++)
	{
		if (GetVolumeStatistics(0, i, sizeof(curdisk),
			&curdisk) != ESUCCESS)
			diskstats[i].diskname[0] = 0;
		else {
			strcpy(diskstats[i].diskname, curdisk.volumeName);
			diskstats[i].total = curdisk.totalBlocks;
			diskstats[i].avail = curdisk.availableBlocks;
			diskstats[i].avail += curdisk.purgableBlocks;
			diskstats[i].sectorsper = curdisk.sectorsPerBlock;
		}

		/* Some NetWare tests indicated a successful status even
		 * though the volumes about which we inquired didn't exist.
		 */
		if (diskstats[i].diskname[0] == ' ')
			diskstats[i].diskname[0] = 0;

		if (diskstats[i].diskname[0] != 0)
		{
			totalsize = diskstats[i].total;
			totalsize *= diskstats[i].sectorsper / 2;
			/* One NetWare sector is 512 bytes, dividing by
			 * 2 then 1024 gets us MB.
			 */
			totalsize /= 1024;

			if (totalsize > mostMB)
			{
				mostMB = totalsize;
				largestvol = i;
			}
		}
	}
}

void DISKS_report(FILE *rptfile)
{
	int i, spacecheck, curused, curtotal;
	int diskclosed;
	char tmpout[80];

	REPORT_page_hdr(rptfile, "Disk Usage Summary", 3);
	DISKS_load_info();

	fprintf(rptfile, "\n\n");
	fprintf(rptfile, TOPBOT);
	fprintf(rptfile, BREAKER);
	for (i = 0; i < MAXDISKS; i++)
		if (diskstats[i].diskname[0] != 0)
		{
			fprintf(rptfile, "%16s [", diskstats[i].diskname);
			curused = diskstats[i].total - diskstats[i].avail;
			curused *= diskstats[i].sectorsper / 2;
			curused /= 1024;
			curtotal = diskstats[i].total;
			curtotal *= diskstats[i].sectorsper / 2;
			curtotal /= 1024;
			
			diskclosed = 0;
			for (spacecheck = 2; spacecheck <= 100; spacecheck += 2)
			{
				if (curused >= (mostMB * spacecheck / 100))
					fprintf(rptfile, "*");
				else if (curtotal > (mostMB * spacecheck / 100))
					fprintf(rptfile, "-");
				else if (diskclosed == 0)
				{
					fprintf(rptfile, "]");
					diskclosed = 1;
				}
				else if (spacecheck % 100 == 0)
					fprintf(rptfile, "|");
				else
					fprintf(rptfile, " ");
			}
			fprintf(rptfile, "\n");
			
			sprintf(tmpout, "%17s|(%dMB total; %dMB free)",
				"", curtotal, curtotal - curused);
			fprintf(rptfile, tmpout);
			fprintf(rptfile, BREAKER + strlen(tmpout));
			fprintf(rptfile, BREAKER);
		}
	fprintf(rptfile, TOPBOT);
}
