/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/acctsum.c,v 1.8 1994/02/15 18:22:57 ms5u Exp $";
/* acctsum.c
 *
 * This module contains some of the routines necessary for scanning
 * the NET$ACCT.DAT file and providing some summary information about
 * the past week's usage.
 *
 * Revision History:
 *   $Log: acctsum.c,v $
 * Revision 1.8  1994/02/15  18:22:57  ms5u
 * Change version to V1.1.
 *
 * Revision 1.7  1993/11/04  22:19:55  ms5u
 * Fix incorrect fixes from previous version.
 *
 * Revision 1.6  1993/11/04  22:15:01  ms5u
 * Modify to remove NULL parameters.
 *
 * Revision 1.5  1993/10/07  19:17:03  ms5u
 * Limit string size for usernames and fullnames in report.
 *
 * Revision 1.4  1993/10/05  15:32:46  ms5u
 * Add copyright header.
 *
 * Revision 1.3  1993/10/05  15:24:11  ms5u
 * Whoops -- forgot to increment topcount when checking top ten lists.
 *
 * Revision 1.2  1993/09/21  21:34:17  ms5u
 * Added initialization of machuniq to zero.
 *
 * Revision 1.1  1993/09/13  19:35:46  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <string.h>
#include <io.h>
#include <nwbindry.h>
#include <nwenvrn.h>
#include <nwfile.h>
#include <nwmisc.h>
#include <process.h>
#include <errno.h>
#include "prototyp.h"
#include "acctrec.h"

/* setTTS is used to set the "Transactional" bit on/off on a given file.
 * This is used to move the NET$ACCT.DAT file around.
 */
void setTTS(char *pathname, int ison)
{
	BYTE extFA;
	
	GetExtendedFileAttributes(pathname, &extFA);
	if (ison == 0)
		extFA &= 0xef;
	else
		extFA |= 0x10;
	SetExtendedFileAttributes(pathname, extFA);
}

void expire_into_old(char *old_name)
{
	FILE *sys_acctfile, *old_acctfile, *new_acctfile;
	char inrec[128];
	WORD inreclen;
	int rectime, isold;

	sys_acctfile = fopen("SYS:\\SYSTEM\\NET$ACCT.DAT", "rb");
	old_acctfile = fopen(old_name, "wb");
	new_acctfile = fopen("SYS:\\SYSTEM\\NET$ACCT.NEW", "wb");
	
	if ((sys_acctfile != NULL) &&
	    (old_acctfile != NULL) &&
	    (new_acctfile != NULL))
		while (!feof(sys_acctfile))
		{
			fread(&inreclen, sizeof(inreclen), 1, sys_acctfile);
			fread(&inrec, IntSwap(inreclen), 1, sys_acctfile);
			isold = 1;
			if (inrec[4] > REPORT_end_year)
				isold = 0;
			if (inrec[4] = REPORT_end_year)
				if (inrec[5] > REPORT_end_month)
					isold = 0;
				else if (inrec[5] = REPORT_end_month)
					if (inrec[6] > REPORT_end_day)
						isold = 0;

			if (isold == 1)
			{
				fwrite(&inreclen, sizeof(inreclen), 1,
					old_acctfile);
				fwrite(&inrec, IntSwap(inreclen), 1,
					old_acctfile);
			} else {
				fwrite(&inreclen, sizeof(inreclen), 1,
					new_acctfile);
				fwrite(&inrec, IntSwap(inreclen), 1,
					new_acctfile);
			}
		}
	
	if (sys_acctfile != NULL)
		fclose(sys_acctfile);
	if (old_acctfile != NULL)
		fclose(old_acctfile);
	if (new_acctfile != NULL)
		fclose(new_acctfile);

	setTTS("SYS:\\SYSTEM\\NET$ACCT.DAT", 0);
	unlink("SYS:\\SYSTEM\\NET$ACCT.DAT");
	rename("SYS:\\SYSTEM\\NET$ACCT.NEW","SYS:\\SYSTEM\\NET$ACCT.DAT");
	setTTS("SYS:\\SYSTEM\\NET$ACCT.DAT", 1);
}

char *get_novell_username(LONG uid)
{
	static char returnname[48];	/* Max length of an obj name */
	WORD objectType;
	
	if (GetBinderyObjectName(uid, &returnname, &objectType) != ESUCCESS)
		sprintf(returnname, "#%08x", uid);

	return(&returnname);
}

char *get_novell_fullname(LONG uid)
{
	static char fullname[130];	/* Max length of a property value */
	BYTE more, flags;

	if (ReadPropertyValue(get_novell_username(uid), OT_USER,
		"IDENTIFICATION", 1, &fullname, &more, &flags) == ESUCCESS)
		return(&fullname);
	else
		return("");
}

char *get_net_name(LONG netaddr)
{
	struct namestruct {
		LONG netaddr;
		char *netname;
		struct namestruct *next;
	};
	
	static struct namestruct *netnames = NULL;
	struct namestruct *tmp;
	static char readbuff[80], namebuff[80];
	LONG addrbuff;
	char *spaceloc;
	FILE *netdefs;
	
	if ((netnames == NULL) &&
	    ((netdefs = fopen("SYS:\\NOVSTAT\\NETNAMES.DAT","r")) != NULL))
	{
		while (fgets(readbuff, sizeof(readbuff) - 1, netdefs) != NULL)
		{
			if (readbuff[strlen(readbuff) - 1] == '\n')
				readbuff[strlen(readbuff) - 1] = 0;
			if (sscanf(readbuff, "%x %s", &addrbuff, &namebuff) == 2)
			{
				spaceloc = strchr(readbuff, ' ');
				while (*spaceloc == ' ')
					spaceloc++;
				if (netnames == NULL)
					tmp = netnames = (struct namestruct *)malloc(sizeof(struct namestruct));
				else
					tmp = tmp->next = (struct namestruct *)malloc(sizeof(struct namestruct));
				tmp->next = NULL;
				tmp->netaddr = addrbuff;
				tmp->netname = (char *)malloc(strlen(spaceloc) + 1);
				strcpy(tmp->netname, spaceloc);
			}
		}
		fclose(netdefs);
	}

	tmp = netnames;
	while ((tmp != NULL) && (tmp->netaddr != netaddr))
		tmp = tmp->next;
	if (tmp == NULL)
		return("");
	else
		return(tmp->netname);
}

char *cvt_to_timestring(LONG secs)
{
	static char timeanswer[20];
	LONG days, hours, minutes;
	
	secs /= 60;	/* to minutes */
	minutes = secs % 60;
	secs /= 60;	/* to hours */
	hours = secs % 24;
	secs /= 24;	/* to days */
	days = secs;
	
	sprintf(timeanswer, "%2d %02d:%02d", days, hours, minutes);
	return(&timeanswer);
}

char *server_info_string()
{
	static char info[80];
	FILE_SERV_INFO serverInfo;
	
	GetServerInformation(sizeof(serverInfo), &serverInfo);
	sprintf(info, "NetWare %d.%d (CLIB %d.%d Rev. %d)",
		serverInfo.netwareVersion,
		serverInfo.netwareSubVersion,
		serverInfo.CLibMajorVersion,
		serverInfo.CLibMinorVersion,
		serverInfo.CLibRevision);
	return(&info);
}

int server_total_users()
{
	char objectName[48], objectHasProperties, objectFlag, objectSecurity;
	WORD objectType;
	int usercount;
	long objectID;
	
	objectID = -1;
	usercount = 0;
	while (ScanBinderyObject("*", OT_USER, &objectID,
				 objectName, &objectType, &objectHasProperties,
				 &objectFlag, &objectSecurity) == ESUCCESS)
		usercount++;
	return(usercount);
}

void REPORT_user_summary(FILE *rptfile, char *acctdat)
{
	struct ACCT_rec newrec;
	int rectyp;
	char username[48];              /* Novell example uses 48 here */
	struct sparsecounter *logins = NULL, *nets = NULL, *tmp;
	struct sparse_m_counter *machines = NULL, *tmp_m;
	int logintot, loginuniq, nettot, netuniq, machuniq;
	int topcount;

	ACCT_initrec(newrec);
	
	if (ACCT_openfile(acctdat))
	{
		fprintf(rptfile, "Unable to produce login report summary.\n");
		return;
	}

#define CURLOGNOTE newrec.variant.NoteInfo.Comment->LoginNote

	while ((rectyp = ACCT_readnextrec(&newrec)) != 0)
	{
		if ((rectyp == NOTERECORD) &&
		    (newrec.variant.NoteInfo.CommentType == LOGOUTNOTE))
		{
			SPARSE_increment(&logins, newrec.ClientID,
				TRACK_logtime(CURLOGNOTE.Net,
					CURLOGNOTE.NodeHigh,
					CURLOGNOTE.NodeLow,
					newrec.TimeStamp));
			SPARSE_increment(&nets,
				CURLOGNOTE.Net,
				TRACK_logtime(CURLOGNOTE.Net,
					CURLOGNOTE.NodeHigh,
					CURLOGNOTE.NodeLow,
					newrec.TimeStamp));
			SPARSE_m_increment(&machines,
				CURLOGNOTE.Net,
				CURLOGNOTE.NodeHigh,
				CURLOGNOTE.NodeLow,
				TRACK_logtime(CURLOGNOTE.Net,
					CURLOGNOTE.NodeHigh,
					CURLOGNOTE.NodeLow,
					newrec.TimeStamp));
		} else if ((rectyp == NOTERECORD) &&
			   (newrec.variant.NoteInfo.CommentType == LOGINNOTE))
			TRACK_addinfo(CURLOGNOTE.Net, CURLOGNOTE.NodeHigh,
				CURLOGNOTE.NodeLow,
				newrec.TimeStamp);
		
		ACCT_disposerec(&newrec);

		ThreadSwitch();      /* We do this to be nice to the server */
	}

	TRACK_cleanup();

	logintot = 0;
	loginuniq = 0;
	SPARSE_sort(logins);

	tmp = logins;
	while (tmp != NULL)
	{
		logintot += tmp->count;
		loginuniq++;
		tmp = tmp->next;
	}
	
	ThreadSwitch();

	nettot = netuniq = machuniq = 0;
	SPARSE_sort(nets);

	tmp = nets;
	while (tmp != NULL)
	{
		nettot += tmp->count;
		netuniq++;
		tmp = tmp->next;
	}
	
	ThreadSwitch();

	tmp_m = machines;
	while (tmp_m != NULL)
	{
		machuniq++;
		tmp_m = tmp_m->next;
	}

	ThreadSwitch();

	fprintf(rptfile, "\n%d unique users logged in %d times",
		loginuniq, logintot);
	fprintf(rptfile, " from %d distinct networks.\n",
		netuniq);
	fprintf(rptfile, "There are a total of %d users on this server.\n",
		server_total_users());
	fprintf(rptfile, "They connected from %d distinct computers.\n\n",
		machuniq);

	tmp = logins;
	topcount = 0;
	fprintf(rptfile, "Top ten users of this NetWare server:\n\n");
	fprintf(rptfile, "         User                 Name              Logins   TimeOnline\n");
	fprintf(rptfile, "     ------------  --------------------------  --------  ----------\n");

	while ((tmp != NULL) && (topcount < 10))
	{
		fprintf(rptfile, "     %-12.12s  %-26.26s  %8d   %8s\n",
			get_novell_username(tmp->value),
			get_novell_fullname(tmp->value),
			tmp->count,
			cvt_to_timestring(tmp->total));
		tmp = tmp->next;
		topcount++;
	}	

	ThreadSwitch();

	tmp = nets;
	topcount = 0;

	fprintf(rptfile, "\n\n\nTop ten network sites for this NetWare server:\n\n");
	fprintf(rptfile, "    Address    Configuration Name    Logins   TimeOnline\n");
	fprintf(rptfile, "    --------  --------------------  --------  ----------\n");
	
	while ((tmp != NULL) && (topcount < 10))
	{
		fprintf(rptfile, "    %08x  %-20.20s  %8d   %8s\n",
			tmp->value, get_net_name(tmp->value), tmp->count,
			cvt_to_timestring(tmp->total));
		tmp = tmp->next;
		topcount++;
	}

	ThreadSwitch();

	SPARSE_cleanup(logins);
	SPARSE_cleanup(nets);

	ACCT_closefile();
}

void REPORT_report_acct(FILE *rptfile, int *downtime)
{
	char old_name[40];
	FILE *testexists;

	/* The first thing that we have to do is copy off the old data
	 * from the NET$ACCT.DAT into a storage file and the new data
	 * into a replacement NET$ACCT.DAT file and swap the two.
	 */
	
	sprintf(old_name, "SYS:\\NOVSTAT\\NET$ACCT.%02d", REPORT_weeknum);

	testexists = fopen(old_name, "rb");
	if (testexists == NULL)
		expire_into_old(old_name);
	else
		fclose(testexists);

	REPORT_page_hdr(rptfile, "Server Use Detail", 2);

	fprintf(rptfile, "\n\nServer is running %s\n\n",
		server_info_string());

	DOWNRPT_printout_downtime(rptfile, downtime);
	free(downtime);

	REPORT_user_summary(rptfile, old_name);
}
