/*
 *  NovStat V1.1
 *
 *  Copyright (C) 1993 by Carnegie Mellon University
 *
 *  This software is freely available for any purpose other than commercial
 *  gain.  You may distribute this code provided this copyright remains
 *  intact.  Any modifications made to this code must be noted as not part
 *  of the original NovStat V1.1 code.
 *
 */

static char rcsid[] = "$Header: /afs/andrew.cmu.edu/system/src/local/novstat/003/src/RCS/acctread.c,v 1.3 1994/02/15 18:22:19 ms5u Exp $";
/*
 * readrec.c
 *
 * Created 4/19/93 by Marc Shannon
 *
 * This module is responsible for reading the next record from the
 * Novell accounting data file and returning it to the caller.  It's
 * up to the caller to actually do something reasonable with the data.
 *
 * Revision History:
 *   $Log: acctread.c,v $
 * Revision 1.3  1994/02/15  18:22:19  ms5u
 * Change version to V1.1.
 *
 * Revision 1.2  1993/10/05  15:32:16  ms5u
 * Add copyright header.
 *
 * Revision 1.1  1993/09/13  18:57:51  ms5u
 * Initial revision
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <nwmisc.h>
#include "acctrec.h"

static FILE *acctfile = NULL;

int ACCT_openfile(char *filename)
{
    if (acctfile != NULL)
        return (-1);            /* If file is already open, fail on
                                 * this call */

    acctfile = fopen(filename, "rb");
    return (acctfile == NULL);
}

/* closefile() will close the acctfile if it has been opened.  This returns
 * the result of fclose().
 */
void ACCT_closefile()
{
    fclose(acctfile);
    acctfile = NULL;
}

/* initrec() simply zeroes out the record for safety
 */
void ACCT_initrec(struct ACCT_rec tobe0)
{
    memset(&tobe0, 0, sizeof(tobe0));
}

/* readnextrec() will read in the next record from acctfile.  The function
 * returns a 0 if there was an error reading the record or the value of
 * readrec->RecordType as a convenience to the caller.
 */ 
int ACCT_readnextrec(struct ACCT_rec *readrec)
{
    if (!fread(&readrec->Length, sizeof(readrec->Length), 1, acctfile))
    return (0);

    readrec->Length = IntSwap(readrec->Length);

    /* We can safely read into the common part of the structure before we
     * have to decide how to treat this record.
     */
    if (!fread(&readrec->ServerID, COMMONLENGTH - sizeof(readrec->Length),
        1, acctfile))
        return (0);

    readrec->ClientID = LongSwap(readrec->ClientID);

    switch (readrec->RecordType) {
        case CHARGERECORD :
            if (!(readrec->variant.ChargeInfo.Comment =
                malloc(readrec->Length - CHARGEBASELEN +
                sizeof(readrec->Length))))
                return (0);
            /* We added sizeof(Length) to the malloc() call since the Length
             * field does not include the Length bytes themselves.
             */
                          
            /* Now, read in the remainder of the base structure first, then
             * read in the Comment data into its newly malloc'ed space.
             */
            if (!fread(&readrec->variant.ChargeInfo,
                CHARGEBASELEN - COMMONLENGTH, 1, acctfile))
                return (0);

	    readrec->variant.ChargeInfo.Amount =
		    LongSwap(readrec->variant.ChargeInfo.Amount);
	    readrec->variant.ChargeInfo.CommentType =
		    IntSwap(readrec->variant.ChargeInfo.CommentType);

            if (!fread(readrec->variant.ChargeInfo.Comment,
                readrec->Length - CHARGEBASELEN +
                sizeof(readrec->Length), 1, acctfile))
                return (0);

#define CTNOTE readrec->variant.ChargeInfo.Comment->ConnectTimeNote
#define DSNOTE readrec->variant.ChargeInfo.Comment->DiskStorageNote

            switch (readrec->variant.ChargeInfo.CommentType)
            {
            case CONNECTTIMENOTE:
		CTNOTE.ConnectTime = LongSwap(CTNOTE.ConnectTime);
		CTNOTE.RequestCount = LongSwap(CTNOTE.RequestCount);
		CTNOTE.BytesReadHigh = IntSwap(CTNOTE.BytesReadHigh);
		CTNOTE.BytesRead2 = IntSwap(CTNOTE.BytesRead2);
		CTNOTE.BytesReadLow = IntSwap(CTNOTE.BytesReadLow);
		CTNOTE.BytesWrittenHigh = IntSwap(CTNOTE.BytesWrittenHigh);
		CTNOTE.BytesWritten2 = IntSwap(CTNOTE.BytesWritten2);
		CTNOTE.BytesWrittenLow = IntSwap(CTNOTE.BytesWrittenLow);
                break;
            case DISKSTORAGENOTE:
		DSNOTE.BlocksOwned = LongSwap(DSNOTE.BlocksOwned);
		DSNOTE.NumberOfHalfHours = LongSwap(DSNOTE.NumberOfHalfHours);
                break;
            default:
                printf("Unknown comment type %d\n",
                readrec->variant.ChargeInfo.CommentType);
                break;
            }
            break;
            
        case NOTERECORD :
            if (!(readrec->variant.NoteInfo.Comment =
                malloc(readrec->Length - NOTEBASELEN +
                sizeof(readrec->Length))))
                return (0);
            /* Now do the same as before, but with different constant
             * values.
             */
            if (!fread(&readrec->variant.NoteInfo,
                NOTEBASELEN - COMMONLENGTH, 1, acctfile))
                return (0);

	    readrec->variant.NoteInfo.CommentType =
		IntSwap(readrec->variant.NoteInfo.CommentType);

            if (!fread(readrec->variant.NoteInfo.Comment,
                readrec->Length - NOTEBASELEN +
                sizeof(readrec->Length), 1, acctfile))
                return (0);


#define LOGNOTE readrec->variant.NoteInfo.Comment->LoginNote
/* Note that LoginNote, LogoutNote, and AccountLockedNote all have the
   same structure.  We'll use this to our advantage. */

            switch (readrec->variant.NoteInfo.CommentType)
            {
            case LOGINNOTE:
            case LOGOUTNOTE:
            case ACCOUNTLOCKEDNOTE:
		LOGNOTE.Net = LongSwap(LOGNOTE.Net);
		LOGNOTE.NodeHigh = LongSwap(LOGNOTE.NodeHigh);
		LOGNOTE.NodeLow = IntSwap(LOGNOTE.NodeLow);
                break;
            case SERVERTIMEMODIFIEDNOTE:
                /* This one is okay */
                break;
            default:
                printf("Unknown comment note %d\n",
                readrec->variant.NoteInfo.CommentType);
                break;
            }

            break;

        default :
            return (0);        /* Unknown record type */
    };

    return (readrec->RecordType);
}
 
/* disposerec() is responsible for deleting the Comment data malloc'ed
 * when the record was read in.
 */
void ACCT_disposerec(struct ACCT_rec *rec)
{
    switch (rec->RecordType) {
        case CHARGERECORD :
            if (rec->variant.ChargeInfo.Comment)
                free(rec->variant.ChargeInfo.Comment);
            break;
        case NOTERECORD :
            if (rec->variant.NoteInfo.Comment)
                free(rec->variant.NoteInfo.Comment);
            break;
    }
}
