/* $Id: rcgid.c,v 1.3 1995/09/13 13:19:02 sudduth Exp $ */
/*
   rcgid.c

   rcgid is a UNIX daemon which implements the "remote CGI" protocol
   over tcp/ip.  it is designed to be run either standalone (-s) or
   via inetd.

   brad@american.com 7/95
*/

#include "rcgid.h"

char *server_root;
int server_port;
int debug;
int server_socket;
int server_alive;
struct sockaddr_in sa_server;
int pending_death;

int localhost_only;
char *restrict_to;
int raw_data = 1;

extern char *optarg;
extern int optind;
extern char *ip_ntoa();

void
usage(bin)
char *bin;
{
    fprintf(stderr,
	    "Usage: %s [-d directory] { -r pattern | -L } [-l] \n",
	    bin);
    fprintf(stderr,"-d directory : specify an alternate directory\n");
    fprintf(stderr,"-L local host (loopback) only\n");
    fprintf(stderr,"-r restict host to 'pattern' (*.xyz.com, a.b.com)\n");
    fprintf(stderr,"-l log to local file\n");
    fprintf(stderr,"-n allow script to emit headers\n");
    fprintf(stderr,"-x turn on debug output\n");
    exit(1);
}

void
set_server_root(arg)
char *arg;
{
    server_root = strdup(arg);
}

void
set_server_port(arg)
char *arg;
{
    server_port = atoi(arg);
}

void
set_debug(arg)
char *arg;
{
    debug = atoi(arg);
}

void
final_death()
{
    debugf("final_death()\n");
    exit(0);
}

void
graceful_death()
{
    debugf("graceful_death()\n");
}

void
set_signals()
{
/* it is better to ignore SIGPIPE because there is a possibility that the
   script may already be done and close the stdin
*/
	signal(SIGPIPE, SIG_IGN);
}

char *
find_fqdn(p)
struct hostent *p;
{
    int x;

    if (strchr(p->h_name, '.') == 0) {
	char domain[MAX_FNAME];
	static char name[MAX_FNAME];

	if (p->h_aliases)
	  for (x = 0; p->h_aliases[x]; ++x) {
	      if ((strchr(p->h_aliases[x], '.')) &&
		  (!strncmp(p->h_aliases[x], p->h_name, strlen(p->h_name))))
		return p->h_aliases[x];
	  }
	
	getdomainname(domain, sizeof(domain));
	sprintf(name, "%s.%s", p->h_name, domain);
	return name;
    }

    return p->h_name;
}

void
map_sockaddr_to_name(client, who, wholen)
struct sockaddr_in *client;
char *who;
int wholen;
{
    struct hostent *h;

    /* map host addr to host name or dotted */
    h = gethostbyaddr(&client->sin_addr, sizeof(struct in_addr), AF_INET);
    if (h) {
	strncpy(who, find_fqdn(h), wholen);
    } else
	sprintf(who, "%s", ip_ntoa(client->sin_addr.s_addr));
}

int
match_suffix(str, pattern)
char *str;
char *pattern;
{
    int ls = strlen(str);
    int lp = strlen(pattern);
    int offset;

    if (lp > ls)
	return 0;

    offset = ls - lp;

    if (memcmp(str+offset, pattern, lp) == 0)
	return 1;

    return 0;
}

int
filter_source(client, who)
struct sockaddr_in *client;
char *who;
{
    if (localhost_only && client->sin_addr.s_addr != INADDR_LOOPBACK)
	return 1;

    if (restrict_to) {
	if (restrict_to[0] == '*') {
	    if (match_suffix(who, restrict_to+1))
		return 0;
	} else
	    if (strcmp(who, restrict_to) == 0)
		return 0;

	return 1;
    }

    return 0;
}

void
standalone_main()
{
    int client_socket, clen, one=1;
    struct sockaddr_in sa_client;
    char who[MAX_FNAME];

    /* open tcp socket */
    if ((server_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
        fprintf(stderr,"rcgid: could not get tcp socket\n");
        perror("socket");
        final_death();
    }

    if ((setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR,
		    (char *)&one, sizeof(one))) == -1)
    {
        fprintf(stderr, "rcgid: could not set tcp socket option\n");
        perror("setsockopt");
        final_death();
    }

    /* bind it to an addres, port */
    memset((char *)&sa_server, 0, sizeof(sa_server));
    sa_server.sin_family = AF_INET;
    sa_server.sin_addr.s_addr = htonl(INADDR_ANY);
    sa_server.sin_port = htons((short)server_port);

    if (bind(server_socket, (struct sockaddr *)&sa_server,
	     sizeof(sa_server)) == -1)
    {
        fprintf(stderr, "rcgid: could not bind tcp socket to port %d\n", 
		server_port);
        perror("bind");
        final_death();
    }

    debugf("listening on port %d\n", ntohs(sa_server.sin_port));
    listen(server_socket, 5);

    set_signals();

    if (log_pid())
        final_death();

    server_alive = 1;

    while (server_alive) {
      retry:
	debugf("top of loop, server_alive = %d\n", server_alive);
        clen = sizeof(sa_client);
	client_socket = accept(server_socket, 
			       (struct sockaddr *)&sa_client, &clen);

        if (client_socket == -1) {
            if (errno == EINTR && server_alive) {
		debugf("retry\n");
                goto retry;
	    }

            log_error("socket error: accept failed");
	    debugf("accept failed, errno %d, server_alive %d\n", 
		   errno, server_alive);

	    /* if we're not being shut down, schedule death */
	    if (server_alive)
		graceful_death();
        } else {
	    debugf("connection!\n");
	    map_sockaddr_to_name(&sa_client, who, sizeof(who));

	    if (!filter_source(&sa_client, who))
		rcgi(client_socket, &sa_client, who);
	    else {
		char msg[MAX_FNAME];

		sprintf(msg, "connection from %s dropped", who);
		log_error(msg);
	    }

	    close(client_socket);
	}

	/* */
	if (pending_death)
	    graceful_death();
    }
}

main(argc, argv)
int argc;
char *argv[];
{
    int c;

    server_root = ".";
    server_port = 8001;

    while ((c = getopt(argc,argv,"d:p:r:x:Llnh")) != -1) {
	char *lp;
	int level;
        switch(c) {
	  case 'd':
	    set_server_root(optarg);
            break;

	  case 'p':
	    set_server_port(optarg);
	    break;

	  case 'r':
	    restrict_to = strdup(optarg);
	    break;

	  case 'x':
	    set_debug(optarg);
	    break;

	  case 'L':
	    localhost_only = 1;
	    break;

	  case 'l':
	    set_local_log();
	    break;
	    
	  case 'n':
	    raw_data = 0;
	    break;

          case '?':
          case 'h':
            usage(argv[0]);
        }
    }

    chdir(server_root);
    standalone_main();

    exit(0);
}

/* ------------------------------------------------------------------ */

int rcgi_state;
char *cmd_line;
char *env[MAX_ENV];
int env_count;

int child_pid;
int child_in_fd;
int child_out_fd;

int in_port;

char *cmds[] = {
    "env",
    "cmd",
    "input",
    "port",
    "end",
    "vers",
    0
};

int
add_env(buf)
char *buf;
{
    debugf("add_env(%s)\n", buf);

    if (env_count >= MAX_ENV)
	return -1;

    env[env_count++] = strdup(buf);
    env[env_count] = 0;

    return 0;
}

void
free_env()
{
    int i;

    debugf("free_env()\n");
    for (i = 0; i < env_count; i++)
	free(env[i]);

    env_count = 0;
}

int
rcgi(skt, sa_client, who)
int skt;
struct sockaddr_in *sa_client;
char *who;
{
    FILE *in, *out;
    int more, i;
    char buf[MAX_LINE_LEN], *p;

    debugf("rcgi()\n");

    if (!(in = fdopen(skt, "r")))
	return -1;

    if (!(out = fdopen(skt, "w"))) {
	fclose(in);
	return -1;
    }

    rcgi_state = 0;
    more = 1;

    while (more) {
	int len;

	if (!fgets(buf, sizeof(buf)-1, in))
	    break;

	len = strlen(buf);

	if (len >= 2 && buf[len-2] == '\r' && buf[len-1] == '\n') {
	    buf[len-2] = 0;
	    len -= 2;
	} else
	    if (len >=1 && buf[len-1] == '\n') {
		buf[len-1] = 0;
		len--;
	    }

	switch(rcgi_state) {
	  case 0:
	    for (p = buf; *p && *p != ' '; p++)
		;
	    if (*p)
		*p++ = 0;
	    for (i = 0; cmds[i]; i++) {
		if (strcasecmp(cmds[i], buf) == 0)
		    break;
	    }
	    debugf("cmd #%d\n", i+1);
	    switch (i+1) {
	      case 1: /* env */
		rcgi_state = 1;
		fprintf(out, "100 Enter env.  End with single '.' on a line\r\n");
		fflush(out);
		break;

	      case 2: /* cmd */
		if (*p)
		    cmd_line = strdup(p);

		fprintf(out, "101 Cmd accepted '%s'\r\n", cmd_line);
		fflush(out);
		break;

	      case 3: /* input */
		rcgi_state = 2;
		break;

	      case 4: /* port */
		rcgi_state = 3;
		break;

	      case 5: /* end */
		more = 0;
		if (child_pid == 0) {
		    start_cmd(cmd_line, env, who, out);
		}

		send_to_proc(0);
		break;
	      case 6: /* vers */
		break;

	      default:
		fprintf(out, "500 Bad command '%s'\r\n", buf);
		fflush(out);
		more = 0;
	    }
	    break;

	  case 1: /* env */
	    if (len == 1 && buf[0] == '.') {
		fprintf(out, "100 Env accepted\r\n");
		fflush(out);
		rcgi_state = 0;
		break;
	    }

	    if (buf[0] == 0)
		break;

	    if (add_env(buf)) {
		fprintf(out, "501 Environment full '%s'\r\n", buf);
		fflush(out);
		more = 0;
	    }
	    break;

	  case 2: /* input */
	    if (child_pid == 0)
		start_cmd(cmd_line, env, who, out);

	    if (len > 1 && buf[0] == '.' && buf[1] == '.')
		;
	    else
		if (len == 1 && buf[0] == '.')
		    rcgi_state = 0;

	    send_to_proc(buf);
	    break;
	  case 3: /* port */
	    in_port = atoi(p);
	    break;
	}
    }

    debugf("end of loop\n");

    if (child_pid) {
	/* pass output back up to client */
	unsigned long osize = 0;

	while (1) {
	    char buf[4096];
	    int r;

	    r = read(child_out_fd, buf, sizeof(buf));
	    if (r <= 0)
		break;

	    /* first bit of output - report success */
	    if (osize == 0 && r > 0) {
		fprintf(out, "200 Cmd started '%s'\r\n%s",
			cmd_line, raw_data ? "\r\n" : "");
		fflush(out);
	    }

	    osize += r;
	    fwrite(buf, r, 1, out);
	}

	/* if no output - report failure */
	if (osize == 0) {
	    fprintf(out, "503 exec of '%s' failed\n", cmd_line);
	    fflush(out);
	}

	close(child_out_fd);
	fflush(out);

	wait_for_child(cmd_line, who);
    }

    child_pid = 0;
    child_out_fd = 0;
    child_in_fd = 0;

    fclose(in);
    fclose(out);

    if (cmd_line) {
	free(cmd_line);
	cmd_line = 0;
    }

    free_env();

    close(skt);
    return 0;
}

int
start_cmd(cmd_line, env, who, out)
char *cmd_line;
char **env;
char *who;
FILE *out;
{
    debugf("start_cmd()\n");

    /* exec command */
    if (exec_cmd(cmd_line, env, who)) {
	fprintf(out, "502 Cmd failed\r\n");
	fflush(out);
	return -1;
    }

    return 0;
}


int
exec_cmd(cmd_line, env, who)
char *cmd_line;
char **env;
char *who;
{
    int pin[2], pout[2];
    char **argv, *argv0;

    debugf("exec_cmd()\n");

    /* clean up path to cmd & validate it */
    if (clean_up_path(cmd_line)) {
	char msg[MAX_FNAME];

	sprintf(msg, "exec of %s failed; bad path", cmd_line);
	log_error(msg);
	return -1;
    }

    if (pipe(pin) < 0) {
        log_error("could not create IPC pipe");
	return -1;
    }

    if (pipe(pout) < 0) {
        log_error("could not create IPC pipe");
	return -1;
    }

    if ((child_pid = fork()) < 0) {
        log_error("could not fork new process");
	return -1;
    }

    if (!child_pid) {
	int fd;
	char *cmd;
	char msg[MAX_FNAME];

	/* child */
	debugf("child\n");

	/* stdin */
	close(pin[1]);
	dup2(pin[0], 0);
	close(pin[0]);

	/* stdout */
	close(pout[0]);
	dup2(pout[1], 1);
	dup2(pout[1], 2);
	close(pout[1]);

	for (fd = 3; fd < 20; fd++)
	     close(fd);

	/* create argv vector */
	argv = create_argv(cmd_line);

	cmd = strdup(argv[0]);

	/* find root of cmd */
	if ((argv0 = strrchr(argv[0], '/')) != NULL)
	    argv[0] = argv0 + 1;

#if 0
	log_error(cmd);
	dumpv("argv", argv);
	dumpv("env", env);
#endif

	if (execve(cmd, argv, env) == -1) {
	    extern char *sys_errlist[];

	    sprintf(msg, "command '%s' from %s failed, %s",
		    cmd, who, sys_errlist[errno]);
	    log_error(msg);

	    exit(errno);
        }
	/* you can never get here */
    } else {
	/* parent */
	debugf("parent\n");

	close(pin[0]);
	close(pout[1]);

	child_in_fd = pin[1];
	child_out_fd = pout[0];
    }

    return 0;
}

int
wait_for_child(cmd_line, who)
char *cmd_line;
char *who;
{
    int status;
    char msg[MAX_FNAME];
    char fait[MAX_FNAME];

    waitpid(child_pid, &status, 0);

    sprintf(msg, "command '%s' from %s ", cmd_line, who);

    if (WTERMSIG(status)) {
	sprintf(fait, "died with signal %d", WTERMSIG(status));
    } else {
	sprintf(fait, "%s, status %d",
		status == 0 ? "ok" : "failed",
		WEXITSTATUS(status));
    }

    strcat(msg, fait);
    log_error(msg);

	return 0;
    }

int
send_to_proc(buf)
char *buf;
{
    if (buf == 0)
	/* close child's input stream */
	close(child_in_fd);
    else {
	write(child_in_fd, buf, strlen(buf));
    }
}

