/* ipdev.c: Internet protocol pseudo device driver */
#include "param.h"
#include "types.h"
#include "in.h"
#include "ip_var.h"
#include "inio.h"
#include "stream.h"
#include "errno.h"
#include "proc.h"
#include "inode.h"
#include "dir.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "signal.h"
#include "user.h"


#define	bclear(bp)	((bp)->rptr = (bp)->wptr = (bp)->base,\
				bp->type = M_IOCACK)

extern	putq(), srv(), nullsys();
static	ipqopen(), ipqclose(), ipdnsrv();

struct	qinit	ipqinit[] = {
/*up*/	{ putq, srv, ipqopen, ipqclose, 2 * 1520, 20 },
/*dn*/	{ putq, ipdnsrv, nullsys, nullsys, 2 * 1520, 20 }
};

static	struct	route	rte_tab[NROUTE];	/* gate ways */
static	struct	ipdev	ipdev[NIPDEV];
extern	struct	ipif	ipif[NIPLD];


static
ipqopen(dev, q)		/* connect to ipdev structure */
	dev_t dev;
	register struct queue *q;
{
	register unit;

	if ((unit = minor(dev)) < 0 || unit > NIPDEV)
		return ENXIO;
	if (ipdev[unit].flag & IPOPEN)
		return ENXIO;
	ipdev[unit].queue = q;
	ipdev[unit].ifsp = NULL;
	q->ptr = (char *)&ipdev[unit];
	OTHERQ(q)->ptr = (char *)&ipdev[unit];
	return 0;
}


ipopen(dev, flag)		/* open device only once */
	dev_t dev;
{
	register unit;
	register struct ipdev *dp;

	if ((unit = minor(dev)) < 0 || unit > NIPDEV) {
		u.u_error = ENXIO;
		return;
	}
	if ((dp = &ipdev[unit])->flag & IPOPEN) {
		u.u_error = ENXIO;
		return;
	}
	dp->flag |= IPOPEN;
}

static
ipqclose(dev, q)	/* shut down a ip device */
	dev_t dev;
	register struct queue *q;
{
	register unit;

	if ((unit = minor(dev)) < 0 || unit > NIPDEV)
		return ENXIO;
	ipdev[unit].queue = NULL;
	return 0;
}

ipclose(dev, flag)		/* close the device */
	dev_t dev;
{
	register unit;

	if ((unit = minor(dev)) < 0 || unit > NIPDEV)
		return;
	ipdev[unit].flag &= ~IPOPEN;
}

/* called from interface line disciplines */

ip_upqueue(proto, bp)	/* pass off a internet frame */
	u_short proto;
	register struct block *bp;
{
	register struct iphdr *p;
	register struct queue *q;

	if (proto < 0 || proto > NIPDEV)
		goto bad;
	if ((q = ipdev[proto].queue) == NULL)
		goto bad;
	(*q->qinfo->putp)(q, bp);
	return;
bad:
	freeb(bp);
}

/*
 * Because of the way the protocols on top of ip work, packets can
 * showup in pieces.  Headers often are in blocks in front of the data
 * block.  The last data block is delimited and rest are not.  There
 * is a pointer in the ipdev structure that is used to point to the
 * currently active interface for this queue.  This is reset on open and
 * after each delimiter.  The selection of interface occures when the
 * pointer is NULL (between packets) and a block appears (the first block
 * of a packet).
 */

static
ipdnsrv(q)		/* find an interface */
	register struct queue *q;
{
	register struct ipif *ifp;
	struct ipif *find_if();
	register struct block *bp;
	register in_addr inaddr;
	register struct iphdr *p;
	struct ipdev *dp;
	static u_short id = 0;

	dp = (struct ipdev *)q->ptr;
	while (q->first) {
		bp = getq(q);
		switch (bp->type) {
		case M_DATA: {
			int delim = 0;

			delim = bp->class & S_DELIM;
			if (dp->ifsp == (struct ipif *)1) {
				freeb(bp);
				if (delim)
					dp->ifsp = NULL;
				continue;
			}
			if (dp->ifsp != NULL) {
				ipld_output(dp->ifsp, bp, dp->gate);
				if (delim)
					dp->ifsp = NULL;
				continue;
			}
			p = (struct iphdr *)bp->rptr;
			p->ver_ihl |= 0x40;	/* version */
			p->tos = 0;
			p->id = id++;
			p->fragnflag = 0;
			p->ttl = 100;
			p->chksum = 0;
			dp->gate = rte_xlate(p->dest);
			if ((ifp = find_if(dp->gate)) == NULL) {
				freeb(bp);
				if (!delim)
					dp->ifsp = (struct ipif *)1; /* ditch */
				continue;
			}
			if (p->src == INADDR_ANY)
				p->src = ifp->thishost;
			p->chksum = cksum(p, 20);
			ipld_output(ifp, bp, dp->gate);
			if (!delim)
				dp->ifsp = ifp;
			continue;
		}
		case M_IOCTL:
			ipdev_ioctl(q, bp);
			continue;
		default:
			freeb(bp);
		}
	}
}



/* TBD: When table is full pick one to over write? */
rte_install(route)	/* insert the gateway for a dest in rte_tab */
	struct route route;
{
	register struct route *p, *ap = NULL;
	register dest;

	dest = route.dst;
	for (p = rte_tab; p < &rte_tab[NROUTE]; p++) {
		if (p->dst == dest) {
			p->gate = route.gate;
			return;
		}
		if (ap == NULL && p->dst == NULL)
			ap = p;
	}
	ap->dst = dest;
	ap->gate = route.gate;
}

rte_remove(route)	/* remove a route */
	struct route route;
{
	register struct route *p;
	register dest, gate;

	dest = route.dst;
	gate = route.gate;
	for (p = rte_tab; p < &rte_tab[NROUTE]; p++)
		if (p->dst == dest)
			p->dst = NULL;
}

rte_xlate(dest)		/* find gateway in route table */
	register in_addr dest;
{
	register struct route *p;

	for (p = rte_tab; p < &rte_tab[NROUTE]; p++)
		if (p->dst == dest)
			return p->gate;
	return dest;
}


ipdev_ioctl(q, bp)	/* handle ioctl */
	register struct queue *q;
	register struct block *bp;
{
	struct route route;
	struct arp a;

	switch (stiocom(bp)) {
	case IPIORESOLVE:
		bcopy(stiodata(bp), (char *)&a, sizeof (struct arp));
		arp_install(a.inaddr, a.enaddr);
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIOROUTE:
		bcopy(stiodata(bp), (char *)&route, sizeof (route));
		rte_install(route);
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIODROUTE:
		bcopy(stiodata(bp), (char *)&route, sizeof(route));
		rte_remove(route);
		bclear(bp);
		qreply(q, bp);
		break;
	case IPIODRESOLVE:
		bcopy(stiodata(bp), (char *)&a, sizeof (struct arp));
		arp_remove(a.inaddr);
		bclear(bp);
		qreply(q, bp);
		break;
	default:
		bp->type = M_IOCNAK;
		qreply(q, bp);
		break;
	}
}

pr_inaddr(inaddr, n)	/* print decimal inaddr */
	register in_addr inaddr;
	register n;
{
	if (n == 0)
		return;
	pr_inaddr(inaddr >> 8, n-1);
	printf("%d", inaddr & 0xff);
	if (n != 4)
		printf(".");
}

struct ipif *
find_if(dest)
	register in_addr dest;
{
	register struct ipif *ifp;

	for (ifp = ipif; ifp < &ipif[NIPLD]; ifp++) {
		if (ifp->queue == NULL)
			continue;
		if ((dest & ifp->mask) == ifp->that)
			return ifp;
	}
	return NULL;
}
