/* pt.c: named pipe */
#include "param.h"
#include "signal.h"
#include "dir.h"
#include "proc.h"
#include "file.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "user.h"
#include "errno.h"
#include "stream.h"

#define	NULL	0
#define NPT	64

static	struct queue *pt[NPT >> 1];
static	int opend[NPT];		/* opened flags */
static	int ptqopen();
extern	int nulldev();
struct	qinit	ptqinit[] = {
	{ nulldev, nulldev, nulldev, nulldev, 0, 0 },
	{ nulldev, nulldev, ptqopen, nulldev, 0, 0 }
};

/*
 * Trick here is to keep around the device end of the file until you
 * have a chance to link it with the other end after the other pt of the
 * pair is opened.  Note that (1) ptqopen is called once for each
 * end of the pair and (2) open is called every time.  
 * The full flag of the device end of the waiting stream is set
 * to keep the kernel end from writing stuff into it.  Its srvp is
 * nulldev also.
 * The pointer
 * in pt[] is for the waiting end of the pipe.  You don't need any pointers
 * after the open of the other end, but it comes in handy for a flag.
 * A separate array keeps track of the open for the masters.
 */

static
ptqopen(dev, q)		
	dev_t dev;
	register struct queue *q;
{
	register struct queue *q1, *q2, **ptp;
	register m;

	if ((m = minor(dev)) < 0 || m > NPT) {
		u.u_error = ENXIO;
		return;
	}
	if (*(ptp = &pt[m >> 1]) == NULL) {
		*ptp = q;
		q->flag |= QFULL;
		return;
	}
	q1 = OTHERQ(OTHERQ(*ptp)->next);	/* stream head of one end */
	q2 = OTHERQ(OTHERQ(q)->next);		/* stream head of the other */
	q2->next = OTHERQ(q1);
	q1->next = OTHERQ(q2);
	qfree(*ptp);
	qfree(q);
	qenable(q1);
	qenable(q2);
	wakeup(ptp);
}

/*
 * If this is called I must have a half open pt.
 */

ptqclose(dev, q)
	dev_t dev;
	register struct queue *q;
{
	flushq(OTHERQ(OTHERQ(q)->next));
}

ptopen(dev)		/* just check for master open */
{
	register m;

	if ((m = minor(dev)) & 01 && opend[m]) {
		u.u_error = EIO;
		return;
	}
	opend[m] = 1;
}

ptclose(dev, flag)	/* mark as closed */
	dev_t dev;
	int flag;
{
	register m;

	pt[(m = minor(dev)) >> 1] = NULL;
	opend[m] = 0;
}
