/* etld.c:  Ethernet/tty line discipline. */

#include "param.h"
#include "types.h"
#include "errno.h"
#include "stream.h"
#include "ethernet.h"
#include "et.h"


#define	NETLDS		8
#define	ETHERSIZE	14
#define	bsize(bp)	((bp)->wptr - (bp)->rptr)

static	etdnsrv(), etupsrv(), etqopen(), etqclose();
static	nulldev(), srv(), putq();

struct	qinit	etqinit[] = {
	{ putq, etupsrv, etqopen, etqclose, 4096, 16 },
	{ putq, etdnsrv, nulldev, nulldev, 4096, 16 },
};


static	struct	left	{		/* length so far */
	int	len;
	int	state;			/* 1 means being used */
	struct	ether eh;
} left[NETLDS];

typedef struct etpacket {
		char	d[6];
		char	s[6];
		short	t;
		unsigned short 	len;
} Etpacket;

/* values for state */

#define	LEADING		1		/* skip leading ethernet header */
#define	BODY		2		/* in the body of the message */
#define	TRAILING	3		/* skip trailing bytes */


static
etqopen(dev, q)		/* get left for up queue */
	dev_t dev;
	register struct queue *q;
{
	register struct left *p;

	for (p = left; p < &left[NETLDS]; p++)
		if (p->state == 0) {
			q->ptr = (caddr_t) p;
			p->state = LEADING;
			p->len = ETHERSIZE;
			return 0;
		}
	return ENODEV;
}

static
etqclose(dev, q)		/* release left structure */
	dev_t dev;
	register struct queue *q;
{
	((struct left *)q->ptr)->state = 0;
}


static
etupsrv(q)		/* process an incomming message */
	register struct queue *q;
{
	register struct block *bp;
	register struct left *l;
	register n;

	l = (struct left *)q->ptr;
	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, bp);
			continue;
		}
		/* must be M_DATA to get here */
		switch (l->state) {
		case LEADING:
			n = min(bsize(bp), l->len);
			bp->rptr += n;
			l->len -= n;
			if (bsize(bp) == 0) {
				freeb(bp);
				break;
			}
			if (l->len > 0)
				break;
			/* here means we are at the len field */
			l->state = BODY;
			l->len = *(u_short *)bp->rptr;
			bp->rptr += sizeof (u_short);
			if (bsize(bp) == 0) {
				freeb(bp);
				break;
			}
			/* fall thru */
		case BODY:
			if ((n = bsize(bp)) > l->len) {
				bp->wptr -= n - l->len;
				if (bp->class & S_DELIM)
					l->state = LEADING;
				else
					l->state = TRAILING;
				bp->class |= S_DELIM;
				l->len = ETHERSIZE;
			}
			(q->next->qinfo->putp)(q->next, bp);
			break;
		case TRAILING:
			if (bp->class & S_DELIM)
				l->state = LEADING;
			freeb(bp);
		}
	}
}


static
etdnsrv(q)		/* going down */
	register struct queue *q;
{
	register struct block *bp, *bp2;
	register struct left *l;
	register Etpacket *ep;

	l = (struct left *)OTHERQ(q)->ptr;
	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->type == M_IOCTL) {
		switch(stiocom(bp)) {
		case ETSET:
			bcopy(stiodata(bp), (caddr_t)&l->eh, sizeof(l->eh));
			bp->type = M_IOCACK;
			bp->rptr = bp->wptr = bp->base;
			qreply(q, bp);
			continue;
		default:
			(*q->next->qinfo->putp)(q->next, bp);
			continue;
		}
		} else if (bp->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, bp);
			continue;
		}
		/* here means data block */
		if (l->eh.type == 0) {
			freeb(bp);
			continue;		/* undefined enet stuff */
		}
		bp2 = allocb(sizeof (Etpacket));
		ep = (Etpacket *)bp2->base;
		bcopy((caddr_t)&l->eh, (caddr_t)ep, sizeof l->eh);
		bp2->wptr += ETHERSIZE+sizeof(short);
		ep->len = bp->wptr - bp->rptr;
		(*q->next->qinfo->putp)(q->next, bp2);
		bp->class |= S_DELIM;	/* mark each as frame */
		(*q->next->qinfo->putp)(q->next, bp);
	}
}

