/* tcp.c: the tcp protocol engine */

/* 
 * This code follows as close as possible RFC793 with corrections taken
 * from RFC1122.
 */

#include "param.h"
#include "types.h"
#include "stream.h"
#include "in.h"
#include "inio.h"
#include "tcp_user.h"
#include "ip_var.h"
#include "tcp.h"

int	pr_sizes_flag = 0;
int	pr_tcb_flag = 0;
int	pr_seg_flag = 0;

/*
 * Macros to do comparisons to mod 32 numbers.
 */

#define	LT(a, b)	((int)((a)-(b)) < 0)
#define	LE(a, b)	((int)((a)-(b)) <= 0)
#define	GT(a, b)	((int)((a)-(b)) > 0)
#define	GE(a, b)	((int)((a)-(b)) >= 0)

/* tcp header length in bytes */
#define	THL(t)	(((((t)->flags)>>12)&0xf)<<2)
#define	RTT_DEFAULT	3		/* in units of 500ms */

#define	OFA(tp)	((tp)->snd_una == (tp)->snd_nxt)
#define	MSL	(120)		/* in seconds */

extern	struct tcb tcb[];
extern	struct	queue *tcpldq;		/* tcpld's down queue */

tcp_open(tu, tp)		/* set up tcb and start thing rolling */
	struct tcpuser *tu;
	register struct tcb *tp;
{
	register struct tcb *p;
	register struct ipif *ifp;
	struct ipif *find_if();

	if (tu->code == TCPC_CONNECT) {
		if ((ifp = find_if(rte_xlate(tu->faddr))) == NULL) {
				tu->code = TCPC_NOROUTE;
				return 0;
		}
		if (tu->laddr == 0)
			tu->laddr = ifp->thishost;
		else if (tu->laddr != ifp->thishost) {
			tu->code = TCPC_BADLOCAL;
			return 0;
		}
	}
	tp->iss = new_iss();
	tp->rtt = RTT_DEFAULT;
	tu->lport = tcp_gport(tu->lport);
	if (tu->code == TCPC_LISTEN) {
		for (p = tcb; p < &tcb[NTCB]; p++)
			if (p->lport == tu->lport)
			if (p->laddr == INADDR_ANY || p->laddr == tu->laddr) {
				tu->code = TCPC_BOUND;
				return 0;
			}
		tp->state = LISTEN;
		tp->lport = tu->lport;
		tp->laddr = tu->laddr;
		tp->fport = 0;
		tp->faddr = INADDR_ANY;
		tu->code = TCPC_OK;
		return 1;
	}
	for (p = tcb; p < &tcb[NTCB]; p++)
		if (p->lport == tu->lport)
		if (p->laddr == tu->laddr)
		if (p->fport == tu->fport)
		if (p->faddr == tu->faddr) {
			tu->code = TCPC_BOUND;
			return 0;
		}
	tp->lport = tu->lport;
	tp->fport = tu->fport;
	tp->laddr = tu->laddr;
	tp->faddr = tu->faddr;
	send_sac(tp, tp->iss, 0, SYN);
	tp->tries = 1;		/* must be before tcprto */
	tp->rt_timer = TCPRTO(tp);
	tp->snd_una = tp->iss;
	tp->snd_nxt = tp->iss + 1;
	tp->state = SYN_SENT;
	tu->code = TCPC_OK;
	return 1;
}

/* should I free the bp on errors? */
tcp_send(tp, bp)	/* send block 'bp' out 'tp' */
	register struct tcb *tp;
	register struct block *bp;
{
	switch (tp->state) {
	case CLOSED:
		printf ("error: connection does not exist\n");
		return 0;
	case LISTEN:
		if (tp->faddr && tp->fport) {
			tp->iss = new_iss();
			send_sac(tp, tp->iss, 0, SYN);
			tp->tries = 1;
			tp->rt_timer = TCPRTO(tp);
			tp->snd_una = tp->iss;
			tp->snd_nxt = tp->iss+1;
			tp->state = SYN_SENT;
		}
		printf ("error: foreign socket unspecified\n");
		return 0;
	case SYN_SENT:
	case SYN_RECEIVED:
		return 1;
	case ESTABLISHED:
	case CLOSE_WAIT:
		send_seg(tp, bp);
		tp->tries = 1;
		tp->rt_timer = TCPRTO(tp);
		/* if urgent, snd_up = snd.nxt-1 */
		return 1;
	default:
		printf ("tcp_send: error: connection closing\n");
		return 0;
	}
}

/* should only be in SYN_RECEIVED or ESTABLISHED state */
tcp_close(tp)	/* close a socket */
	register struct tcb *tp;
{
	switch (tp->state) {
	case CLOSED:
		printf ("error: connection does not exist\n");
		return;
	case LISTEN:
	case SYN_SENT:
		free_tcb(tp);
		return;
	case SYN_RECEIVED:
		send_sac(tp, tp->snd_nxt++, tp->rcv_nxt, ACK|FIN);
		tp->tries = 1;
		tp->rt_timer = TCPRTO(tp);
		tp->state = FIN_WAIT_1;
		return;
	case ESTABLISHED:
		send_sac(tp, tp->snd_nxt++, tp->rcv_nxt, ACK|FIN);
		tp->tries = 1;
		tp->rt_timer = TCPRTO(tp);
		tp->state = FIN_WAIT_1;
		return;
	case FIN_WAIT_1:
	case FIN_WAIT_2:
		printf ("error: connection closed\n");
		return;
	case CLOSE_WAIT:
		send_sac(tp, tp->snd_nxt++, tp->rcv_nxt, ACK|FIN);
		tp->tries = 1;
		tp->rt_timer = TCPRTO(tp);
		tp->state = LAST_ACK;
		return;
	default:
		printf ("error: connection closing\n");
		return;
	}
}


tcp_in(tp, bp)		/* recv ip/tcp block */
	register struct tcb *tp;
	register struct block *bp;
{
	register struct iphdr *ip;
	register struct tcp	*t;
	register ackflag;		/* used in case SYN_SENT: */

	pr_seg("tcp_in", bp);
	ip = (struct iphdr *)bp->rptr;
	t = (struct tcp *) ((char *)ip + ((ip->ver_ihl & 0x0f) << 2));
	switch (tp->state) {
	case CLOSED:
		if ((t->flags & RST) == 0)
			if ((t->flags & ACK) == 0)
				send_sac(tp, 0, t->seq + seqlen(t, bp), RST|ACK);
			else
				send_sac(tp, t->ack, 0, RST);
		freeb(bp);
		return;
	case LISTEN:
		if (t->flags & RST) {
			freeb(bp);
			return;
		}
		if (t->flags & ACK) {
			freeb(bp);
			send_sac(tp, t->ack, 0, RST);
			return;
		}
		if (t->flags & SYN) {
			register struct tcb *ntp;
			struct tcb *alloc_tcb();

			ntp = alloc_tcb();
			if (ntp == NULL) {
				freeb(bp);
				printf ("dropped conn request\n");
				return;
			}
			*ntp = *tp;
			ntp->upq = ntp->dnq = NULL;
			ntp->flags &= ~TCP_ISOPEN;
			/* security stuff not implemented */
			ntp->rcv_nxt = t->seq+1;
			ntp->irs = t->seq;
			ntp->iss = new_iss();
			ntp->faddr = ip->src;
			ntp->fport = t->src;
			ntp->laddr = ip->dest;
			tcp_ring(tp, ntp);
			ntp->rcv_wnd = 0;
			send_sac(ntp, ntp->iss, ntp->rcv_nxt, SYN|ACK);
			ntp->tries = 1;
			ntp->rt_timer = TCPRTO(ntp);
			ntp->snd_nxt = ntp->iss+1;
			ntp->snd_una = ntp->iss;
			ntp->state = SYN_RECEIVED;
			freeb(bp);
			return;
		}
		printf ("error: can't get here tcp_in\n");
		freeb(bp);
		return;
	case SYN_SENT:
		ackflag = 0;
		if (t->flags & ACK) {
			if (LE(t->ack, tp->iss) || GT(t->ack, tp->snd_nxt)) {
				if ((t->flags & RST) == 0)
					send_sac(tp, t->ack, 0, RST);
				freeb(bp);
				return;
			}
			ackflag++;	/* una <= ack <= nxt */
		}
		if (t->flags & RST) {
			if (ackflag) {
				tcp_hangup(tp);
				free_tcb(tp);
				tp->state = CLOSED;
				printf ("error: connection reset\n");
			}
			freeb(bp);
			return;
		}
		/* check security and precedence */
		if (t->flags & SYN) {
			tp->rcv_nxt = t->seq+1;
			tp->irs = t->seq;
			if (t->flags & ACK)
				tp->snd_una = t->ack;
			if (GT(tp->snd_una, tp->iss)) {
				tp->snd_wnd = t->wnd;
				tp->snd_wl1 = t->seq;
				tp->snd_wl2 = t->ack;
				if (tp->dnq)
					qenable(tp->dnq);
				tp->state = ESTABLISHED;
				tp->tries = 0;
				tp->rt_timer = 0;
				if (tp->dnq)
					qenable(tp->dnq);
				send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
				freeb(bp);
				return;
			} else {
				tp->state = SYN_RECEIVED;
				send_sac(tp, tp->iss, tp->rcv_nxt, SYN|ACK);
				tp->tries = 1;
				tp->rt_timer = TCPRTO(tp);
				freeb(bp);
				return;
			}
		}
		/* if I got here, not SYN or RST */
		freeb(bp);
		return;
	case SYN_RECEIVED:
	case ESTABLISHED:
	case FIN_WAIT_1:
	case FIN_WAIT_2:
	case CLOSE_WAIT:
	case CLOSING:
	case LAST_ACK:
	case TIME_WAIT:
	default:
		tcp_in2(tp, bp, t);	/* do normal processing */
	}
}

tcp_ring(tp, ntp)		/* incoming connection */
	register struct tcb *tp, *ntp;
{
	register struct tcpuser *up;
	register struct block *bp;

	bp = allocb(sizeof (struct tcpuser));
	bp->wptr += sizeof (struct tcpuser);
	up = (struct tcpuser *)bp->rptr;
	up->faddr = ntp->faddr;
	up->fport = ntp->fport;
	up->laddr = ntp->laddr;
	up->lport = ntp->lport;
	up->param = ntp - &tcb[0];	/* index */
	up->code = TCPC_OK;
	(*tp->upq->qinfo->putp)(tp->upq, bp);
}

tcp_in2(tp, bp, t)		/* do normal processing */
	register struct tcb *tp;
	register struct block *bp;
	register struct tcp *t;
{
	/* in this cut we dont save or rearrange segments */
	if (acceptable(tp, bp, t) == 0) {
		if ((t->flags & RST) == 0)
			send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
		freeb(bp);
		return;
	}
	/*
	 * This code required that acceptable be restrictive.  It should
	 * only allow segments that don't spill the window and start
	 * at rcv_nxt.
	 */
	if (t->flags & RST) {
		switch (tp->state) {
		case SYN_RECEIVED:
			/* should check for passive and return to listen */
			free_tcb(tp);
			tp->state = CLOSED;
			freeb(bp);
			return;
		case ESTABLISHED:
		case FIN_WAIT_1:
		case FIN_WAIT_2:
		case CLOSE_WAIT:
			tcp_hangup(tp);
			free_tcb(tp);
			tp->state = CLOSED;
			freeb(bp);
			return;
		case CLOSING:
		case LAST_ACK:
		case TIME_WAIT:
			free_tcb(tp);
			tp->state = CLOSED;
			freeb(bp);
			return;
		}
	}
	/* TBD: check security and precedence */
	if (t->flags & SYN) {
		switch (tp->state) {
		case SYN_RECEIVED:
		case ESTABLISHED:
		case FIN_WAIT_1:
		case FIN_WAIT_2:
		case CLOSE_WAIT:
		case CLOSING:
		case LAST_ACK:
		case TIME_WAIT:
			tcp_hangup(tp);
			free_tcb(tp);
			tp->state = CLOSED;
			freeb(bp);
			return;
		}
	}
	if ((t->flags & ACK) == 0) {
		freeb(bp);
		return;
	}
	/* the ack bit is on */
	switch (tp->state) {
	case SYN_RECEIVED:
		if (LE(tp->snd_una, t->ack) && LE(t->ack, tp->snd_nxt)) {
			tp->tries = 0;
			tp->rt_timer = 0;
			tp->state = ESTABLISHED;
			tp->snd_wnd = t->wnd;
			tp->snd_wl1 = t->seq;
			tp->snd_wl2 = t->ack;
			if (tp->dnq)
				qenable(tp->dnq);
		} else {
			send_sac(tp, t->ack, 0, RST);
			freeb(bp);
			return;
		}
		/* fall through */
	case CLOSE_WAIT:
	case ESTABLISHED:
		if (ack_estab(tp, bp, t) == 0) {
			freeb(bp);
			return;
		}
		break;
	case FIN_WAIT_1:
		if (ack_estab(tp, bp, t) == 0) {
			freeb(bp);
			return;
		}
		if (tp->snd_una == tp->snd_nxt) {
			tp->state = FIN_WAIT_2;
			tp->rt_timer = 0;
			tp->tries = 0;
		}
		break;
	case FIN_WAIT_2:
		tcp_hangup(tp);
		if (ack_estab(tp, bp, t) == 0) {
			freeb(bp);
			return;
		}
		break;
	case CLOSING:
		if (ack_estab(tp, bp, t) == 0) {
			freeb(bp);
			return;
		}
		if (t->ack == tp->snd_una) {
			tp->state = TIME_WAIT;
			tp->rt_timer = 0;
			tp->tries = 0;
			tp->tw_timer = 2*MSL;
		} else {
			freeb(bp);
			return;
		}
		break;
	case LAST_ACK:
		if (t->ack == tp->snd_una+1) {
			free_tcb(tp);
			tp->state = CLOSED;
		}
		freeb(bp);
		return;
	case TIME_WAIT:
		/*
		 * The only thing that can arrive in this state is
		 * a retransmittion of the remote fin.  Ack it and
		 * restart a 2 MSL timeout.
		 */
		send_sac(tp, t->seq, 0, ACK);
		tp->tw_timer = 2 * MSL;
		freeb(bp);
		return;
	}
	/* sixth, check urg bit.  TBD */
	/* seventh, process segment */
	switch (tp->state) {
	case ESTABLISHED:
	case FIN_WAIT_1:
	case FIN_WAIT_2: {
		register urgptr;

		if (tp->dnq)
			qenable (tp->dnq);
		urgptr = (t->flags & URG) ? t->urgptr : 0;
		bp->rptr = (u_char *) t;
		bp->rptr += THL(t);
		if (pr_sizes_flag && bp->wptr - bp->rptr)
			printf ("input: %d bytes\n", bp->wptr - bp->rptr);
		if (bp->rptr < bp->wptr)
		if (tp->upq) {
			register l = bp->wptr - bp->rptr;
			register struct block *dbp = dupb(bp);
			bp->rptr += urgptr;
			if (bp->rptr < bp->wptr)
				(*tp->upq->qinfo->putp)(tp->upq, bp);
			else
				freeb(bp);
			bp = dbp;	/* I need a copy of it */
			tp->rcv_nxt += l;
			tp->rcv_wnd -= l;
			tp->rcv_wnd += urgptr;
			send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
			break;
		}
		break;
	}
	case CLOSE_WAIT:
	case CLOSING:
	case LAST_ACK:
	case TIME_WAIT:
		freeb(bp);
		return;
	}
	/* eighth, check the FIN bit */
	if ((t->flags & FIN) == 0) {
		freeb(bp);
		return;
	}
	switch (tp->state) {
	case CLOSED:
	case LISTEN:
	case SYN_SENT:
		freeb(bp);
		return;
	}
	tcp_hangup(tp);
	tp->rcv_nxt++;		/* advance over FIN */
	send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
	switch (tp->state) {
	case SYN_RECEIVED:
	case ESTABLISHED:
		tp->state = CLOSE_WAIT;
		break;
	case FIN_WAIT_1:
		if (t->ack == tp->snd_una) {
			tp->state = TIME_WAIT;
			timers_off(tp);
			tp->tw_timer = 2 * MSL;
		} else
			tp->state = CLOSING;
		break;
	case FIN_WAIT_2:
		tp->state = TIME_WAIT;
		timers_off(tp);
		tp->tw_timer = 2 * MSL;
		break;
	case CLOSE_WAIT:
	case CLOSING:
	case LAST_ACK:
		/* stay in same state */
		break;
	case TIME_WAIT:
		break;
	}
	freeb(bp);
}

tcp_twexpire(tp)
	register struct tcb *tp;
{
	free_tcb(tp);
	tp->state = CLOSED;
	if (pr_sizes_flag)
		printf ("tcp_twexpire fired\n");
}

/*
 * the persistance timer is used for two things. 
 * 	1) Window probs that start in the normal resend time
 *	2) Keep alive that start KEEPALIVE_TIME after being idle.
 */

tcp_perexpire(tp)
	register struct tcb *tp;
{
	/* send unacceptable to get the ack back */
	send_sac(tp, tp->snd_nxt-1, tp->rcv_nxt, ACK);
	if (++tp->tries > MAXTRIES) {
		tcp_hangup(tp);
		free_tcb(tp);
		tp->state = CLOSED;
		return;
	}
	tp->per_timer = TCPRTO(tp);
}


tcp_rtexpire(tp)		/* retransmisison timer expired */
	register struct tcb *tp;
{
	if (pr_sizes_flag)
		printf ("tcp_rtexpire fired; state %d\n", tp->state);
	if (++tp->tries > MAXTRIES) {
		tcp_hangup(tp);
		free_tcb(tp);
		tp->state = CLOSED;
		return;
	}
	switch (tp->state) {
	case SYN_SENT:
		send_sac(tp, tp->iss, tp->rcv_nxt, SYN);
		tp->rt_timer = TCPRTO(tp);
		return;
	case SYN_RECEIVED:
		if (tp - &tcb[0] & 1)	/* origination */
			send_sac(tp, tp->iss, tp->rcv_nxt, ACK);
		else			/* passive */
			send_sac(tp, tp->iss, tp->rcv_nxt, ACK|SYN);
		tp->rt_timer = TCPRTO(tp);
		return;
	case ESTABLISHED:
		if (tp->dnq == NULL || tp->dnq->first == NULL)
			break;
		send_seg(tp, dupb(tp->dnq->first));
		tp->rt_timer = TCPRTO(tp);
		return;
	case CLOSE_WAIT:
	case CLOSING:
		send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
		tp->rt_timer = TCPRTO(tp);
		return;
	case FIN_WAIT_1:
	case LAST_ACK:
		send_sac(tp, tp->snd_nxt, tp->rcv_nxt, FIN);
		tp->rt_timer = TCPRTO(tp);
		return;
	}
}

tcp_clexpire(tp)
	register struct tcb *tp;
{
	tp->flags |= TCP_CLEXPIRED;
	if (tp->dnq);
		qenable(tp->dnq);
}


	



ack_estab(tp, bp, t)	/* handle normal ack */
	register struct tcb *tp;
	register struct block *bp;
	register struct tcp *t;
{
	if (LT(tp->snd_una, t->ack) && LE(t->ack, tp->snd_nxt)) {
		acked(tp, t->ack);
		tp->snd_una = t->ack;
		if (tp->snd_una == tp->snd_nxt) {
			tp->rt_timer = 0;
			tp->tries = 0;
		}
	}
#ifdef notdef
	if (LE(t->ack , tp->snd_una))
		return 1;		/* ignore the ack field */
#endif
	if (GT(t->ack, tp->snd_nxt)) {
		send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);	/* wind */
		freeb(bp);
		return 0;		/* don't continue */
	}
	if (LE(tp->snd_una, t->ack) && LE(t->ack, tp->snd_nxt)) {
		if (LT(tp->snd_wl1, t->seq) 
		|| (tp->snd_wl1 == t->seq && LE(tp->snd_wl2, t->ack))) {
			tp->snd_wnd = t->wnd;
			tp->snd_wl1 = t->seq;
			tp->snd_wl2 = t->ack;
			if (tp->dnq)
				qenable(tp->dnq);
		}
	}
	if (tp->dnq)
		qenable(tp->dnq);
	return 1;
}


send_sac(tp, seq, ack, ctl)	/* send empty segment */
	struct tcb *tp;
{
	struct block *bp;
	register struct iphdr *ip;
	register struct tcp *to;

	bp = allocb(64);
	ip = (struct iphdr *)bp->rptr;
	bp->wptr += sizeof (struct iphdr) + sizeof (struct tcp);
	ip->ver_ihl = (4<<4)|sizeof (struct iphdr)>>2;
	ip->tos = 0;
	ip->tlen = bp->wptr - bp->rptr;
	ip->ttl = 20;
	ip->proto = 6;
	ip->src = tp->laddr;
	ip->dest = tp->faddr;
	to = (struct tcp *)&ip[1];
	to->src = tp->lport;
	to->dest = tp->fport;
	to->seq = seq;
	to->ack = ack;
	to->flags = ctl;
	to->flags |= (6<<12);	/* always the same */
	to->wnd = tp->rcv_wnd;
	to->urgptr = 0;
	to->options = 0;
	to->chksum = 0;
	to->chksum = tcp_dosums(to, tp->faddr, tp->laddr, sizeof(struct tcp));
	bp->class |= S_DELIM;
	pr_seg("send_sac", bp);
	if (tcpldq)
		(*tcpldq->qinfo->putp)(tcpldq, bp);
	else
		freeb(bp);
	pr_tcb("send_sac: ", tp);
}


/*
 * send a segment.  advances snd_nxt length of bp data past
 * snd_una.  This means this will have to change for more than
 * one unacknowledged segment at a time.
 */

send_seg(tp, bp)	/* send a segment */
	struct tcb *tp;
	struct block *bp;
{
	register struct block *obp;

	obp = allocb(1500);
	if (obp->lim - obp->base < 1500)
		panic("send_seg: need bigger block");
	setip(obp, tp, bp->wptr - bp->rptr);
	settcp(obp, tp, tp->snd_una, tp->rcv_nxt, ACK);
	tp->snd_nxt = tp->snd_una + (bp->wptr - bp->rptr);
	setdata(obp, bp->rptr, bp->wptr - bp->rptr);
	if (pr_sizes_flag)
		printf("send_seg: %d bytes\n", bp->wptr - bp->rptr);
	freeb(bp);
	setcks(obp);
	obp->class |= S_DELIM;
	pr_seg("send_seg", obp);
	if (tcpldq)
		(*tcpldq->qinfo->putp)(tcpldq, obp);
	else
		freeb(obp);
	pr_tcb("send_seg: ", tp);
}

setip(bp, tp, n)	/* build ip header */
	register struct block *bp;
	register struct tcb *tp;
	register n;	/* size of data, not tcp + ip */
{
	register struct iphdr *ip;

	ip = (struct iphdr *)bp->rptr;
	bp->wptr += sizeof (struct iphdr);
	ip->ver_ihl = (4<<4) | sizeof (struct iphdr) >> 2;
	ip->tos = 0;
	ip->tlen = n;
	ip->tlen += sizeof (struct iphdr) + sizeof (struct tcp);
	ip->ttl = 20;
	ip->proto = 6;
	ip->src = tp->laddr;
	ip->dest = tp->faddr;
}

settcp(bp, tp, seq, ack, ctl)	/* build tcp header */
	register struct block *bp;
	register struct tcb *tp;
{
	register struct tcp *to;
	
	to = (struct tcp *)(bp->rptr + sizeof (struct iphdr));
	bp->wptr += sizeof (struct tcp);
	to->src = tp->lport;
	to->dest = tp->fport;
	to->seq = seq;
	to->ack = ack;
	to->flags = ctl;
	to->flags |= (6<<12);
	to->wnd = tp->rcv_wnd;
	to->options = 0;
	to->urgptr = 0;
	to->chksum = 0;
}

setdata(bp, p, n)	/* add data to end of ip/tcp block */
	register struct block *bp;
	register u_char *p;
	register n;
{
	register u_char *tp;

	tp = bp->wptr;
	while (n--)
		*tp++ = *p++;
	bp->wptr = tp;
}

setcks(bp)		/* set tcp check sum */
	register struct block *bp;
{
	register struct iphdr *ip;
	register struct tcp *to;
	register n;

	ip = (struct iphdr *)bp->rptr;
	to = (struct tcp *)&ip[1];
	n = bp->wptr - (u_char *)&ip[1];
	if (n & 01)
		bp->wptr++;
	to->chksum = tcp_dosums(to, ip->src, ip->dest, n);
}


free_tcb(tp)
	register struct tcb *tp;
{
	register s;

	if (tp->flags & (TCP_CLOSING || TCP_CLEXPIRED))
		wakeup(tp);
	s = tp->flags & TCP_ISOPEN;
	bzero((char *)tp, sizeof (struct tcb));
	tp->flags = s;
}

new_iss()		/* create new iss to use */
{
	extern int time;		/* clock.c thing */
	static int counter = 0;

	return ((time<<16) | ++counter & 0xffff);
}


/*
 * Restricted version of acceptable.  This simplifies the protocol.
 * since out of order packets in this application are mostly
 * the result of a dropped packet, there is no reason to keep the
 * packet until the other parts show up.  I just ignore them because
 * only the next segment in the sequence is allowed.
 */

static
acceptable(tp, bp, t)		/* does seq fall in window */
	register struct tcb *tp;
	register struct block *bp;
	register struct tcp *t;
{
	return t->seq == tp->rcv_nxt 
		&& LE(t->seq+seqlen(t, bp), tp->rcv_nxt + tp->rcv_wnd);
}

seqlen(t, bp)		/* segment length in sequence space */
	register struct tcp *t;
	register struct block *bp;	/* where t lives */
{
	return (bp->wptr - ((u_char *)t + THL(t)) +
		(t->flags & SYN ? 1 : 0));
}



timers_off(tp)
	register struct tcb *tp;
{
	tp->rt_timer = tp->tw_timer = 0;
	tp->per_timer = 0;
	tp->tries = 0;
}


acked(tp, seq)		/* release stuff on retransmit queue */
	register struct tcb *tp;
	register unsigned long seq;
{
	register struct block *bp;
	register unsigned long l, n;

	if (tp->dnq == NULL) {
		return;
	}
	bp = tp->dnq->first;
	if (bp == NULL) {
		printf ("acked: %d: null bp\n", tp - tcb);
		return;
	}
	if (bp->type != M_DATA) {
		printf ("acked: %d: missing data\n", tp - &tcb[0]);
		return;
	}
	l = seq - tp->snd_una;
	do {
		n = min(l, bp->wptr - bp->rptr);
		bp->rptr += n;
		l -= n;
		if (bp->rptr == bp->wptr) {
			freeb(getq(tp->dnq));
			bp = tp->dnq->first;
		}
	} while (l > 0 && bp != NULL);
	if (tp->dnq)
		qenable(tp->dnq);
}


/*
 * This is another version of the cksum routine.
 * This one takes an initial value argument so the header
 * doesn't have to be contigous.
 */

tcp_cksum(buf, nbytes, init)
{
	asm("movl sp@(12), d1");	/* d1 is nbytes */
	asm("addl #1, d1");		/* round up to words */
	asm("movl sp@(8),  a0");	/* a0 is pointer to buffer */
	asm("movl sp@(16), d0");	/* d0 is accumulator */
	asm("notw d0");			/* un complement the accumulator */
	asm("lsrl #1, d1");		/*   convert nbytes into */
	asm("subl #1, d1");		/*     value for dbra */
	asm("1: addw a0@+, d0");		/* add to accumulator 16 bits */
	asm("bccs 2f");			/* if we didn't carry out skip */
	asm("addw #1, d0");		/* otherwise, rotate to lsb */
	asm("2: dbra d1, 1b");		/* for nbyte/2 times */
	asm("notw d0");			/* 1's complement the 1's complement sum */
}

tcp_dosums(t, faddr, laddr, l)	/* calculate checksum */
	register struct tcp *t;
	in_addr faddr, laddr;
	register l;
{
	u_short sum;
	u_long word;

	sum = tcp_cksum(&faddr, 4, ~0);
	sum = tcp_cksum(&laddr, 4, sum);
	word = (6<<16)| l;
	sum = tcp_cksum(&word, 4, sum);
	if (l & 01)		/* odd number of bytes */
		((char *)t)[l] = 0;		/* pad */
	sum = tcp_cksum((char *)t, l, sum);
	return sum;
}

struct tcb *
alloc_tcb()	/* for incomming connection */
{
	register struct tcb *tp;

	for (tp = tcb; tp < &tcb[NTCB]; tp += 2)	/* check evens */
		if (tp->state == CLOSED || tp->state == NULL)
			return tp;
	return NULL;
}

tcp_inittcb()
{
	register struct tcb *tp;

	for (tp = tcb; tp < &tcb[NTCB]; tp++)
		tp->state = CLOSED;
}

/*
 * Debugging stuff.
 */

pr_seg(s, bp)
	char *s;
	register struct block *bp;
{
	register struct tcp *tcp;
	register struct iphdr *ip;

	if (pr_seg_flag == 0)
		return;
	ip = (struct iphdr *)bp->rptr;
	tcp = (struct tcp *)&ip[1];
	printf ("%s:", s);
	printf("tlen %d, ", ip->tlen);
	printf("seq: %d, ack %d, flags: 0%o, wnd: %d\n", 
		tcp->seq, tcp->ack, tcp->flags, tcp->wnd);
}

pr_tcb(s, tcp)		/* print tcb */
	register char *s;
	register struct tcb *tcp;
{
	extern struct tcb tcb[];
	static char *sn[] = {
		"CLOSED",
		"LISTEN", "SYN_SENT", "SYN_RECEIVED", "ESTABLISHED", "FIN_WAIT_1",
		"FIN_WAIT_2", "CLOSE_WAIT", "CLOSING", "LAST_ACK", "TIME_WAIT",
	};

	if (pr_tcb_flag == 0)
		return;
	printf("     %s ", s);
	printf("%d:", tcp - &tcb[0]);	/* which tcb */
	if (tcp->state < 0 || tcp->state > TIME_WAIT)
		printf("UNKNOWN");
	else
		printf("%s ", sn[tcp->state]);
	printf(" send(%d, %d, %d), rcv(%d, %d), ",
		tcp->snd_una, tcp->snd_nxt, tcp->snd_wnd,
		tcp->rcv_nxt, tcp->rcv_wnd);
	printf("\n");
}

tcp_gport(port)	/* alloc new port number */
	register short port;
{
	static tcpport = 1024;	/* scratch space starts here */
	register struct tcb *tp;

	if (port != 0)
		return port;
	for (;;) {		/* very large name space */
		port = tcpport;
		if (++tcpport > 0x7fff)
			tcpport = 1024;
		for (tp = tcb; tp < &tcb[NTCB]; tp++)
			if (tp->state != CLOSED || tp->state != NULL)
			if (port == tp->lport)
				break;
		if (tp >= &tcb[NTCB])
			return port;
	}
}

tcp_hangup(tp)		/* send device down up stream */
	register struct tcb *tp;
{
	register struct block *bp;

	if (tp->upq == NULL)
		return;
	bp = allocb(0);
	bp->type = M_HANGUP;
	(*tp->upq->qinfo->putp)(tp->upq, bp);
}

tcp_tick()
{
	register struct tcb *tp;
	register s;

	s = splnet();
	for (tp = &tcb[0]; tp < &tcb[NTCB]; tp++) {
		if (tp->rt_timer && --tp->rt_timer == 0)
			tcp_rtexpire(tp);
		if (tp->tw_timer && --tp->tw_timer == 0)
			tcp_twexpire(tp);
		if (tp->per_timer && --tp->per_timer == 0)
			tcp_perexpire(tp);
		if (tp->cl_timer && --tp->cl_timer == 0)
			tcp_clexpire(tp);
	}
	timeout(tcp_tick, 0, 30);
	splx(s);
}
