/* entest.c:  ethernet driver test */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/enio.h>
#include <sys/ethernet.h>
#include <sys/inet/in.h>
#include <sys/inet/ip_var.h>
#include <sys/filio.h>
#include <sys/inio.h>
#include <signal.h>

#define	NARPC	200	/* number of entries in arp cache */
int	dflag;		/* print ARP requests on standard out */
int	fflag;		/* arpstorm workaround */
char	*progname;
char	*ipdev;		/* name of ip-device */
char	*arpdev;	/* name of arp device */
in_addr	aton();	/* convert ascii to in_addr */
in_addr	localhost;	/* address of ip-device */
u_char	our_enet[6];	/* the enet address of our controller */
in_addr	network;	/* address of connected network */
in_addr	mask;		/* mask for subnetting */
u_short	arp_type = 0x0806;
u_short	ip_type = 0x0800;
FILE	*logfp;		/* file pointer to logfile */
char	logfile[] = "/usr/ipc/log/ipconfig";
int	ipfd, arpfd;
int	icmpfd;		/* fd of icmp ip device */
int	frwdfd;		/* fd of forward ip device */

main(argc, argv)
	int argc;
	char *argv[];
{
	fd_set mv, rv;
	extern int ip_ld;

	progname = *argv;
	if (argc < 4) {
		usage(progname);
		return 1;
	}
	setpgrp(getpid(), getpid());
	for (++argv, --argc; **argv == '-'; argv++, argc--)
		if (argv[0][1] == 'm') {
			mask = aton(*++argv);
			argc--;
		} else if (argv[0][1] == 'd')
			dflag++;
		else if (argv[0][1] == 'f')
			fflag++;
		else {
			usage(progname);
			return 1;
		}
	if (argc < 3) {
		usage(progname);
		return 1;
	}
	ipdev = *argv++;
	localhost = in_address(*argv++);
	network = in_address(*argv++);
	argc -= 3;
	if (argc > 0)
		arpdev = *argv;
	open_log();
	FD_ZERO(mv);
	FD_ZERO(rv);
	ipfd = eopen(ipdev, 2);
	FD_SET(ipfd, mv);
	if (arpdev) {
		eioctl(ipfd, ENIOADDR, our_enet);
		eioctl(ipfd, ENIOTYPE, &ip_type);
	}
	eioctl(ipfd, FIOPUSHLD, &ip_ld);
	if (arpdev)
		eioctl(ipfd, IPIOARP, NULL);
	eioctl(ipfd, IPIOLOCAL, &localhost);
	eioctl(ipfd, IPIONET, &network);
	eioctl(ipfd, IPIOMASK, &mask);
	if (arpdev) {
		arpfd = eopen(arpdev, 2);
		FD_SET(arpfd, mv);
		eioctl(arpfd, ENIOTYPE, &arp_type);
	}
	icmpfd = open("/dev/ipicmp", 2);
	if (icmpfd != -1) 
		FD_SET(icmpfd, mv);
	if ((frwdfd = open("/dev/ipforw", 2)) != -1)
		FD_SET(frwdfd, mv);
	for (;;) {
		rv = mv;
		select(NOFILE, &rv, NULL, 60 * 1000);
		if (FD_ISSET(ipfd, rv))
			do_ip();
		if (FD_ISSET(arpfd, rv))
			do_arp();
		if (FD_ISSET(icmpfd, rv))
			do_icmp();
		if (FD_ISSET(frwdfd, rv))
			do_frwd();
		/* do time stuff here */
	}
}


#define	SIZEOF_ARPC	42
#define	ARPC_REQUEST	1
#define	ARPC_REPLY	2
struct	arppkt	{
	char	edest[6];
	char	esrc[6];
	u_short	type;		/* end of ethernet frame */
	u_short	hrd;		/* hardware address space */
	u_short	pro;		/* protocol address space*/
	u_char	hln;		/* length of each hardware address */
	u_char	pln;		/* length of each protocol address */
	u_short	op;		/* opcode */
	u_char	sha[6];		/* hardware address of sender */
	u_char	spa[4];		/* sender protocol address */
	u_char	tha[6];		/* target hardware address */
	u_char	tpa[4];		/* target hardware address */
};

struct	arpcache	{	/* translations I know about */
	in_addr	ip;
	u_char	enet[6];
};
struct	arpcache arptab[NARPC];


do_arp()		/* process an arp message from 'fd' */
{
	char frame[1520];	/* place to put a frame */
	register struct arppkt *p;
	in_addr ip;
	int n;

	if ((n = read(arpfd, frame, sizeof frame)) < SIZEOF_ARPC) {
		note("short arp frame: %d", (char *)n);
		return;
	}
	p = (struct arppkt *)frame;
	if (p->type != 0x806 || p->hrd != 1 || p->pro != 0x800) {
		note("Bad arp packet: ");
		prarpp(p, logfp);
		return;
	}
	if (dflag)
		prarpp(p, stdout);
	bcopy((char *)p->spa, (char *)&ip, sizeof ip);
	iplookup(ip, p->sha);
	if (memcmp(p->tpa, (char *)&localhost, sizeof localhost) != 0)
		return;
	if (p->op != ARPC_REQUEST)
		return;
	memcpy(p->tha, p->sha, sizeof p->tha);
	memcpy(p->tpa, p->spa, sizeof p->tpa);
	memcpy(p->edest, p->sha, sizeof p->edest);
	memcpy(p->sha, our_enet, sizeof p->sha);
	memcpy(p->spa, (char *)&localhost, sizeof p->spa);
	p->op = ARPC_REPLY;
	write(arpfd, frame, SIZEOF_ARPC);
}

prarpp(p, fp)	/* print arp packet on file 'fp' */
	register struct arppkt *p;
	FILE *fp;
{
	pf(fp, "edest", p->edest, sizeof p->edest, 'h');
	pf(fp, "esrc", p->esrc, sizeof p->esrc, 'h');
	pf(fp, "type", &p->type, sizeof p->type, 'h');
	pf(fp, "hrd", &p->hrd, sizeof p->hrd, 'h');
	pf(fp, "pro", &p->pro, sizeof p->pro, 'h');
	pf(fp, "hln", &p->hln, sizeof p->hln, 'h');
	pf(fp, "pln", &p->pln, sizeof p->pln, 'h');
	pf(fp, "op", &p->op, sizeof p->op, 'h');
	pf(fp, "sha", p->sha, sizeof p->sha, 'x');
	pf(fp, "spa", p->spa, sizeof p->spa, 'd');
	pf(fp, "tha", p->tha, sizeof p->tha, 'x');
	pf(fp, "tpa", p->tpa, sizeof p->tpa, 'd');
	fprintf(fp, "\n");
}

pf(fp, s, p, n, type)	/* put data in hex */
	FILE *fp;
	char *s;
	u_char *p;
	register n;
	char type;
{
	fprintf(fp, "%s: ", s);
	while (n--)
		if (type == 'h')
			fprintf(fp, "%02x", *p++);
		else if (type == 'd')
			fprintf(fp, "%d%c", *p++, n > 0 ? '.' : ' ');
		else if (type == 'x')
			fprintf(fp, "%02x%c", *p++, n > 0 ? '.' : ' ');
	fprintf(fp, "\n");
}

usage(s)
	char *s;
{
	fprintf(stderr, "usage: %s [-m mask][-df] ip-device localhost network [arp-device]\n", s);
}

eopen(name, how)
	char *name;
{
	register fd;

	if ((fd = open(name, how)) == -1)
		error("open failed: %s", name);
	return fd;
}

eioctl(fd, cmd, arg)	/* error version of ioctl */
	int fd;
	int cmd;
	char *arg;
{
	if (ioctl(fd, cmd, arg) == -1)
		error("ioctl 0x%x failed", (char *)cmd);
}

error(s1, s2)
	register char *s1, *s2;
{
	extern int errno, sys_nerr;
	extern char *sys_errlist[], *progname;

	if (progname)
		fprintf(stderr, "%s: ", progname);
	fprintf(stderr, s1, s2);
	if (errno > 0 && errno < sys_nerr)
		fprintf(stderr, " (%s)", sys_errlist[errno]);
	fprintf(stderr, "\n");
	exit(1);
}

open_log()	/* start log file */
{
	long now;

	if ((logfp = fopen(logfile, "a")) == NULL)
		error("Can't open logfile '%s'", logfile);
	time(&now);
	fprintf(logfp, "%s started on %s", progname, ipdev);
	if (arpdev)
		fprintf(logfp, " %s", arpdev);
	fprintf(logfp, " on %s", ctime(&now));
	fflush(logfp);
}

note(s1, s2)		/* make a note in the log file */
	char *s1, *s2;
{
	fprintf(logfp, "%08x: ", localhost);
	fprintf(logfp, s1, s2);
	fprintf(logfp, "\n");
	lseek(fileno(logfp), 0, 2);	/* skip to end */
	fflush(logfp);
}

in_addr
aton(s)		/* convert internet ascii to in_addr */
	register char *s;
{
	int i1 = 0, i2 = 0, i3 = 0, i4 = 0;

	if (index(s, '.') != NULL) {
		if (sscanf(s, "%d.%d.%d.%d", &i1, &i2, &i3, &i4) != 4)
			error("Bad IP addr %s", s);
	} else if (sscanf(s, "%x", &i4) != 1)
		error("Bad IP addr %s", s);
	else
		error("Bad IP addr (%s)", s);
	return (i1<<24|i2<<16|i3<<8|i4);
}

do_ip()	/* note in_addr of IP not in arptable */
{
	u_char frame[5120];
	struct arppkt a;
	in_addr who;
	int n;
	u_char *p;

	n = read(ipfd, frame, sizeof frame);
	who = *(in_addr *)frame;
	memset(a.edest, 0xff, 6);
	a.type = arp_type;
	a.hrd = 1;
	a.pro = 0x800;
	a.hln = 6;
	a.pln = 4;
	a.op = ARPC_REQUEST;
	memcpy(a.sha, our_enet, sizeof a.sha);
	memcpy(a.spa, (char *)&localhost, sizeof a.spa);
	memset(a.tha, 0, sizeof a.tha);
	memcpy(a.tpa, (char *)&who, sizeof a.tpa);
	write(arpfd, (char *)&a, sizeof a);
	note("requested arp for %s", in_host(who));
}

/* 
 * iplookup controls access to the cache.  This routine could be
 * rewritten for better performance.  It currently doesn't
 * timeout any entries.
 * 
 * It doesn't ioctl the changes down to the ip device yet.
 */

iplookup(ip, enet)	/* look up 'ip' in cache */
	in_addr ip;
	u_char enet[6];
{
	register struct arpcache *p, *ap;
	struct arp r;

	for (p = arptab; p < &arptab[NARPC]; p++) {
		if (p->ip == ip) {
			if (memcmp(p->enet, enet, sizeof p->enet))
				return;
			memcpy(p->enet, enet, sizeof enet);
			goto setit;
		}
		if (ap == NULL && p->ip == NULL)
			ap = p;
	}
	ap->ip = ip;
	memcpy(ap->enet, enet, sizeof ap->enet);
setit:
	memcpy(r.enaddr, enet, sizeof r.enaddr);
	r.inaddr = ip;
	ioctl(ipfd, IPIORESOLVE, &r);
}

dump_cache()		/* print out cache */
{
	register struct arpcache *p;
	u_char ip[4];

	for (p = arptab; p < &arptab[NARPC]; p++)
		if (p->ip != NULL) {
			memcpy((char *)ip, &p->ip, sizeof ip);
			printf("%d.%d.%d.%d: ", ip[0], ip[1], ip[2], ip[3]);
			prenet(p->enet);
			printf("\n");
		}
}

prenet(enet)
	u_char enet[6];
{
	register n = 6;
	register u_char *p = enet;

	while (n--)
		printf("%02x%c", *p++, n > 0 ? '.' : '\n');
}

pr_fdv(n, vp, s)
	register n;
	fd_set *vp;
	char *s;
{
	register int i;
		

	printf("%s: ", s);
	for (i = 0; i < n; i++)
		if (FD_ISSET(i, *vp))
			printf ("%d ", i);
	printf("\n");
}


/*
 * print message about icmp message.
 */


#define ICMP_REPLY	0
#define	ICMP_REQUEST	8

cksum(p, len)
char *p;
int len;
{

	asm("movl	sp@(0x0c), d1");
	asm("movl	sp@(8), a0");
	asm("lsrl	#1,d1");
	asm("subl	#1,d1");
	asm("clrl	d0");
	asm("1:	addw	a0@+,d0");
	asm("bccs	2f");
	asm("addw	#1,d0");
	asm("2:	dbra	d1,1b");
	asm("notw	d0");
}



struct	icmp	{
	u_char	type;
	u_char	code;
	u_short checksum;
	u_short id;
	u_short seq;
};

do_icmp()
{
	char buf[1500];
	register struct iphdr *ip;
	register struct icmp *p;
	int n;
	u_long tmp_addr;

	n = read(icmpfd, buf, sizeof buf);
	ip = (struct iphdr *) buf;
	p = (struct icmp *)(buf + sizeof (struct iphdr));
	if (p->type == ICMP_REQUEST) {
		p->type = ICMP_REPLY;
		p->checksum = 0;
		p->checksum = cksum((char *)p, n - sizeof(struct iphdr));
		tmp_addr = ip->src;
		ip->src = ip->dest;
		ip->dest = tmp_addr;
		if (write(icmpfd, buf, n) < 0)
			perror("ipconfig:do_icmp:write");
	} else {
		note("icmp: type %d, code %d\n", p->type, p->code);
	}
}

do_frwd()		/* loop back.  could be a ld */
{
	char frame[2048];
	int n;

	n = read(frwdfd, frame, sizeof frame);
	if (n > 0)
		write(frwdfd, frame, n);
}
