/* tftpd.c:  trivial file transfer protocol daemon */
#include <sys/param.h>
#include <sys/inet/in.h>
#include <sys/inet/udp_user.h>
#include <signal.h>
#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include "tftp.h"

#define	ntohs(s)	(s)
#define	htons(s)	(s)

#define	DEBUG
#define	BUFFER	(512+2+2)

extern	int errno;
int	f;
char	buf[BUFFER];
int	trace;		/* print trace stuff to stderr */

main(argc, argv)
	char *argv[];
{
	register struct tftphdr *tp;
	register int n;
	struct in_service *sp;

	if (argc > 1)
		trace++;
	setpgrp(getpid(), getpid());
	if ((sp = in_service("tftp", "udp", 0)) == NULL) {
		printf("tftpd: udp/tftp: unknown service\n");
		exit(1);
	}
#ifndef DEBUG
	if (fork())
		exit(0);
	for (f = 0; f < 10; f++)
		(void) close(f);
	(void) open("/", 0);
	(void) dup2(0, 1);
	(void) dup2(0, 2);
	setpgrp(getpid());
#endif
	if ((f = udp_datagram(sp->port)) == -1) {
		perror(argv[0]);
		exit(1);
	}
	for (;;) {
		errno = 0;
		while ((n = read(f, buf, sizeof buf)) <= 0)
			if (n < 0)
				perror("receive");
		tp = (struct tftphdr *)(buf + sizeof(struct udpaddr));
		tp->th_opcode = ntohs(tp->th_opcode);
		if (tp->th_opcode == RRQ || tp->th_opcode == WRQ) {
			if (fork() == 0)
				if (fork() == 0)
					tftp((struct udpaddr *)buf, tp, n);
			else
				exit(0);
			wait(NULL);
		}
	}
}

int	validate_access();
int	sendfile(), recvfile();

struct formats {
	char	*f_mode;
	int	(*f_validate)();
	int	(*f_send)();
	int	(*f_recv)();
} formats[] = {
	{ "netascii",	validate_access,	sendfile,	recvfile },
	{ "octet",	validate_access,	sendfile,	recvfile },
#ifdef notdef
	{ "mail",	validate_user,		sendmail,	recvmail },
#endif
	{ 0 }
};

int	fd;			/* file being transferred */

/*
 * Handle initial connection protocol.
 */
tftp(client, tp, size)
	struct udpaddr *client;
	struct tftphdr *tp;
	int size;
{
	char *cp;
	int ecode;
	register struct formats *pf;
	char *filename, *mode;

	close(f);
	if ((f = udp_connect(0, client->host, client->port)) == -1) {
		perror("connect");
		exit(1);
	}
	cp = tp->th_stuff;
	size -= 2;	/* skip opcode */
	get_strings(cp, &filename, &mode, size);
	for (cp = mode; *cp; cp++)
		if (isupper(*cp))
			*cp = tolower(*cp);
	for (pf = formats; pf->f_mode; pf++)
		if (strcmp(pf->f_mode, mode) == 0)
			break;
	if (pf->f_mode == 0) {
		nak(EBADOP);
		exit(1);
	}
	ecode = (*pf->f_validate)(filename, client, tp->th_opcode);
	if (ecode) {
		nak(ecode);
		exit(1);
	}
	(*(tp->th_opcode == WRQ ? pf->f_recv : pf->f_send))(pf);
	exit(0);
}

/*
 * Validate file access.  Since we
 * have no uid or gid, for now require
 * file to exist and be publicly
 * readable/writable.
 * Note also, full path name must be
 * given as we have no login directory.
 */
validate_access(file, client, mode)
	char *file;
	struct udpaddr *client;
	int mode;
{
	struct stat stbuf;

	if (*file != '/')
		return (EACCESS);
	if (stat(file, &stbuf) < 0)
		return (errno == ENOENT ? ENOTFOUND : EACCESS);
	if (mode == RRQ) {
		if ((stbuf.st_mode&(S_IREAD >> 6)) == 0)
			return (EACCESS);
	} else {
		if ((stbuf.st_mode&(S_IWRITE >> 6)) == 0)
			return (EACCESS);
	}
	fd = open(file, mode == RRQ ? 0 : 1);
	if (fd < 0)
		return (errno + 100);
	return (0);
}

int timeout;

timer()
{
	timeout += TIMEOUT;
	if (timeout >= MAXTIMEOUT)
		exit(1);
	alarm(TIMEOUT);
	signal(SIGALRM, timer);
}

/*
 * Send the requested file.
 */
sendfile(pf)
	struct format *pf;
{
	register struct tftphdr *tp;
	register int block = 1, size, n;
	char buf[BUFFER];

	signal(SIGALRM, timer);
	tp = (struct tftphdr *)buf;
	do {
		if ((size = read(fd, tp->th_data, SEGSIZE)) < 0) {
			nak(errno + 100);
			break;
		}
		timeout = 0;
		do {
			alarm(TIMEOUT);
			tp->th_opcode = htons((u_short)DATA);
			tp->th_block = htons((u_short)block);
			if (write(f, buf, size + 4) != size + 4) {
				perror("send");
				goto out;
			}
			if (trace)
				tpacket("sent", buf, size+4);
			while ((n = read(f, buf, sizeof buf)) == 0)
				;
			if (n == -1) {
				if (errno == EINTR)
					continue;
				alarm(0);
				perror("send");
				goto out;
			}
			if (trace)
				tpacket("recv", buf, n);
			alarm(0);
			tp->th_opcode = ntohs(tp->th_opcode);
			tp->th_block = ntohs(tp->th_block);
			if (tp->th_opcode == ERROR)
				goto out;
		} while (tp->th_opcode != ACK || tp->th_block != block);
		block++;
	} while (size == SEGSIZE);
out:
	(void) close(fd);
}

/*
 * Receive a file.
 */
recvfile(pf)
	struct format *pf;
{
	register struct tftphdr *tp;
	register int block = 0, n, size;
	char buf[BUFFER];

	signal(SIGALRM, timer);
	tp = (struct tftphdr *)buf;
	do {
		timeout = 0;
		alarm(TIMEOUT);
		block++;
		do {
			tp->th_opcode = htons((u_short)ACK);
			tp->th_block = htons((u_short)block);
			if (write(f, buf, 4) != 4) {
				perror("ack");
				goto out;
			}
			if (trace)
				tpacket("sent", buf, sizeof buf);
			while ((n = read(f, buf, sizeof buf)) == 0)
				;
			if (n == -1) {
				if (errno == EINTR)
					continue;
				alarm(0);
				perror("receive");
				goto out;
			}
			if (trace)
				tpacket("recv", buf, n);
			alarm(0);
			tp->th_opcode = ntohs(tp->th_opcode);
			tp->th_block = ntohs(tp->th_block);
			if (tp->th_opcode == ERROR)
				goto out;
		} while (tp->th_opcode != DATA || tp->th_block != block);
		alarm(0);
		if ((size = write(fd, tp->th_data, n - 4)) < 0) {
			nak(errno + 100);
			goto out;
		}
	} while (size == SEGSIZE);
out:
	tp->th_opcode = htons((u_short)ACK);
	tp->th_block = htons((u_short)(block));
	(void) write(f, buf, 4);
	(void) close(fd);
}

struct errmsg {
	int	e_code;
	char	*e_msg;
} errmsgs[] = {
	{ EUNDEF,	"Undefined error code" },
	{ ENOTFOUND,	"File not found" },
	{ EACCESS,	"Access violation" },
	{ ENOSPACE,	"Disk full or allocation exceeded" },
	{ EBADOP,	"Illegal TFTP operation" },
	{ EBADID,	"Unknown transfer ID" },
	{ EEXISTS,	"File already exists" },
	{ ENOUSER,	"No such user" },
	{ -1,		0 }
};

/*
 * Send a nak packet (error message).
 * Error code passed in is one of the
 * standard TFTP codes, or a UNIX errno
 * offset by 100.
 */
nak(error)
	int error;
{
	register struct tftphdr *tp;
	int length;
	char buf[BUFFER];
	register struct errmsg *pe;
	extern char *sys_errlist[];

	tp = (struct tftphdr *)buf;
	tp->th_opcode = htons((u_short)ERROR);
	tp->th_code = htons((u_short)error);
	for (pe = errmsgs; pe->e_code >= 0; pe++)
		if (pe->e_code == error)
			break;
	if (pe->e_code < 0)
		pe->e_msg = sys_errlist[error - 100];
	strcpy(tp->th_msg, pe->e_msg);
	length = strlen(pe->e_msg);
	tp->th_msg[length] = '\0';
	length += 5;
	if (write(f, buf, length) != length)
		perror("nak");
}

/*
 * return pointer to '*cpp', advancing '*cpp' and checkig
 * for null termination of buffer 'n' bytes long.
 */

get_strings(cp, fnp, mdp, n)
	char *cp;
	char **fnp, **mdp;
	int n;
{
	*fnp = cp;
	while (*cp != '\0' && n--)
		cp++;
	if (n == 0)
		goto bad;
	cp++;
	n--;
	*mdp = cp;
	while (*cp != '\0' && n--)
		cp++;
	if (n == 0)
		goto bad;
	return;
bad:
	nak(EBADOP);
	exit(1);
}


dump(p, n)	/* print it out in hex */
	register unsigned char *p;
	register n;
{
	while (n--)
		printf("%02x ", *p++);
}


tpacket(s, tp, n)
	struct tftphdr *tp;
	int n;
{
	static char *opcodes[] =
	   { "#0", "RRQ", "WRQ", "DATA", "ACK", "ERROR" };
	register char *cp, *file;
	u_short op = ntohs(tp->th_opcode);
	char *index();

	if (op < RRQ || op > ERROR)
		printf("%s opcode=%x ", s, op);
	else
		printf("%s %s ", s, opcodes[op]);
	switch (op) {

	case RRQ:
	case WRQ:
		n -= 2;
		file = cp = tp->th_stuff;
		cp = index(cp, '\0');
		printf("<file=%s, mode=%s>\n", file, cp + 1);
		break;

	case DATA:
		printf("<block=%d, %d bytes>\n", ntohs(tp->th_block), n - 4);
		break;

	case ACK:
		printf("<block=%d>\n", ntohs(tp->th_block));
		break;

	case ERROR:
		printf("<code=%d, msg=%s>\n", ntohs(tp->th_code), tp->th_msg);
		break;
	}
}

