/*
 * TFTP User Program -- Protocol Machines
 */
#include <stdio.h>
#include <sys/param.h>
#include <sys/inet/in.h>
#include <sys/inet/udp_user.h>
#include <signal.h>
#include <errno.h>
#include <setjmp.h>
#include "tftp.h"

#define	ntohs(x)	(x)
#define	htons(x)	(x)
extern	int errno;
extern	struct	udpaddr	ua;
extern	char mode[];
int	f;
int	trace;
int	verbose;
int	connected;
char	buf[1024];
int	timeout;
jmp_buf	toplevel;
time_t	time();

timer()
{
	timeout += TIMEOUT;
	if (timeout >= MAXTIMEOUT) {
		printf("Transfer timed out.\n");
		longjmp(toplevel, -1);
	}
	signal(SIGALRM, timer);
	alarm(TIMEOUT);
}

/*
 * Send the requested file.
 */
sendfile(fd, name)
	int fd;
	char *name;
{
	register struct tftphdr *tp;
	register int block = 0, size, n, amount = 0;
	struct udpaddr oend;	/* other end */
	struct udpaddr *uap;
	time_t start = time(0), delta;

	uap = (struct udpaddr *)buf;
	tp = (struct tftphdr *)(buf + sizeof (struct udpaddr));
	size = makerequest(WRQ, name);
	size -= 12;
	timeout = 0;
	signal(SIGALRM, timer);
	oend = ua;
	do {
		if (block != 0) {
			size = read(fd, tp->th_data, SEGSIZE);
			if (size < 0) {
				nak(errno + 100);
				break;
			}
			tp->th_opcode = htons((u_short)DATA);
			tp->th_block = htons((u_short)block);
		}
		timeout = 0;
rexmt:
		if (trace)
			tpacket("sent", tp, size);
		*uap = oend;
		if (write(f, buf, size + 12) != size + 12) {
			perror("send");
			break;
		}
again:
		alarm(TIMEOUT);
		n = read(f, buf, sizeof buf);
		oend = *uap;
		if (n <= 0) {
			if (n == 0)
				goto again;
			perror ("read");
			if (errno == EINTR)
				goto rexmt;
			alarm(0);
			perror("receive");
			break;
		}
		alarm(0);
		if (trace)
			tpacket("received", tp, n);
		tp->th_opcode = ntohs(tp->th_opcode);
		tp->th_block = ntohs(tp->th_block);
		if (tp->th_opcode == ERROR) {
			printf("Error code %d: %s\n", tp->th_code,
				tp->th_msg);
			break;
		}
		if (tp->th_opcode != ACK || block != tp->th_block)
			goto again;
		if (block > 0)
			amount += size;
		block++;
	} while (size == SEGSIZE || block == 1);
	alarm(0);
	(void) close(fd);
	if (amount > 0) {
		delta = time(0) - start;
		printf("Sent %d bytes in %ld seconds.\n", amount, delta);
	}
}

/*
 * Receive a file.
 */
recvfile(fd, name)
	int fd;
	char *name;
{
	register struct tftphdr *tp;
	register int block = 1, n, size, amount = 0;
	struct udpaddr *uap;
	struct udpaddr oend;	/* other end address */
	time_t start = time(0), delta;

	uap = (struct udpaddr *)buf;
	tp = (struct tftphdr *)(buf + sizeof (struct udpaddr));
	size = makerequest(RRQ, name);
	size -= 12;
	timeout = 0;
	signal(SIGALRM, timer);
	oend = ua;
	goto rexmt;
	do {
		timeout = 0;
		alarm(TIMEOUT);
		tp->th_opcode = htons((u_short)ACK);
		tp->th_block = htons((u_short)(block));
		block++;
		size = 4;
rexmt:
		if (trace)
			tpacket("sent", tp, size);
		*uap = oend;
		if (write(f, buf,  size + 12) !=  size + 12) {
			perror("send");
			break;
		}
again:
		alarm(TIMEOUT);
		n = read(f, buf, sizeof buf);
		oend = *uap;
		if (n <= 0) {
			if (n == 0)
				goto again;
			if (errno == EINTR)
				goto rexmt;
			alarm(0);
			perror("receive");
			break;
		}
		alarm(0);
		if (trace)
			tpacket("received", tp, n);
		tp->th_opcode = ntohs(tp->th_opcode);
		tp->th_block = ntohs(tp->th_block);

		if (tp->th_opcode == ERROR) {
			printf("Error code %d: %s\n", tp->th_code,
				tp->th_msg);
			break;
		}
		if (tp->th_opcode != DATA || block != tp->th_block)
			goto again;
		size = write(fd, tp->th_data, n - 12);
		if (size < 0) {
			nak(errno + 100);
			break;
		}
		amount += size;
	} while (size == SEGSIZE);
	alarm(0);
	tp->th_opcode = htons((u_short)ACK);
	tp->th_block = htons((u_short)block);
	*uap = oend;
	write(f, buf, 4 + 8);
	(void) close(fd);
	if (amount > 0) {
		delta = time(0) - start;
		printf("Received %d bytes in %ld seconds.\n", amount, delta);
	}
}

makerequest(request, name)
	int request;
	char *name;
{
	register struct tftphdr *tp;
	int size;
	register char *cp;

	tp = (struct tftphdr *)(buf + sizeof (struct udpaddr));
	tp->th_opcode = htons((u_short)request);
	strcpy(tp->th_stuff, name);
	size = strlen(name);
	cp = tp->th_stuff + strlen(name);
	*cp++ = '\0';
	strcpy(cp, mode);
	cp += sizeof ("netascii") - 1;
	*cp++ = '\0';
	return (cp - buf);
}

struct errmsg {
	int	e_code;
	char	*e_msg;
} errmsgs[] = {
	{ EUNDEF,	"Undefined error code" },
	{ ENOTFOUND,	"File not found" },
	{ EACCESS,	"Access violation" },
	{ ENOSPACE,	"Disk full or allocation exceeded" },
	{ EBADOP,	"Illegal TFTP operation" },
	{ EBADID,	"Unknown transfer ID" },
	{ EEXISTS,	"File already exists" },
	{ ENOUSER,	"No such user" },
	{ -1,		0 }
};

/*
 * Send a nak packet (error message).
 * Error code passed in is one of the
 * standard TFTP codes, or a UNIX errno
 * offset by 100.
 */
nak(error)
	int error;
{
	register struct tftphdr *tp;
	int length;
	struct udpaddr *uap;
	register struct errmsg *pe;
	extern char *sys_errlist[];

	uap = (struct udpaddr *)buf;
	tp = (struct tftphdr *)(buf + sizeof (struct udpaddr));
	tp->th_opcode = htons((u_short)ERROR);
	tp->th_code = htons((u_short)error);
	for (pe = errmsgs; pe->e_code >= 0; pe++)
		if (pe->e_code == error)
			break;
	if (pe->e_code < 0)
		pe->e_msg = sys_errlist[error - 100];
	strcpy(tp->th_msg, pe->e_msg);
	length = strlen(pe->e_msg) + 4;
	if (trace)
		tpacket("sent", tp, length);
	length += sizeof (struct udpaddr);
	uap->port = ua.port;
	uap->host = ua.host;
	if (write(f, buf, length) != length)
		perror("nak");
}

tpacket(s, tp, n)
	struct tftphdr *tp;
	int n;
{
	static char *opcodes[] =
	   { "#0", "RRQ", "WRQ", "DATA", "ACK", "ERROR" };
	register char *cp, *file;
	u_short op = ntohs(tp->th_opcode);
	char *index();

	if (op < RRQ || op > ERROR)
		printf("%s opcode=%x ", s, op);
	else
		printf("%s %s ", s, opcodes[op]);
	switch (op) {

	case RRQ:
	case WRQ:
		n -= 2;
		file = cp = tp->th_stuff;
		cp = index(cp, '\0');
		printf("<file=%s, mode=%s>\n", file, cp + 1);
		break;

	case DATA:
		printf("<block=%d, %d bytes>\n", ntohs(tp->th_block), n - 4);
		break;

	case ACK:
		printf("<block=%d>\n", ntohs(tp->th_block));
		break;

	case ERROR:
		printf("<code=%d, msg=%s>\n", ntohs(tp->th_code), tp->th_msg);
		break;
	}
}
