/* showq:  display stuff about stream queues */

#include <stdio.h>
#include <nlist.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stream.h>
#include <sys/inode.h>

char	*kernel = "/unix";
int	coreflag;
char	*mem = "/dev/kmem";
int	mfd;
struct	inode	inode[NINODE];
struct	stdata	an_stdata;
struct	class	class[6];

#define	pf(m, chr)	(printf("%c", (m) ? (chr) : ' '))
#define	pfs(flag, str)	(printf("%s%s", (flag) ? str : "", (flag) ? " " : ""))
#define	IPADDR(x) (&inode[((int)(x) - n[SINODE].n_value)/sizeof (struct inode)])
struct	nlist	n[] = {
#define	SCLASS	0
	{ "_class" },
#define	SSTDATA	1
	{ "_stdata" },
#define	SINODE	2
	{ "_inode" },
#define	SQUEUE 3
	{ "_queue" },
	{ "_kqinit" },
	{ "_cmqinit" },
	{ "_enqinit" },
	{ "_etqinit" },
	{ "_ptqinit" },
	{ "_ttqinit" },
	{ "_ipqinit" },
	{ "_ipldqinit" },
	{ "_lbqinit" },
	{ "_tcpdqinit" },
	{ "_tcpl" },
	{ "_udpldqinit" },
	{ "_udpqinit" },
	{ "_pimqinit" },
	{ "_lrqinit" },
	{ "_lrldqinit" },
	{ 0 }
};
long	notes[NINODE];		/* have we seen the inode? */
char	*progname;
struct	class	cltab[6];
int	qcount = 0;		/* used to keep count of queues */


main(argc, argv)
	int argc;
	char *argv[];
{
	register struct inode *ip;
	register i, sum = 0;

	progname = argv[0];
	nlist(kernel, n);
	if (n[SCLASS].n_type == 0) {
		fprintf(stderr, "%s has no name list\n", kernel);
		exit(1);
	}
	if (argc > 1)
		kernel = argv[1];
	if (argc > 2) {
		mem = argv[2];
		coreflag++;
	}
	mfd = eopen(mem, 0);
	get(n[SCLASS].n_value, (char *)cltab, sizeof cltab);
	printf("blocks max: (");
	for (i = 0; i < 6; i++)
		printf("%d%c", cltab[i].max, (i==5) ? ')' : ',');
	printf("\n");
	printf("blocks cur: (");
	for (i = 0; i < 6; i++)
		printf("%d%c", cltab[i].nblks, (i==5) ? ')' : ',');
	printf("\n");
	get(n[SINODE].n_value, (char *)inode, sizeof inode);
	qcount = 0;
	for (ip = inode; ip < &inode[NINODE]; ip++) {
		if (ip->i_stream == 0)
			continue;
		if (note(ip))
			continue;
		pr_inode(ip);
		pr_stdata(ip->i_stream);
		pr_stream(ip->i_stream);
	}
	printf("%3d queues in use\n", qcount);
	qcount = 0;
	fqcount();	/* count free queues */
	printf("%3d queues free\n", qcount);
}


pr_inode(ip)		/* print out the interesting parts */
	struct inode *ip;
{
	if (ip->i_mode & IFCHR)
		p("(%d, %d) ", major(ip->i_rdev), minor(ip->i_rdev));
	p("ino %d, ", ip->i_number);
	p("count = %d, ", ip->i_count);
}

pr_stdata(loc)		/* get and print stream head data */
	long loc;
{
	struct stdata s;

	get(loc, (char *)&s, sizeof s);
	p(" pgrp = %d ", s.pgrp);
	pfs(s.flag & IOCWAIT, "IOCWAIT");
	pfs(s.flag & RSLEEP, "RSLEEP");
	pfs(s.flag & WSLEEP, "WSLEEP");
	pfs(s.flag & HUNGUP, "HUNGUP");
	pfs(s.flag & RSEL, "RSEL");
	pfs(s.flag & WSEL, "WSEL");
	pfs(s.flag & EXECL, "EXECL");
	pfs(s.flag & STWOPEN, "STWOPEN");
	p("\n");
}

pr_stream(loc)		/* display a stream of queues */
	long loc;
{
	struct stdata s;
	struct queue q, q2;
	long qsize = sizeof q;
	int nq, flip = 0;
	struct inode *ip;

	get(loc, (char *)&s, sizeof s);
	for (nq = (long)s.wrq; nq; nq = (long)q.next) {
		get(nq, (char *)&q, sizeof q);
		get(q.flag & QREADR ? nq + qsize : nq - qsize, (char *)&q2, sizeof q2);
		if (!flip && q.flag & QREADR) {
			p("\t\t<>\n");
			flip++;
		}
		if (q.flag & QREADR) {
			pr_queue(nq, &q);
			pr_queue(nq+sizeof q, &q2);
		} else {
			pr_queue(nq-qsize, &q2);
			pr_queue(nq, &q);
		}
		qcount += 2;
	}
	if (is_stdata(q.ptr)) {
		get((long)q.ptr, (char *)&s, sizeof s);
		ip = IPADDR(s.inode);
		pr_inode(ip);
		pr_stdata(q.ptr);
		note(ip);
	}
	p("\n");
}



pr_queue(loc, q)
	register struct queue *q;
{
	p("    %s #", q->flag & QREADR ? "RQ" : "WQ");
	p("%x:", loc);
	p("%5d ", q->count);
	if (q->flag & QREADR)
		pr_ld(q->qinfo);
	else
		p("   ");
	pfs(q->flag & QENAB, "enabled");
	pfs(q->flag & QWANTR, "wantr");
	pfs(q->flag & QWANTW, "wantw");
	pfs(q->flag & QFULL, "full");
	pfs(q->flag & QREADR, "read");
	pfs(q->flag & QUSE, "use");
	pfs(q->flag & QNOENB, "noenable");
	pfs(q->flag & QDELIM, "delim.");
	pfs(q->flag & QBIGB, "bigb");
	p("\n");
}


p(fmt, a, b, c, d, e, f, g, h)
	char *fmt;
{
	printf(fmt, a, b, c, d, e, g, h);
}

get(loc, place, size)		/* get a structure from mem */
	char *place;
{
	loc &= ~0xC0000000;
	if (coreflag)
		loc += ctob(USIZE);
	lseek(mfd, loc, 0);
	read(mfd, place, size);
}

eopen(name, mode)
	char *name;
{
	register fd;

	if ((fd = open(name, mode)) == -1) {
		fprintf(stderr, "%s: can't open %s\n", progname, name);
		exit(1);
	}
	return fd;
}

note(l)		/* have I seen this inode before? */
	long l;
{
	register long *p;

	for (p = notes; p < &notes[NINODE]; p++)
		if (*p == 0) {
			*p = l;
			return 0;
		} else if (*p == l)
			return 1;
}

is_stdata(loc)		/* does loc fall in the stdata table? */
	long loc;
{
	long b, e;

	b = n[SSTDATA].n_value;
	e = b + (NSTREAMS * sizeof (struct stdata));
	return (b <= loc && loc <= e);
}

pr_ld(qinfo)		/* print name of line discpline */
	struct qinit *qinfo;
{
	register struct nlist *np;

	for (np = n; np->n_name; np++)
		if (np->n_value == (int)qinfo) {
			p("%s; ", np->n_name+1);
			return;
		}
	p("??; ");
}

struct	queue	xqueue[NQUEUES];

fqcount()	/* count free queues */
{
	register struct queue *qp;

	get(n[SQUEUE].n_value, xqueue, sizeof xqueue);
	for (qp = xqueue; qp < &xqueue[NQUEUES]; qp++)
		if (!(qp->flag & QUSE))
			qcount++;
}
