/*
    Wrapper DLL for SNAP interface for K Video Accelerator
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <os2.h>

#include "snap/graphics.h"

#include "kva.h"

#define SNAP_FOURCC     ( gaVideoYUV422 | gaVideoYUYV )

static GA_devCtx       *m_dc = NULL;
static GA_initFuncs     m_initFuncs = { 0, };
static GA_modeInfo      m_modeInfo = { 0, };
static GA_videoInf     *m_videoInf = NULL;
static GA_videoFuncs    m_videoFuncs = { 0, };
static GA_bufferFuncs   m_bufmgr = { 0 , };
static REF2D_driver    *m_ref2d = NULL;

static LONG m_lBrightness;
static LONG m_lContrast;
static LONG m_lSaturation;
static LONG m_lHue;
static LONG m_lGamma = 0;

APIRET APIENTRY swLoadDriver( VOID );
APIRET APIENTRY swUnloadDriver( VOID );
APIRET APIENTRY swSetVideoOutput( PVOID pVideoBuf, LONG srcX, LONG srcY, LONG srcCX, LONG srcCY, LONG dstX, LONG dstY, LONG dstCX, LONG dstCY );
APIRET APIENTRY swMoveVideoOutput(LONG srcX, LONG srcY, LONG srcCX, LONG srcCY, LONG dstX, LONG dstY, LONG dstCX, LONG dstCY );
APIRET APIENTRY swDisableVideoOutput( VOID );
APIRET APIENTRY swSetDstVideoColorKey( ULONG ulKeyColor );
APIRET APIENTRY swAllocVideoBuffers( PPVOID ppVideoBuf, LONG cx, LONG cy, LONG nBufs );
APIRET APIENTRY swFreeVideoBuffers( PVOID pVideoBuf );
APIRET APIENTRY swLockBuffer( PVOID pVideoBuf, PPVOID ppSurface, PULONG pulBPL );
APIRET APIENTRY swUnlockBuffer( PVOID pVideoBuf );
APIRET APIENTRY swCaps( PKVACAPS pkvac );
APIRET APIENTRY swQueryVideoBrightness( PULONG pulValue );
APIRET APIENTRY swSetVideoBrightness( PULONG pulValue );
APIRET APIENTRY swQueryVideoContrast( PULONG pulValue );
APIRET APIENTRY swSetVideoContrast( PULONG pulValue );
APIRET APIENTRY swQueryVideoSaturation( PULONG pulValue );
APIRET APIENTRY swSetVideoSaturation( PULONG pulValue );
APIRET APIENTRY swQueryVideoHue( PULONG pulValue );
APIRET APIENTRY swSetVideoHue( PULONG pulValue );
APIRET APIENTRY swQueryVideoGammaCorrect( PULONG pulValue );
APIRET APIENTRY swSetVideoGammaCorrect( PULONG pulValue );

APIRET APIENTRY swLoadDriver( VOID )
{
    m_dc = NULL;
    memset( &m_initFuncs, 0, sizeof( m_initFuncs ));
    memset( &m_modeInfo, 0, sizeof( m_modeInfo ));
    m_videoInf = NULL;
    memset( &m_videoFuncs, 0, sizeof( m_videoFuncs ));
    memset( &m_bufmgr, 0, sizeof( m_bufmgr ));
    m_ref2d = NULL;

    m_dc = GA_loadDriver( 0, false );
    if( !m_dc )
        return -1;

    m_initFuncs.dwSize = sizeof( m_initFuncs );
    if( !GA_queryFunctions( m_dc, GA_GET_INITFUNCS, &m_initFuncs ))
        goto exit_error;

    m_modeInfo.dwSize = sizeof( m_modeInfo );
    m_initFuncs.GetCurrentVideoModeInfo( &m_modeInfo );

    m_ref2d = GA_getCurrentRef2d( 0 );
    if( !m_ref2d )
        goto exit_error;

    m_videoFuncs.dwSize = sizeof( m_videoFuncs );
    if( !REF2D_queryFunctions( m_ref2d, GA_GET_VIDEOFUNCS, &m_videoFuncs ))
        goto exit_error;

    m_bufmgr.dwSize = sizeof( m_bufmgr );
    if( !REF2D_queryFunctions( m_ref2d, GA_GET_BUFFERFUNCS, &m_bufmgr ))
        goto exit_error;

    m_videoInf = m_modeInfo.VideoWindows[ 0 ];

    m_lBrightness = m_videoInf->VideoBrightnessDefault;
    m_lContrast = m_videoInf->VideoContrastDefault;
    m_lSaturation = m_videoInf->VideoSaturationDefault;
    m_lHue = m_videoInf->VideoHueDefault;
    m_lGamma = 0;

    return 0;

exit_error :
    GA_unloadDriver( m_dc );

    return -1;
}

APIRET APIENTRY swUnloadDriver( VOID )
{
    m_videoFuncs.DisableVideoOutput( 0 );

    GA_unloadDriver( m_dc );

    return 0;
}

APIRET APIENTRY swSetVideoOutput( PVOID pVideoBuf,
                                  LONG srcX, LONG srcY, LONG srcCX, LONG srcCY,
                                  LONG dstX, LONG dstY, LONG dstCX, LONG dstCY )
{
    ULONG ulVOFlags = SNAP_FOURCC;

    if( m_videoInf->VideoOutputFlags & gaVideoColorKeyDstSingle )
        ulVOFlags |= gaVideoColorKeyDstSingle;

    if( m_videoInf->VideoOutputFlags & gaVideoXInterp )
        ulVOFlags |= gaVideoXInterp;

    if( m_videoInf->VideoOutputFlags & gaVideoYInterp )
        ulVOFlags |= gaVideoYInterp;

    return !m_videoFuncs.SetVideoOutput( 0, 0, pVideoBuf,
                                         srcX, srcY, srcCX, srcCY,
                                         dstX, m_modeInfo.YResolution - dstY, dstCX, dstCY,
                                         ulVOFlags );
}

APIRET APIENTRY swMoveVideoOutput( LONG srcX, LONG srcY, LONG srcCX, LONG srcCY,
                                   LONG dstX, LONG dstY, LONG dstCX, LONG dstCY )
{
    m_videoFuncs.MoveVideoOutput( 0,
                                  srcX, srcY, srcCX, srcCY,
                                  dstX, m_modeInfo.YResolution - dstY, dstCX, dstCY,
                                  0 );

    return 0;
}

APIRET APIENTRY swDisableVideoOutput( VOID )
{
    m_videoFuncs.DisableVideoOutput( 0 );

    return 0;
}

APIRET APIENTRY swSetDstVideoColorKey( ULONG ulKeyColor )
{
    m_videoFuncs.SetDstVideoColorKey( 0, ulKeyColor, ulKeyColor );

    return 0;
}

APIRET APIENTRY swAllocVideoBuffers( PPVOID ppVideoBuf, LONG cx, LONG cy, LONG nBufs )
{
    *ppVideoBuf = m_videoFuncs.AllocVideoBuffers( cx, cy, SNAP_FOURCC, nBufs );
    if( !*ppVideoBuf )
        return -1;

    return 0;
}

APIRET APIENTRY swFreeVideoBuffers( PVOID pVideoBuf )
{
    m_videoFuncs.FreeVideoBuffers( pVideoBuf );

    return 0;
}

APIRET APIENTRY swLockBuffer( PVOID pVideoBuf, PPVOID ppBuffer, PULONG pulBPL )
{
    m_bufmgr.LockBuffer( pVideoBuf );

    *ppBuffer = (( GA_buf * )pVideoBuf )->Surface;
    *pulBPL = (( GA_buf * )pVideoBuf )->Stride;

    return 0;
}

APIRET APIENTRY swUnlockBuffer( PVOID pVideoBuf )
{
    m_bufmgr.UnlockBuffer( pVideoBuf );

    return 0;
}

APIRET APIENTRY swCaps( PKVACAPS pkvac )
{
    pkvac->ulMode = KVAM_SNAP;
    pkvac->ulDepth = m_modeInfo.BitsPerPixel;
    pkvac->cxScreen = m_modeInfo.XResolution;
    pkvac->cyScreen = m_modeInfo.YResolution;
    pkvac->fccScreen = 0;   // todo

    return 0;
}

#define ATTR_FROM_SNAP( value, min, max )   ((( value ) - ( min )) * 255 / (( max ) - ( min )))
#define ATTR_TO_SNAP( value, min, max )     ((( value ) + 1 ) * (( max ) - ( min )) / 255 + ( min ))

APIRET APIENTRY swQueryVideoBrightness( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoBrightness )
        return -1;

    *pulValue = ATTR_FROM_SNAP( m_lBrightness, m_videoInf->VideoBrightnessMin, m_videoInf->VideoBrightnessMax );

    return 0;
}

APIRET APIENTRY swSetVideoBrightness( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoBrightness )
        return -1;

    if( *pulValue == ( ULONG )-1 )
        *pulValue = ATTR_FROM_SNAP( m_videoInf->VideoBrightnessDefault, m_videoInf->VideoBrightnessMin, m_videoInf->VideoBrightnessMax );

    if(( LONG )*pulValue < 0 )
        *pulValue = 0;

    if( *pulValue > 255 )
        *pulValue = 255;

    m_lBrightness = ATTR_TO_SNAP( *pulValue, m_videoInf->VideoBrightnessMin, m_videoInf->VideoBrightnessMax );

    m_videoFuncs.SetVideoBrightness( 0, m_lBrightness );

    return 0;
}

APIRET APIENTRY swQueryVideoContrast( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoContrast )
        return -1;

    *pulValue = ATTR_FROM_SNAP( m_lContrast, m_videoInf->VideoContrastMin, m_videoInf->VideoContrastMax );

    return 0;
}

APIRET APIENTRY swSetVideoContrast( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoContrast )
        return -1;

    if( *pulValue == ( ULONG )-1 )
        *pulValue = ATTR_FROM_SNAP( m_videoInf->VideoContrastDefault, m_videoInf->VideoContrastMin, m_videoInf->VideoContrastMax );

    if(( LONG )*pulValue < 0 )
        *pulValue = 0;

    if( *pulValue > 255 )
        *pulValue = 255;

    m_lContrast = ATTR_TO_SNAP( *pulValue, m_videoInf->VideoContrastMin, m_videoInf->VideoContrastMax );

    m_videoFuncs.SetVideoContrast( 0, m_lContrast );

    return 0;
}

APIRET APIENTRY swQueryVideoSaturation( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoSaturation )
        return -1;

    *pulValue = ATTR_FROM_SNAP( m_lSaturation, m_videoInf->VideoSaturationMin, m_videoInf->VideoSaturationMax );

    return 0;
}

APIRET APIENTRY swSetVideoSaturation( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoSaturation )
        return -1;

    if( *pulValue == ( ULONG )-1 )
        *pulValue = ATTR_FROM_SNAP( m_videoInf->VideoSaturationDefault, m_videoInf->VideoSaturationMin, m_videoInf->VideoSaturationMax );

    if(( LONG )*pulValue < 0 )
        *pulValue = 0;

    if( *pulValue > 255 )
        *pulValue = 255;

    m_lSaturation = ATTR_TO_SNAP( *pulValue, m_videoInf->VideoSaturationMin, m_videoInf->VideoSaturationMax );

    m_videoFuncs.SetVideoSaturation( 0, m_lSaturation );

    return 0;
}

APIRET APIENTRY swQueryVideoHue( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoHue )
        return -1;

    *pulValue = ATTR_FROM_SNAP( m_lHue, m_videoInf->VideoHueMin, m_videoInf->VideoHueMax );

    return 0;
}

APIRET APIENTRY swSetVideoHue( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoHue )
        return -1;

    if( *pulValue == ( ULONG )-1 )
        *pulValue = ATTR_FROM_SNAP( m_videoInf->VideoHueDefault, m_videoInf->VideoHueMin, m_videoInf->VideoHueMax );

    if(( LONG )*pulValue < 0 )
        *pulValue = 0;

    if( *pulValue > 255 )
        *pulValue = 255;

    m_lHue = ATTR_TO_SNAP( *pulValue, m_videoInf->VideoHueMin, m_videoInf->VideoHueMax );

    m_videoFuncs.SetVideoHue( 0, m_lHue );

    return 0;
}

/* Gamma range : 0.01 - 2.55, default = 0 */

APIRET APIENTRY swQueryVideoGammaCorrect( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoGammaCorrect )
        return -1;

    *pulValue = m_lGamma;

    return 0;
}

APIRET APIENTRY swSetVideoGammaCorrect( PULONG pulValue )
{
    if( !m_videoFuncs.SetVideoGammaCorrect )
        return -1;

    if( *pulValue == ( ULONG )-1 )
        *pulValue = 0;

    if(( LONG )*pulValue < 0 )
        *pulValue = 0;

    if( *pulValue > 255 )
        *pulValue = 255;

    m_lGamma = *pulValue;

    m_videoFuncs.SetVideoGammaCorrect( 0, m_lGamma );

    return 0;
}


