/*
    KMP : Image library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This file is part of KMP.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Changes :
        KO Myung-Hun <komh@chollian.net> 2007/02/25
            - Added imgQueryAttr(), imgSetAttr() and imgResetAttr()
            - Changed imgInit() to initialize attribute
            - Reset attribute on imgDone()

        KO Myung-Hun <komh@chollian.net> 2007/03/03
            - Changed the prototype of imgCreateYUV() to use m_kvas instead
              of parameters
            - Allocate memory for YUV buffer more accurately in imgCreateYUV()
            - Changed imgConvert() to use m_kvas instead of pSrcBmp->w and
              pSrcBmp->h

        KO Myung-Hun <komh@chollian.net> 2007/03/10
            - Use swscale library instead of yuv2rgb library

        KO Myung-Hun <komh@chollian.net> 2007/06/11
            - Fixed wrong color format for 15 bpp and 16 bpp mode

        KO Myung-Hun <komh@chollian.net> 2007/12/23
            - Added support of SNAP
*/

#include <os2.h>

#include <fourcc.h>

#include <stdlib.h>
#include <types.h>

#define HAVE_AV_CONFIG_H
#include "dsputil.h"
#include "swscale.h"
#include "rgb2rgb.h"

#include "kva.h"

#include "kmp_img.h"

int sws_flags;

static KVASETUP m_kvas = { 0 };

static struct SwsContext *m_sws;

static APIRET imgConvertInit( LONG cx, LONG cy, FOURCC *pfcc )
{
    KVACAPS kvac;
    int     dst_pix_fmt;

    mm_flags = mm_support();
    sws_flags = SWS_FAST_BILINEAR | SWS_PRINT_INFO;

    if( mm_flags & MM_MMX )
        sws_flags |= SWS_CPU_CAPS_MMX;

    if( mm_flags & MM_MMXEXT )
        sws_flags |= SWS_CPU_CAPS_MMX2;

    if( mm_flags & MM_3DNOW )
        sws_flags |= SWS_CPU_CAPS_3DNOW;

    sws_rgb2rgb_init( sws_flags );

    kvaCaps( &kvac );

    switch( kvac.ulMode )
    {
        case KVAM_WO :
        case KVAM_SNAP :
            *pfcc = FOURCC_Y422;
            dst_pix_fmt = PIX_FMT_YUYV422;
            break;

        case KVAM_DIVE :
        default :
            switch( kvac.fccScreen )
            {
                case FOURCC_BGR4 :
                case FOURCC_BGR3 :
                case FOURCC_LUT8 :  // maybe best T.T
                    *pfcc = FOURCC_BGR3;
                    dst_pix_fmt = PIX_FMT_BGR24;
                    break;

                case FOURCC_R565 :
                    *pfcc = FOURCC_R565;
                    dst_pix_fmt = PIX_FMT_BGR565;
                    break;

                case FOURCC_R555 :
                    *pfcc = FOURCC_R555;
                    dst_pix_fmt = PIX_FMT_BGR555;
                    break;
            }
            break;
    }

    m_sws = sws_getContext( cx, cy, PIX_FMT_YUV420P, cx, cy, dst_pix_fmt,
                            sws_flags, NULL, NULL, NULL );

    return 0;
}

static VOID imgConvertDone( VOID )
{
    sws_freeContext( m_sws );
}

static VOID imgConvert( PVOID pDstBuffer, ULONG ulDstBPL, YUV *pSrcBmp )
{
    sws_scale( m_sws, pSrcBmp->data, pSrcBmp->linesize, 0, m_kvas.szlSrcSize.cy,
               &pDstBuffer, &ulDstBPL );

    emms_c();
}

VOID imgInit( ULONG kvaMode, HWND hwnd, ULONG ulKeyColor, LONG cx, LONG cy,
              ULONG ulRatio, PULONG pulAttrValue )
{
    int i;

    kvaInit( kvaMode, hwnd, ulKeyColor );

    m_kvas.ulLength = sizeof( KVASETUP );
    m_kvas.szlSrcSize.cx = cx;
    m_kvas.szlSrcSize.cy = cy;
    m_kvas.rclSrcRect.xLeft = 0;
    m_kvas.rclSrcRect.yTop = 0;
    m_kvas.rclSrcRect.xRight = cx;
    m_kvas.rclSrcRect.yBottom = cy;
    m_kvas.ulRatio = ulRatio;

    imgConvertInit( cx, cy, &m_kvas.fccSrcColor );

    kvaSetup( &m_kvas );

    for( i = 0; i < KVAA_LAST; i++ )
        kvaSetAttr( i, &pulAttrValue[ i ] );

}

VOID imgDone( VOID )
{
    imgConvertDone();

    imgResetAttr();

    kvaDone();
}

YUV *imgCreateYUV( VOID )
{
    YUV  *p;
    LONG lSize = m_kvas.szlSrcSize.cx * m_kvas.szlSrcSize.cy;

    p = av_malloc( sizeof( YUV ));
    if( !p )
        return NULL;

    p->p = av_malloc(( lSize * 3 ) / 2 );

    if( !p->p )
    {
        av_free( p );

        return NULL;
    }

    p->linesize[ 0 ] = m_kvas.szlSrcSize.cx;
    p->linesize[ 1 ] = m_kvas.szlSrcSize.cx / 2;
    p->linesize[ 2 ] = m_kvas.szlSrcSize.cx / 2;
    p->linesize[ 3 ] = 0;

    p->data[ 0 ] = p->p;
    p->data[ 1 ] = p->data[ 0 ] + lSize;
    p->data[ 2 ] = p->data[ 1 ] + lSize / 4;
    p->data[ 3 ] = NULL;

    return p;
}

VOID imgFreeYUV( YUV *p )
{
    if( p )
    {
        if( p->p )
            av_free( p->p );

        av_free( p );
    }
}

VOID imgDisplayYUV( YUV *bmp )
{
    PVOID       pBuffer;
    ULONG       ulBPL;

    if( !kvaLockBuffer( &pBuffer, &ulBPL ))
    {
        imgConvert( pBuffer, ulBPL, bmp );

        kvaUnlockBuffer();
    }
}

VOID imgClearRect( PRECTL prcl )
{
    kvaClearRect( prcl );
}

VOID imgSetAspectRatio( ULONG ulRatio )
{
    m_kvas.ulRatio = ulRatio;

    kvaSetup( &m_kvas );
}

APIRET imgQueryAttr( ULONG ulAttr, PULONG pulValue )
{
    return kvaQueryAttr( ulAttr, pulValue );
}

APIRET imgSetAttr( ULONG ulAttr, PULONG pulValue )
{
    return kvaSetAttr( ulAttr, pulValue );
}

VOID imgResetAttr( VOID )
{
    kvaResetAttr();
}

