/*
    KMP : OSD library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This file is part of KMP.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define INCL_WIN
#define INCL_GPI
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "kmp_osd.h"

static BOOL     m_fErase;
static HDC      m_hdc;
static HPS      m_hps;
static HBITMAP  m_hbm;
static RECTL    m_rclCurOSD;
static RECTL    m_rclInvalidOSD;
static LONG     m_lHoriFontRes;
static LONG     m_lVertFontRes;
static FIXED    m_fxPointSize;
static CHAR     m_szFaceName[ FACESIZE + 1 ];
static CHAR     m_szText[ 256 ];

static VOID obmDelete( VOID );

static VOID obmInit( HAB hab, PCSZ pcszFontNameSize )
{
    HPS     hps;
    HDC     hdc;
    char   *facename;
    SIZEL   sizl;

    hps = WinGetScreenPS( HWND_DESKTOP );
    hdc = GpiQueryDevice( hps );
    DevQueryCaps( hdc, CAPS_HORIZONTAL_FONT_RES, 1, &m_lHoriFontRes );
    DevQueryCaps( hdc, CAPS_VERTICAL_FONT_RES, 1, &m_lVertFontRes );
    WinReleasePS( hps );

    m_fxPointSize = MAKEFIXED( strtol( pcszFontNameSize, &facename, 0 ), 0 );
    strcpy( m_szFaceName, facename + 1 );

    m_hdc = DevOpenDC( hab, OD_MEMORY, "*", 0L, NULL, NULLHANDLE);

    sizl.cx = 0;
    sizl.cy = 0;
    m_hps = GpiCreatePS( hab, m_hdc, &sizl,
                         PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC );
}

static VOID obmDone( VOID )
{
    obmDelete();

    GpiDestroyPS( m_hps );

    DevCloseDC( m_hdc );
}

#define SHADOW_DISTANCE 1

static VOID outliner( HPS hps, INT x, INT y, PSZ szMsg,
                      LONG color, LONG colorOutline, LONG colorShadow, BOOL f3d )
{
    static LONG outlineColor[] = { 0x000000, };
    INT         outlineN = sizeof( outlineColor ) / sizeof( outlineColor[ 0 ] );
    INT         i, j;
    INT         len = strlen( szMsg );
    POINTL      ptl;
    LONG        oldColor;

    oldColor = GpiQueryColor( hps );

    outlineColor[ 0 ] = colorOutline;

    if( f3d )
    {
        GpiSetColor( hps, colorShadow );
        ptl.x = x + ( outlineN + SHADOW_DISTANCE );
        ptl.y = y - ( outlineN + SHADOW_DISTANCE );
        GpiCharStringAt( hps, &ptl, len, szMsg );
    }

    for( i = outlineN; i > 0; i-- )
    {

        GpiSetColor( hps, outlineColor[ i - 1 ]);

        for( j = -i; j <= i; j++ )
        {
            ptl.x = x - i;
            ptl.y = y + j;
            GpiCharStringAt( hps, &ptl, len, szMsg );

            ptl.x = x + i;
            ptl.y = y + j;
            GpiCharStringAt( hps, &ptl, len, szMsg );
        }

        for( j = -( i - 1 ); j <= ( i - 1 ); j++ )
        {
            ptl.x = x + j;
            ptl.y = y + i;
            GpiCharStringAt( hps, &ptl, len, szMsg );

            ptl.x = x + j;
            ptl.y = y - i;
            GpiCharStringAt( hps, &ptl, len, szMsg );
        }
    }

    GpiSetColor( hps, color );
    ptl.x = x;
    ptl.y = y;
    GpiCharStringAt( hps, &ptl, len, szMsg );

    GpiSetColor( hps, oldColor );
}

static VOID findFont( HPS hps, FIXED fxPointSize, PCSZ pcszFacename, PFATTRS pfat, PSIZEF psizf )
{
    LONG         cFonts = 0;
    PFONTMETRICS pfm;
    int          i;

    memset( pfat, 0, sizeof( FATTRS ));
    pfat->usRecordLength = sizeof( FATTRS );
    strcpy( pfat->szFacename, pcszFacename );

    memset( psizf, 0, sizeof( SIZEF ));

    cFonts = GpiQueryFonts( hps, QF_PUBLIC | QF_PRIVATE, pcszFacename, &cFonts,
                            sizeof( FONTMETRICS ), NULL );

    pfm = malloc( sizeof( FONTMETRICS ) * cFonts );

    GpiQueryFonts( hps, QF_PUBLIC | QF_PRIVATE, pcszFacename, &cFonts,
                   sizeof( FONTMETRICS ), pfm );

    for( i = 0; i < cFonts; i++ )
    {
        if( pfm[ i ].fsDefn & FM_DEFN_OUTLINE ) // outline font
            break;

        if(( pfm[ i ].sXDeviceRes == m_lHoriFontRes ) &&
           ( pfm[ i ].sYDeviceRes == m_lVertFontRes ) &&
           ( pfm[ i ].sNominalPointSize / 10 == FIXEDINT( fxPointSize )))
           break;
    }

    if( i < cFonts )    // found wanted font
    {
        if( pfm[ i ].fsDefn & FM_DEFN_OUTLINE ) // outline font
        {
            psizf->cx = (( fxPointSize / 72 * m_lHoriFontRes ) + 0x10000L ) & -0x20000L;
            psizf->cy = fxPointSize / 72 * m_lVertFontRes;
        }
        else    // raster font
        {
            pfat->lMaxBaselineExt = pfm[ i ].lMaxBaselineExt;
            pfat->lAveCharWidth = pfm[ i ].lAveCharWidth;

            psizf->cx = pfm[ i ].lEmInc;
            psizf->cy = pfm[ i ].lEmHeight;
        }
    }

    free( pfm );
}

static VOID obmCreate( LONG x, LONG y,
                       ULONG ulColor, ULONG ulColorOutline, ULONG ulColorShadow,
                       BOOL f3d )
{
    FATTRS            fat;
    SIZEF             sizf;
    BITMAPINFOHEADER2 bmih2;
    FONTMETRICS       fm;
    POINTL            aptl[ TXTBOX_COUNT ];
    int               y1;
    int               bmp_width, bmp_height;

    if( m_hbm )
        return;

    findFont( m_hps, m_fxPointSize, m_szFaceName, &fat, &sizf );
    fat.fsFontUse = FATTR_FONTUSE_NOMIX;

    GpiCreateLogFont( m_hps, NULL, 1, &fat );
    GpiSetCharSet( m_hps, 1 );

    GpiSetCharBox( m_hps, &sizf );

    GpiQueryTextBox( m_hps, strlen( m_szText ), m_szText, TXTBOX_COUNT, aptl );
    bmp_width = aptl[ TXTBOX_CONCAT ].x - aptl[ TXTBOX_BOTTOMLEFT ].x + 3; // 3 for outline and shadow

    GpiQueryFontMetrics( m_hps, sizeof( FONTMETRICS ), &fm );
    bmp_height = fm.lMaxBaselineExt + fm.lExternalLeading + 3; // 3 for outline and shadow

    memset( &bmih2, 0, sizeof( BITMAPINFOHEADER2 ));

    bmih2.cbFix = sizeof( BITMAPINFOHEADER2 );
    bmih2.cx = bmp_width;
    bmih2.cy = bmp_height;
    bmih2.cPlanes = 1;
    bmih2.cBitCount = 24;

    m_hbm = GpiCreateBitmap( m_hps, &bmih2, 0, NULL, NULL );

    GpiSetBitmap( m_hps, m_hbm );

    GpiCreateLogColorTable( m_hps, 0, LCOLF_RGB, 0, 0, NULL );

    y1 = fm.lMaxDescender + 2; // 2 for outline and shadow

                  // 1 for left shadow
    outliner( m_hps, 1, y1, m_szText,
              ulColor, ulColorOutline, ulColorShadow, f3d );

    m_rclCurOSD.xLeft = x;
    m_rclCurOSD.xRight = m_rclCurOSD.xLeft + bmp_width;
    m_rclCurOSD.yBottom = y;
    m_rclCurOSD.yTop = m_rclCurOSD.yBottom + bmp_height;
}

static VOID obmDelete( VOID )
{
    GpiSetBitmap( m_hps, NULLHANDLE );
    GpiDeleteBitmap( m_hbm );
    m_hbm = NULLHANDLE;
}

APIRET osdInit( HAB hab, PCSZ pcszFontNameSize )
{
    obmInit( hab, pcszFontNameSize );

    return 0;
}

VOID osdDone( VOID )
{
    obmDone();
}

VOID osdDisplay( HPS hps, LONG x, LONG y,
                 ULONG ulColor, ULONG ulColorOutline, ULONG ulColorShadow,
                 BOOL f3d )
{
    POINTL aptl[ 3 ];

    if( !m_szText[ 0 ] )
        return;

    if( !m_hbm )
        obmCreate( x, y, ulColor, ulColorOutline, ulColorShadow, f3d );

    // target
    aptl[ 0 ].x = m_rclCurOSD.xLeft;
    aptl[ 0 ].y = m_rclCurOSD.yBottom;
    aptl[ 1 ].x = m_rclCurOSD.xRight;
    aptl[ 1 ].y = m_rclCurOSD.yTop;

    // source
    aptl[ 2 ].x = 0;
    aptl[ 2 ].y = 0;

    GpiBitBlt( hps, m_hps, 3, aptl, ROP_SRCPAINT, BBO_IGNORE );
}

VOID osdErase( HPS hps, ULONG ulColor )
{
    if( m_fErase )
    {
        GpiCreateLogColorTable( hps, 0, LCOLF_RGB, 0, 0, NULL );
        WinFillRect( hps, &m_rclInvalidOSD, ulColor );

        m_fErase = FALSE;
    }
}

FIXED osdQueryFontSize( VOID )
{
    return m_fxPointSize;
}

FIXED osdSetFontSize( FIXED fxPointSize )
{
    m_fxPointSize = fxPointSize;

    if( FIXEDINT( m_fxPointSize ) < 6 )
        m_fxPointSize = MAKEFIXED( 6, 0 );
    if( FIXEDINT( m_fxPointSize ) > 144 )
        m_fxPointSize = MAKEFIXED( 144, 0 );

    osdInvalidate();

    return m_fxPointSize;
}

PSZ osdQueryFontName( PSZ pszFontName, ULONG ulLen )
{
    strncpy( pszFontName, m_szFaceName, ulLen );
    pszFontName[ ulLen - 1 ] = 0;

    return pszFontName;
}

VOID osdSetFontName( PCSZ pcszFontName )
{
    strncpy( m_szFaceName, pcszFontName, sizeof( m_szFaceName ));
    m_szFaceName[ sizeof( m_szFaceName ) - 1 ] = 0;

    osdInvalidate();
}

VOID osdInvalidate( VOID )
{
    obmDelete();

    m_rclInvalidOSD = m_rclCurOSD;
    m_fErase = TRUE;

    m_szText[ 0 ] = 0;
}

VOID osdSetText( PCSZ pcszText )
{
    osdInvalidate();

    strncpy( m_szText, pcszText, sizeof( m_szText ));
    m_szText[ sizeof( m_szText ) - 1 ] = 0;
}

BOOL osdOn( VOID )
{
    return m_szText[ 0 ];
}
