/*
    Direct Audio Interface library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>
    Copyright (C) by Alex Strelnikov
    Copyright (C) 1998 by Andrew Zabolotny <bit@eltech.ru>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Changes :
        KO Myung-Hun <komh@chollian.net> 2007/02/03
            - if ulNumBuffer in dartInit() is 0, it is set to DART_MIN_BUFFERS
              not the suggested thing by the mixer device.

        KO Myung-Hun <komh@chollian.net> 2007/02/11
            - Use MCI_SET_AUDIO_* macros instead of DART_CH_* macros.
*/

#define INCL_DOS
#define INCL_OS2MM
#include <os2.h>
#include <os2me.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dart.h"

// DART need 2 buffers at least to play.
#define DART_MIN_BUFFERS     2

DARTSTRUCT DART = { 0 };

static BOOL m_fDartInited = FALSE;

APIRET APIENTRY dartError( APIRET rc )
{
    if(( USHORT )rc )
    {
        mciGetErrorString( rc,
                           ( PSZ )DART.szErrorCode,
                           sizeof( DART.szErrorCode ));

        fprintf( stdout, "\nDART error(%lx):%s\n", rc, DART.szErrorCode );

        return rc;
    }

    DART.szErrorCode[ 0 ] = 0;

    return 0;
}


APIRET APIENTRY dartStop(void)
{
    MCI_GENERIC_PARMS GenericParms;
    ULONG             rc;

    if( !m_fDartInited )
        return -1;

    memset( &GenericParms, 0, sizeof( GenericParms ));

    GenericParms.hwndCallback = 0;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_STOP,
                         MCI_WAIT,
                         ( PVOID )&GenericParms,
                         0 );
    if( dartError( rc ))
        return rc;

    DART.fStopped = TRUE;

    return 0;
}

APIRET APIENTRY dartClearBuffer( VOID )
{
    int i;

    if( !m_fDartInited )
        return -1;

    for( i = 0; i < DART.ulNumBuffers; i++)
       memset( DART.pMixBuffers[ i ].pBuffer, DART.bSilence, DART.ulBufferSize );

    return 0;
}

static ULONG dartFillBuffer( PMCI_MIX_BUFFER pBuffer )
{
    ULONG ulWritten = 0;

    DART.fSamplesPlayed = FALSE;
    memset( pBuffer->pBuffer, DART.bSilence, DART.ulBufferSize );

    if( DART.pfndicb )
        ulWritten = DART.pfndicb( pBuffer->pBuffer, DART.ulBufferSize );

    if( ulWritten < DART.ulBufferSize )
    {
        pBuffer->ulFlags = MIX_BUFFER_EOS;
        DART.fWaitStreamEnd = TRUE;
    }
    else
        pBuffer->ulFlags = 0;

    return ulWritten;
}


APIRET APIENTRY dartFreeBuffers( VOID )
{
    APIRET  rc;

    if( !m_fDartInited )
        return -1;

    if( DART.pMixBuffers == NULL )
        return 0;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_BUFFER,
                         MCI_WAIT | MCI_DEALLOCATE_MEMORY,
                         ( PVOID )&DART.BufferParms, 0 );
    if( dartError( rc ))
        return rc;

    if( DART.pMixBuffers )
        free( DART.pMixBuffers );

    DART.pMixBuffers = NULL;

    return 0;
}


LONG APIENTRY MixHandler( ULONG ulStatus, PMCI_MIX_BUFFER pBuffer, ULONG ulFlags )
{
    switch( ulFlags )
    {
        case MIX_STREAM_ERROR | MIX_WRITE_COMPLETE:
            // on error, fill next buffer and continue
        case MIX_WRITE_COMPLETE:
        {
            DART.ulBytesPlayed += pBuffer->ulBufferLength;

            // If this is the last buffer, stop
            if( pBuffer->ulFlags & MIX_BUFFER_EOS)
                dartStop();
            else if( !DART.fWaitStreamEnd )
            {
                // Transfer buffer to DART
                dartFillBuffer( pBuffer );

                DART.MixSetupParms.pmixWrite( DART.MixSetupParms.ulMixHandle, pBuffer, 1 );
                DART.fSamplesPlayed = TRUE;
            }
            break;
        }
    }

    return TRUE;
}

APIRET APIENTRY dartChNum( VOID )
{
    ULONG               ulDartFlags;
    ULONG               ulChannels;
    MCI_AMP_OPEN_PARMS  AmpOpenParms;
    MCI_GENERIC_PARMS   GenericParms;
    MCI_MIXSETUP_PARMS  MixSetupParms;

    ulChannels = 0;
    memset( &AmpOpenParms, 0, sizeof( MCI_AMP_OPEN_PARMS ));
    ulDartFlags = MCI_WAIT | MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE;
    AmpOpenParms.usDeviceID = 0;
    AmpOpenParms.pszDeviceType = (PSZ)( MCI_DEVTYPE_AUDIO_AMPMIX |
                                        ( 0 << 16 ));
    if( mciSendCommand( 0, MCI_OPEN, ulDartFlags, ( PVOID )&AmpOpenParms, 0 ))
        return ulChannels;

    ulChannels = 6; // first, try 6 channels
    memset( &MixSetupParms, 0, sizeof( MCI_MIXSETUP_PARMS ));
    MixSetupParms.ulBitsPerSample = 16;
    MixSetupParms.ulSamplesPerSec = 48000;
    MixSetupParms.ulFormatTag = MCI_WAVE_FORMAT_PCM;
    MixSetupParms.ulChannels = ulChannels;
    MixSetupParms.ulFormatMode = MCI_PLAY;
    MixSetupParms.ulDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
    MixSetupParms.pmixEvent = NULL;
    if( mciSendCommand( AmpOpenParms.usDeviceID,
                        MCI_MIXSETUP, MCI_WAIT | MCI_MIXSETUP_INIT,
                        ( PVOID )&MixSetupParms, 0 ))
    {
        ulChannels = 4; // failed. try 4 channels
        MixSetupParms.ulChannels = ulChannels;
        if( mciSendCommand( AmpOpenParms.usDeviceID,
                            MCI_MIXSETUP, MCI_WAIT | MCI_MIXSETUP_INIT,
                            ( PVOID )&MixSetupParms, 0 ))
        ulChannels = 2; // failed again...so, drivers support only 2 channels
    }

    mciSendCommand( AmpOpenParms.usDeviceID,
                    MCI_CLOSE, MCI_WAIT,
                    ( PVOID )&GenericParms, 0 );

    return ulChannels;
}

APIRET APIENTRY dartInit( USHORT usDeviceIndex, ULONG ulBitsPerSample, ULONG ulSamplingRate,
                          ULONG ulDataFormat, ULONG ulChannels, ULONG ulNumBuffers,
                          ULONG ulBufferSize, BOOL fShareable, PFNDICB pfndicb )
{
    APIRET              rc;
    ULONG               ulDartFlags;
    MCI_AMP_OPEN_PARMS  AmpOpenParms;
    MCI_GENERIC_PARMS   GenericParms;

    m_fDartInited = FALSE;

    memset( &DART, 0, sizeof( DARTSTRUCT ));

    DART.bSilence = ( ulBitsPerSample == BPS_16 ) ? 0x00 : 0x80;

    DART.fShareable = fShareable;
    if( DART.fShareable )
        ulDartFlags = MCI_WAIT | MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE;
    else
        ulDartFlags = MCI_WAIT | MCI_OPEN_TYPE_ID;

    memset( &AmpOpenParms, 0, sizeof( MCI_AMP_OPEN_PARMS ));

    AmpOpenParms.usDeviceID = 0;
    AmpOpenParms.pszDeviceType = (PSZ)( MCI_DEVTYPE_AUDIO_AMPMIX |
                                        (( ULONG )usDeviceIndex << 16 ));
    rc = mciSendCommand( 0,
                         MCI_OPEN,
                         ulDartFlags,
                         ( PVOID )&AmpOpenParms,
                         0 );
    if( dartError( rc ))
        return rc;

    DART.usDeviceID = AmpOpenParms.usDeviceID;

    if( !DART.fShareable )
    {
        // Grab exclusive rights to device instance (NOT entire device)
        GenericParms.hwndCallback = 0;
        rc = mciSendCommand( DART.usDeviceID,
                             MCI_ACQUIREDEVICE,
                             MCI_EXCLUSIVE_INSTANCE,
                             ( PVOID )&GenericParms,
                             0 );
        if( dartError( rc ))
            goto exit_release;
    }

    // Setup the mixer for playback of wave data
    memset( &DART.MixSetupParms, 0, sizeof( MCI_MIXSETUP_PARMS ));

    DART.MixSetupParms.ulBitsPerSample = ulBitsPerSample;
    DART.MixSetupParms.ulSamplesPerSec = ulSamplingRate;
    DART.MixSetupParms.ulFormatTag = ulDataFormat;
    DART.MixSetupParms.ulChannels = ulChannels;
    DART.MixSetupParms.ulFormatMode = MCI_PLAY;
    DART.MixSetupParms.ulDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
    DART.MixSetupParms.pmixEvent = ( MIXEREVENT * )MixHandler;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_MIXSETUP,
                         MCI_WAIT | MCI_MIXSETUP_INIT,
                         ( PVOID )&DART.MixSetupParms,
                         0 );

    if( dartError( rc ))
        goto exit_close;

    if( ulNumBuffers < DART_MIN_BUFFERS )
        ulNumBuffers = DART_MIN_BUFFERS;

    // Use the suggested buffer number and size provide by the mixer device if 0
    if( ulNumBuffers == 0 )
        ulNumBuffers = DART.MixSetupParms.ulNumBuffers;

    if( ulBufferSize == 0 )
        ulBufferSize = DART.MixSetupParms.ulBufferSize;

    // Allocate mixer buffers
    DART.pMixBuffers = ( MCI_MIX_BUFFER * )malloc( sizeof( MCI_MIX_BUFFER ) * ulNumBuffers );

    // Set up the BufferParms data structure and allocate device buffers
    // from the Amp-Mixer
    DART.BufferParms.ulStructLength = sizeof( MCI_BUFFER_PARMS );
    DART.BufferParms.ulNumBuffers = ulNumBuffers;
    DART.BufferParms.ulBufferSize = ulBufferSize;
    DART.BufferParms.pBufList = DART.pMixBuffers;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_BUFFER,
                         MCI_WAIT | MCI_ALLOCATE_MEMORY,
                         ( PVOID )&DART.BufferParms,
                         0 );
    if( dartError( rc ))
        goto exit_deallocate;

    // The mixer possibly changed these values
    DART.ulNumBuffers = DART.BufferParms.ulNumBuffers;
    DART.ulBufferSize = DART.BufferParms.ulBufferSize;

    DART.ulSeekPosition = 0;
    DART.pfndicb = pfndicb;

    m_fDartInited = TRUE;

    return 0;

exit_deallocate :
    free( DART.pMixBuffers );
    DART.pMixBuffers = NULL;

exit_release :
    if( !DART.fShareable )
    {
        // Release exclusive rights to device instance (NOT entire device)
        mciSendCommand( DART.usDeviceID,
                        MCI_ACQUIREDEVICE,
                        MCI_RELEASEDEVICE,
                        ( PVOID )&GenericParms,
                        0 );
    }

exit_close :
    mciSendCommand( DART.usDeviceID,
                    MCI_CLOSE,
                    MCI_WAIT,
                    ( PVOID )&GenericParms,
                    0 );

    return rc;
}


APIRET APIENTRY dartClose( VOID )
{
    MCI_GENERIC_PARMS   GenericParms;
    APIRET              rc;

    if( !m_fDartInited )
        return -1;

    dartStop();
    dartFreeBuffers();

    GenericParms.hwndCallback = 0;

    if( !DART.fShareable )
    {
        // Release exclusive rights to device instance (NOT entire device)
        rc = mciSendCommand( DART.usDeviceID,
                             MCI_ACQUIREDEVICE,
                             MCI_RELEASEDEVICE,
                             ( PVOID )&GenericParms,
                             0 );
        if( dartError( rc ))
            return rc;
    }

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_CLOSE,
                         MCI_WAIT,
                         ( PVOID )&GenericParms,
                         0 );
    if( dartError( rc ))
        return rc;

    m_fDartInited = FALSE;

    return 0;
}

APIRET APIENTRY dartPlay( VOID )
{
    int   i;
    ULONG rc;

    if( !m_fDartInited )
        return -1;

    DART.fStopped = DART.fWaitStreamEnd = FALSE;

    DART.ulBytesPlayed = 0;

    for( i = 0; i < DART.ulNumBuffers; i++ )
    {
        DART.pMixBuffers[ i ].ulBufferLength = DART.ulBufferSize;
        DART.pMixBuffers[ i ].ulFlags = 0;
    }

    for( i = 0; i < DART_MIN_BUFFERS; i++ )
    {
        dartFillBuffer( &DART.pMixBuffers[ i ]);
        if( DART.fWaitStreamEnd )
            break;
    }

    if( i < DART_MIN_BUFFERS )
        i++;

    rc = DART.MixSetupParms.pmixWrite( DART.MixSetupParms.ulMixHandle,
                                       DART.pMixBuffers, i );
    if( dartError( rc ))
        return rc;

    return 0;
}

APIRET APIENTRY dartPause( VOID )
{
    MCI_GENERIC_PARMS GenericParms;
    ULONG             rc;

    if( !m_fDartInited )
        return -1;

    DART.ulCurrVolume = dartGetVolume();
    rc = mciSendCommand( DART.usDeviceID,
                         MCI_PAUSE,
                         MCI_WAIT,
                         ( PVOID )&GenericParms,
                         0 );
    if( dartError( rc ))
        return rc;

    DART.fPaused = TRUE;

    return 0;
}


APIRET APIENTRY dartResume( VOID )
{
    MCI_GENERIC_PARMS GenericParms;
    ULONG             rc;

    if( !m_fDartInited )
        return -1;


    rc = mciSendCommand( DART.usDeviceID,
                         MCI_RESUME,
                         MCI_WAIT,
                         ( PVOID )&GenericParms,
                         0 );
    if( dartError( rc ))
        return rc;

    // setting volume of channels separately can be failed.
    if( LOUSHORT( DART.ulCurrVolume ) == HIUSHORT( DART.ulCurrVolume ))
        dartSetVolume( MCI_SET_AUDIO_ALL, LOUSHORT( DART.ulCurrVolume ));
    else
    {
        dartSetVolume( MCI_SET_AUDIO_LEFT, LOUSHORT( DART.ulCurrVolume ));
        dartSetVolume( MCI_SET_AUDIO_RIGHT, HIUSHORT( DART.ulCurrVolume ));
    }

    DART.fPaused = FALSE;

    return 0;
}

APIRET APIENTRY dartGetPos( VOID )
{
    MCI_STATUS_PARMS StatusParms;
    ULONG            rc;

    if( !m_fDartInited )
        return 0;

    memset( &StatusParms, 0, sizeof( MCI_STATUS_PARMS ));
    StatusParms.ulItem = MCI_STATUS_POSITION;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_STATUS,
                         MCI_WAIT | MCI_STATUS_ITEM,
                         ( PVOID )&StatusParms,
                         0 );
    if( dartError( rc ))
        return 0;

    return StatusParms.ulReturn;
}


APIRET APIENTRY dartSetPos( ULONG ulNewPos )
{
    APIRET          rc;
    MCI_SEEK_PARMS  SeekParms;

    if( !m_fDartInited )
        return -1;

    DART.ulSeekPosition = ulNewPos;

    SeekParms.hwndCallback = 0;
    SeekParms.ulTo = ulNewPos;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_SEEK,
                         MCI_WAIT | MCI_TO,
                         ( PVOID )&SeekParms,
                         0 );
    if( dartError( rc ))
        return rc;

    return 0;
}


APIRET APIENTRY dartSetSoundState( ULONG ulCh, BOOL fState)
{
    MCI_SET_PARMS SetParms;
    USHORT        usSt;
    ULONG         rc;

    if( !m_fDartInited )
        return -1;

    if( fState)
        usSt = MCI_SET_ON;
    else
        usSt = MCI_SET_OFF;

    SetParms.ulAudio = ulCh;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_SET,
                         MCI_WAIT | MCI_SET_AUDIO | usSt,
                         ( PVOID)&SetParms, 0 );
    if( dartError( rc ))
        return rc;

    return 0;
}


APIRET APIENTRY dartSetVolume( ULONG ulCh, USHORT usVol)
{
    MCI_SET_PARMS SetParms;
    ULONG         rc;

    if( !m_fDartInited )
        return -1;

    SetParms.ulLevel = usVol;
    SetParms.ulAudio = ulCh;
    rc = mciSendCommand( DART.usDeviceID,
                         MCI_SET,
                         MCI_WAIT | MCI_SET_AUDIO |
                         MCI_SET_VOLUME,
                         ( PVOID )&SetParms, 0);
    if( dartError( rc ))
        return rc;

    return 0;
}


APIRET APIENTRY dartGetVolume( VOID )
{
    MCI_STATUS_PARMS StatusParms;
    ULONG            rc;

    if( !m_fDartInited )
        return 0;

    memset(&StatusParms, 0, sizeof( MCI_STATUS_PARMS ));
    StatusParms.ulItem = MCI_STATUS_VOLUME;

    rc = mciSendCommand( DART.usDeviceID,
                         MCI_STATUS,
                         MCI_WAIT | MCI_STATUS_ITEM,
                         ( PVOID )&StatusParms,
                         0 );
    if( dartError( rc ))
        return 0;

    return StatusParms.ulReturn;
}

/******************************************************************************/
// OS/2 32-bit program to query the Physical Device Driver name
// for the default MMPM/2 WaveAudio device.  Joe Nord 10-Mar-1999
/******************************************************************************/
APIRET APIENTRY OSLibGetAudioPDDName( PSZ pszPDDName )
{
    ULONG                   ulRC;
    CHAR                    szAmpMix[9] = "AMPMIX01";

    MCI_SYSINFO_PARMS       SysInfo;
    MCI_SYSINFO_LOGDEVICE   SysInfoParm;
    MCI_SYSINFO_QUERY_NAME  QueryNameParm;

    memset( &SysInfo, '\0', sizeof( SysInfo ));
    memset( &SysInfoParm, '\0', sizeof( SysInfoParm ));
    memset( &QueryNameParm, '\0', sizeof( QueryNameParm ));

    SysInfo.ulItem       = MCI_SYSINFO_QUERY_NAMES;
    SysInfo.usDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
    SysInfo.pSysInfoParm = &QueryNameParm;

    strcpy( QueryNameParm.szLogicalName, szAmpMix );

    ulRC = mciSendCommand( 0,
                           MCI_SYSINFO,
                           MCI_SYSINFO_ITEM | MCI_WAIT,
                           ( PVOID )&SysInfo,
                           0 );
    if( dartError( ulRC ))
        return ulRC;

//   printf("Audio:\n install name [%s]\n logical name [%s]\n alias [%s]\n type num: %i\n ord num: %i\n",
//          QueryNameParm.szInstallName,  /*  Device install name. */
//          QueryNameParm.szLogicalName,  /*  Logical device name. */
//          QueryNameParm.szAliasName,    /*  Alias name. */
//          QueryNameParm.usDeviceType,   /*  Device type number. */
//          QueryNameParm.usDeviceOrd);   /*  Device type o */

    // Get PDD associated with our AmpMixer
    // Device name is in pSysInfoParm->szPDDName
    SysInfo.ulItem       = MCI_SYSINFO_QUERY_DRIVER;
    SysInfo.usDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
    SysInfo.pSysInfoParm = &SysInfoParm;

    strcpy( SysInfoParm.szInstallName, QueryNameParm.szInstallName );

    ulRC = mciSendCommand( 0,
                           MCI_SYSINFO,
                           MCI_SYSINFO_ITEM | MCI_WAIT,
                           ( PVOID )&SysInfo,
                           0 );
    if( dartError( ulRC ))
        return ulRC;

//    strcpy( pszPDDName, SysInfoParm.szPDDName );
    strcpy ( pszPDDName, SysInfoParm.szProductInfo );
//    printf("Audio:\n product info [%s]\n\n",SysInfoParm.szProductInfo);
//    printf("Audio:\n inst name [%s]\n version [%s]\n MCD drv [%s]\n VSD drv [%s]\n res name: [%s]\n",
//           SysInfoParm.szInstallName,
//           SysInfoParm.szVersionNumber,
//           SysInfoParm.szMCDDriver,
//           SysInfoParm.szVSDDriver,
//           SysInfoParm.szResourceName);

   return 0;
}

