/*
 * Copyright 1997-2003 Samuel Audet <guardia@step.polymtl.ca>
 *                     Taneli Lepp <rosmo@sektori.com>
 *
 * Copyright 2004-2005 Dmitry A.Steklenev <glass@ptv.ru>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define  INCL_WIN
#define  INCL_GPI
#define  INCL_DOS
#define  INCL_DOSERRORS
#define  INCL_WINSTDDRAG
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <math.h>
#include <float.h>

#include <utilfct.h>
#include <list.h>

#include "format.h"
#include "decoder_plug.h"
#include "output_plug.h"
#include "filter_plug.h"
#include "plugin.h"
#include "pm123.h"
#include "plugman.h"
#include "bookmark.h"
#include "button95.h"
#include "pfreq.h"
#include "httpget.h"
#include "genre.h"
#include "copyright.h"

#define  AMP_REFRESH_CONTROLS ( WM_USER + 121   )
#define  AMP_DISPLAYMSG       ( WM_USER + 76    )
#define  AMP_PAINT            ( WM_USER + 77    )
#define  AMP_STOP             ( WM_USER + 78    )

#define  TID_UPDATE_TIMERS    ( TID_USERMAX - 1 )
#define  TID_UPDATE_PLAYER    ( TID_USERMAX - 2 )
#define  TID_ONTOP            ( TID_USERMAX - 3 )

int       amp_playmode = AMP_NOFILE;
HPOINTER  mp3;      /* Song file icon   */
HPOINTER  mp3play;  /* Played file icon */

PLRECORD* currentf; /* The pointer to playlist record of the
                       currently played file, the pointer is
                       NULL if such record is not present. */

/* Contains startup path of the program without its name.  */
char   startpath[_MAX_PATH];
/* Contains a name of the currently loaded file. */
char   current_filename[_MAX_PATH];
/* Other parameters of the currently loaded file. */
int    current_bitrate     =  0;
int    current_channels    = -1;
int    current_length      =  0;
int    current_freq        =  0;
char   current_track       =  0;
char   current_cd_drive[4] = "";
tune   current_tune;
char   current_decoder[128];
char   current_decoder_info_string[128];

static HAB   hab        = NULLHANDLE;
static HWND  hplaylist  = NULLHANDLE;
static HWND  hbookmarks = NULLHANDLE;
static HWND  heq        = NULLHANDLE;
static HWND  hframe     = NULLHANDLE;
static HWND  hplayer    = NULLHANDLE;
static HPIPE hpipe      = NULLHANDLE;

/* Pipe name decided on startup. */
static char  pipename[_MAX_PATH];

static BOOL  is_have_focus    = FALSE;
static BOOL  is_fast_forward  = FALSE;
static BOOL  is_fast_backward = FALSE;
static BOOL  is_paused        = FALSE;
static BOOL  is_volume_drag   = FALSE;
static BOOL  is_seeking       = FALSE;
static BOOL  is_slider_drag   = FALSE;

/* Current seeking time. Valid if is_seeking == TRUE. */
static int   seeking_pos = 0;
static int   upd_options = 0;

int initial_shuffle = 0;
int smooth_scroller = 0;
int o_manager = 0;
ULONG action;
extern int manager_open;
BOOL save_stream = FALSE;

extern OUTPUT_PARAMS out_params;

HMQ hmq;
QMSG qmsg;
HPS hps;
SWP swp;
char buf[512];
char metadata_buffer[128]; // loaded in curtun on decoder's demand WM_METADATA

HSWITCH eqSwits;
SWCNTRL eqSdata;
extern HSWITCH mgrSwits;
extern SWCNTRL mgrSdata;

extern HWND hwndManager;
char http_proxy[256], http_auth[256];

LIST_NODE *dock_list = NULL; // keeps docked windows in memory

static char last_error[2048];

void _System
keep_last_error( char *error )
{
  strncpy( last_error, error, sizeof( last_error ) - 1 );
  last_error[ sizeof( last_error ) - 1 ] = 0;
}

void _System
display_info( char *info )
{
  char* info_display = strdup( info );
  if( info_display ) {
    amp_post_message( AMP_DISPLAYMSG, MPFROMP( info_display ), 0 );
  }
}

/* Sets audio volume to the current selected level. */
void
amp_volume_to_normal( void )
{
  int i = cfg.defaultvol;
  if( i > 100 ) {
      i = 100;
  }
  out_set_volume( i );
}

/* Sets audio volume to below current selected level. */
void
amp_volume_to_lower( void )
{
  int i = cfg.defaultvol;
  if( i > 100 ) {
      i = 100;
  }
  out_set_volume((float)i*3/5);
}

/* Adjusts audio volume to level accordingly current playing mode. */
static void
amp_volume_adjust( void )
{
  if( is_fast_forward || is_fast_backward ) {
    amp_volume_to_lower ();
  } else {
    amp_volume_to_normal();
  }
}

/* Draws all player timers and the position slider. */
static void
amp_paint_timers( HPS hps )
{
  int play_time = 0;
  int play_left = current_length;
  int list_left = 0;

  if( cfg.mode == CFG_MODE_REGULAR )
  {
    if( decoder_playing()) {
      if( !is_seeking ) {
        play_time = time_played();
      } else {
        play_time = seeking_pos;
      }
    }

    if( play_left > 0 ) {
      play_left -= play_time;
    }

    if( amp_playmode == AMP_PLAYLIST && !cfg.rpt ) {
      list_left = pl_playtime( cfg.shf ) - play_time;
    }

    bmp_draw_slider( hps, play_time, time_total());
    bmp_draw_timer ( hps, play_time );

    bmp_draw_tiny_timer( hps, POS_TIME_LEFT, play_left );
    bmp_draw_tiny_timer( hps, POS_PL_LEFT,   list_left );
  }
}

/* Draws all attributes of the currently loaded file. */
static void
amp_paint_fileinfo( HPS hps )
{
  if( amp_playmode == AMP_PLAYLIST ) {
    bmp_draw_plind( hps, pl_index( currentf ), pl_size());
  } else {
    bmp_draw_plind( hps, 0, 0 );
  }

  bmp_draw_plmode  ( hps );
  bmp_draw_timeleft( hps );
  bmp_draw_rate    ( hps, current_bitrate );
  bmp_draw_channels( hps, current_channels );
  bmp_draw_text    ( hps );
  amp_paint_timers ( hps );
}

/* Marks the player window as needed of redrawing. */
void
amp_invalidate( int options )
{
  if( options == UPD_ALL ) {
    WinInvalidateRect( hplayer, NULL, 1 );
  } else if( options & UPD_DELAYED ) {
    upd_options |= ( options & ~UPD_DELAYED );
  } else {
    WinPostMsg( hplayer, AMP_PAINT, MPFROMLONG( options ), 0 );
  }
}

/* Returns the handle of the player window. */
HWND
amp_player_window( void )
{
  return hplayer;
}

/* Posts a message to the message queue associated with
   the player window. */
BOOL
amp_post_message( ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  return WinPostMsg( hplayer, msg, mp1, mp2 );
}

/* Sets bubble text for specified button. */
static void
amp_set_bubbletext( USHORT id, const char *text )
{
  WinPostMsg( WinWindowFromID( hplayer, id ),
                               WM_SETTEXT, MPFROMP(text), 0 );
}

/* Loads the specified playlist record into the player. */
void
amp_pl_load_record( PLRECORD* rec )
{
  DECODER_INFO info;

  strcpy( current_filename, rec->full );
  strcpy( current_decoder, rec->decoder_module_name );
  strcpy( current_cd_drive, rec->cd_drive );
  strcpy( current_decoder_info_string, rec->info_string );

  if( is_track( current_filename )) {
    dec_trackinfo( rec->cd_drive, rec->track, &info, NULL );
  } else {
    dec_fileinfo ( current_filename, &info, NULL );
  }

  strncpy( current_tune.title, rec->songname, sizeof( current_tune.title ));
  amp_gettag( rec->full, &info, &current_tune );

  out_params.formatinfo = rec->format;

  current_bitrate  = rec->bitrate;
  current_channels = rec->channels;
  current_length   = info.songlength/1000;
  current_freq     = rec->freq;
  current_track    = rec->track;

  if( amp_playmode == AMP_PLAYLIST ) {
    currentf = rec;
  }

  amp_display_filename();
  amp_invalidate( UPD_FILEINFO );
}

/* Loads the specified playlist record into the player and
   plays it if this is specified in the player properties or
   the player is already playing. */
void
amp_pl_play_record( PLRECORD* rec )
{
  BOOL decoder_was_playing = decoder_playing();

  if( decoder_was_playing ) {
    amp_stop();
  }

  if( rec  ) {
    amp_pl_load_record( rec );

    if( amp_playmode != AMP_PLAYLIST ) {
      amp_playmode = AMP_SINGLE;
    }
    if( cfg.playonload == 1 || decoder_was_playing ) {
      amp_play();
    }
  }
}

/* Returns a pointer to the first playable playlist record. */
static PLRECORD*
amp_pl_first_record( void )
{
  PLRECORD* rec;

  if( cfg.shf ) {
    rec = pl_random_record();
  } else {
    rec = pl_first_record();
  }

  return rec;
}

/* Returns a pointer to the next playable playlist record. */
static PLRECORD*
amp_pl_next_record( void )
{
  PLRECORD* rec;

  if( cfg.shf ) {
    for( rec = pl_first_record(); rec; rec = pl_next_record( rec )) {
      if( !rec->played ) {
        rec = pl_random_record();
        while( rec->played ) {
          rec = pl_random_record();
        }
        break;
      }
    }
  } else {
    rec = pl_next_record( currentf );
  }

  if( !rec && cfg.rpt ) {
    if( cfg.shf ) {
      pl_clean_shuffle();
    }
    rec = amp_pl_first_record();
  }

  return rec;
}

/* Returns a pointer to the previous playable playlist record. */
static PLRECORD*
amp_pl_prev_record( void )
{
  PLRECORD* rec;

  if( cfg.shf ) {
    return amp_pl_next_record();
  } else {
    rec = pl_prev_record( currentf );
  }

  if( !rec && cfg.rpt ) {
    rec = pl_last_record();
  }

  return rec;
}

/* Activates the current playlist. */
void
amp_pl_use( void )
{
  if( pl_size()) {
    if( amp_playmode == AMP_SINGLE && decoder_playing()) {
      amp_stop();
    }

    amp_playmode = AMP_PLAYLIST;

    if( !currentf ) {
      amp_pl_load_record( amp_pl_first_record());
    }

    if( cfg.playonuse && !decoder_playing()) {
      amp_play();
    }
  }
}

/* Loads a standalone file or CD track to player. */
BOOL
amp_load_singlefile( const char *filename, int options )
{
  DECODER_INFO info;

  int   i;
  ULONG rc;
  char  module_name[128];
  char  cd_drive[4] = "1:";
  int   cd_track    = -1;

  if( is_playlist( filename )) {
    return pl_load( filename, PL_LOAD_CLEAR );
  }

  memset( &info, 0, sizeof( info ));

  if( is_track( filename )) {
    sscanf( filename, "cd:///%1c:\\Track %d", cd_drive, &cd_track );
    if( cd_drive[0] == '1' || cd_track == -1 ) {
      amp_error( hplayer, "Invalid CD URL:\n%s", filename );
      return FALSE;
    }
    rc = dec_trackinfo( cd_drive, cd_track, &info, module_name );
  } else {
    rc = dec_fileinfo((char*)filename, &info, module_name );
    cd_drive[0] = 0;
  }

  if( rc != 0 ) { /* error, invalid file */
    amp_reset();
    handle_dfi_error( rc, filename );
    return FALSE;
  }

  amp_stop();

  amp_playmode = AMP_SINGLE;
  currentf = NULL;

  strcpy( current_filename, filename );
  strcpy( current_decoder, module_name );
  strcpy( current_cd_drive, cd_drive );
  strcpy( current_decoder_info_string, info.tech_info );
  strcpy( current_cd_drive, cd_drive );

  amp_gettag( filename, &info, &current_tune );
  out_params.formatinfo = info.format;

  current_bitrate  = info.bitrate;
  current_channels = info.mode;
  current_length   = info.songlength / 1000;
  current_freq     = info.format.samplerate;
  current_track    = cd_track;

  amp_display_filename();
  amp_invalidate( UPD_FILEINFO );

  if( !( options & AMP_LOAD_NOT_PLAY )) {
    if( cfg.playonload ) {
      amp_play();
    }
  }

  if( !( options & AMP_LOAD_NOT_RECALL ) && !is_track( filename )) {
    for( i = MAX_RECALL - 2; i >= 0; i-- ) {
      strcpy( cfg.last[i + 1], cfg.last[i] );
    }
    strcpy( cfg.last[0], current_filename );
  }

  return TRUE;
}

/* Stops the playing of the current file. Must be called
   from main thread. */
static void
amp_stop_playing( void )
{
  QMSG qms;

  amp_msg( MSG_STOP, 0, 0 );
  amp_set_bubbletext( BMP_PLAY, "Starts playing" );

  WinSendDlgItemMsg( hplayer, BMP_PLAY,  WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_PAUSE, WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_FWD,   WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_REW,   WM_DEPRESS, 0, 0 );
  WinSetWindowText ( hframe, VERSION );

  if( amp_playmode == AMP_PLAYLIST ) {
    pl_mark_as_play( currentf, FALSE );
  }

  // Discards WM_PLAYSTOP message, posted by decoder.
  WinPeekMsg( hab, &qms, hplayer, WM_PLAYSTOP, WM_PLAYSTOP, PM_REMOVE );
  amp_invalidate( UPD_ALL );
}

/* Stops the player. */
void
amp_stop( void )
{
  // The stop of the player always should be initiated from
  // the main thread. Otherwise we can receive the unnecessary
  // WM_PLAYSTOP message.
  WinSendMsg( hplayer, AMP_STOP, 0, 0 );
}

/* Starts playing of the currently loaded file. */
void
amp_play( void )
{
  MSG_PLAY_STRUCT msgplayinfo = { 0 };
  char caption[_MAX_PATH];

  if( amp_playmode == AMP_NOFILE ) {
    WinSendDlgItemMsg( hplayer, BMP_PLAY, WM_DEPRESS, 0, 0 );
    return;
  }

  msgplayinfo.filename = current_filename;
  msgplayinfo.out_filename = current_filename;
  msgplayinfo.drive = current_cd_drive;
  msgplayinfo.track = current_track;
  msgplayinfo.hMain = hplayer;
  msgplayinfo.decoder_needed = current_decoder;

  amp_msg( MSG_PLAY, &msgplayinfo, 0 );

  is_fast_backward = FALSE;
  is_fast_forward  = FALSE;
  is_paused        = FALSE;

  amp_set_bubbletext( BMP_PLAY, "Stops playback" );

  WinSendDlgItemMsg( hplayer, BMP_FWD,   WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_REW,   WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_PAUSE, WM_DEPRESS, 0, 0 );
  WinSendDlgItemMsg( hplayer, BMP_PLAY,  WM_PRESS  , 0, 0 );

  if( amp_playmode == AMP_PLAYLIST ) {
    pl_mark_as_played( currentf, TRUE );
    pl_mark_as_play  ( currentf, TRUE );
  }

  sprintf( caption, "%s - ", VERSION );
  sfnameext( strchr( caption, 0 ), current_filename );
  WinSetWindowText( hframe, caption );
}

/* Shows the context menu of the playlist. */
static void
amp_show_context_menu( HWND parent )
{
  static HWND menu = NULLHANDLE;

  POINTL   pos;
  SWP      swp;
  char     file[_MAX_PATH];
  HWND     mh;
  MENUITEM mi;
  short    id;
  int      i;
  int      count;

  if( menu == NULLHANDLE )
  {
    menu = WinLoadMenu( HWND_OBJECT, 0, MNU_MAIN );

    WinSendMsg( menu, MM_QUERYITEM,
                MPFROM2SHORT( IDM_M_LOAD, TRUE ), MPFROMP( &mi ));

    WinSetWindowULong( mi.hwndSubMenu, QWL_STYLE,
      WinQueryWindowULong( mi.hwndSubMenu, QWL_STYLE ) | MS_CONDITIONALCASCADE );

    WinSendMsg( mi.hwndSubMenu, MM_SETDEFAULTITEMID,
                                MPFROMLONG( IDM_M_LOADFILE ), 0 );
  }

  WinQueryPointerPos( HWND_DESKTOP, &pos );
  WinMapWindowPoints( HWND_DESKTOP, parent, &pos, 1 );

  if( WinWindowFromPoint( parent, &pos, TRUE ) == NULLHANDLE )
  {
    // The context menu is probably activated from the keyboard.
    WinQueryWindowPos( parent, &swp );
    pos.x = swp.cx/2;
    pos.y = swp.cy/2;
  }

  // Update regulars.
  WinSendMsg( menu, MM_QUERYITEM,
              MPFROM2SHORT( IDM_M_LOAD, TRUE ), MPFROMP( &mi ));

  mh    = mi.hwndSubMenu;
  count = LONGFROMMR( WinSendMsg( mh, MM_QUERYITEMCOUNT, 0, 0 ));

  // Remove all items from the load menu except of three first
  // intended for a choice of object of loading.
  for( i = 3; i < count; i++ )
  {
    id = LONGFROMMR( WinSendMsg( mh, MM_ITEMIDFROMPOSITION, MPFROMSHORT(3), 0 ));
    WinSendMsg( mh, MM_DELETEITEM, MPFROM2SHORT( id, FALSE ), 0 );
  }

  if( *cfg.last[0] )
  {
    // Add separator.
    mi.iPosition = MIT_END;
    mi.afStyle = MIS_SEPARATOR;
    mi.afAttribute = 0;
    mi.id = 0;
    mi.hwndSubMenu = (HWND)NULLHANDLE;
    mi.hItem = 0;

    WinSendMsg( mh, MM_INSERTITEM, MPFROMP( &mi ), NULL );

    // Fill the recall list.
    for( i = 0; i < MAX_RECALL; i++ ) {
      if( *cfg.last[i] )
      {
        sprintf( file, "~%u ", i + 1 );

        if( is_http( cfg.last[i] )) {
          strcat( file, "[http] " );
        }

        sfnameext( file + strlen(file), cfg.last[i] );

        mi.iPosition = MIT_END;
        mi.afStyle = MIS_TEXT;
        mi.afAttribute = 0;
        mi.id = (IDM_M_LAST + 1) + i;
        mi.hwndSubMenu = (HWND)NULLHANDLE;
        mi.hItem = 0;

        WinSendMsg( mh, MM_INSERTITEM, MPFROMP( &mi ), MPFROMP( file ));
      }
    }
  }

  // Update bookmarks.
  WinSendMsg( menu, MM_QUERYITEM,
              MPFROM2SHORT( IDM_M_BOOKMARKS, TRUE ), MPFROMP( &mi ));

  load_bookmark_menu( mi.hwndSubMenu );

  // Update plug-ins.
  WinSendMsg( menu, MM_QUERYITEM,
              MPFROM2SHORT( IDM_M_PLUG, TRUE ), MPFROMP( &mi ));

  load_plugin_menu( mi.hwndSubMenu );

  // Update status
  mn_enable_item( menu, IDM_M_TAG,    is_file( current_filename ));
  mn_enable_item( menu, IDM_M_SMALL,  bmp_is_mode_supported( CFG_MODE_SMALL   ));
  mn_enable_item( menu, IDM_M_NORMAL, bmp_is_mode_supported( CFG_MODE_REGULAR ));
  mn_enable_item( menu, IDM_M_TINY,   bmp_is_mode_supported( CFG_MODE_TINY    ));
  mn_enable_item( menu, IDM_M_FONT1,  bmp_is_font_supported( 0 ));
  mn_enable_item( menu, IDM_M_FONT2,  bmp_is_font_supported( 1 ));

  mn_check_item ( menu, IDM_M_FLOAT,  cfg.floatontop );
  mn_check_item ( menu, IDM_M_SAVE,   save_stream    );
  mn_check_item ( menu, IDM_M_FONT1,  cfg.font == 0  );
  mn_check_item ( menu, IDM_M_FONT2,  cfg.font == 1  );
  mn_check_item ( menu, IDM_M_SMALL,  cfg.mode == CFG_MODE_SMALL   );
  mn_check_item ( menu, IDM_M_NORMAL, cfg.mode == CFG_MODE_REGULAR );
  mn_check_item ( menu, IDM_M_TINY,   cfg.mode == CFG_MODE_TINY    );

  WinPopupMenu( parent, parent, menu, pos.x, pos.y, 0,
                PU_HCONSTRAIN   | PU_VCONSTRAIN |
                PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 | PU_KEYBOARD   );
}

/* Adds HTTP file to the playlist or load it to the player. */
void
amp_add_url( HWND owner, int options )
{
  char  url[512];
  HWND  hwnd =
        WinLoadDlg( HWND_DESKTOP, owner, WinDefDlgProc, NULLHANDLE, DLG_URL, 0 );

  if( hwnd == NULLHANDLE ) {
    return;
  }

  WinSendDlgItemMsg( hwnd, ENT_URL, EM_SETTEXTLIMIT, MPFROMSHORT(512), 0 );
  do_warpsans( hwnd );

  if( WinProcessDlg( hwnd ) == DID_OK ) {
    WinQueryDlgItemText( hwnd, ENT_URL, 1024, url );
    if( !is_http( url )) {
      amp_error( owner, "Only HTTP protocol supported for streaming." );
    } else {
      if( options & URL_ADD_TO_LIST ) {
        if( is_playlist( url )) {
          pl_load( url, 0);
        } else {
          pl_add_file( url, NULL, 0 );
        }
      } else {
        amp_load_singlefile( url, 0 );
      }
    }
  }
  WinDestroyWindow( hwnd );
}

/* Processes messages of the dialog of addition of CD tracks. */
static MRESULT EXPENTRY
amp_add_tracks_dlg_proc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch( msg ) {
    case WM_CONTROL:
      if( SHORT1FROMMP(mp1) == CB_DRIVE && SHORT2FROMMP(mp1) == CBN_EFCHANGE ) {
        WinPostMsg( hwnd, WM_COMMAND,
                    MPFROMSHORT( PB_REFRESH ), MPFROM2SHORT( CMDSRC_OTHER, FALSE ));
      }
      break;

    case WM_COMMAND:
      if( COMMANDMSG(&msg)->cmd == PB_REFRESH )
      {
        char drive[3];
        char track[32];
        int  options = WinQueryWindowULong( hwnd, QWL_USER );
        int  i;

        DECODER_CDINFO cdinfo;

        WinQueryDlgItemText( hwnd, CB_DRIVE, sizeof( drive ), drive );
        WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_DELETEALL, 0, 0 );

        if( dec_cdinfo( drive, &cdinfo ) == 0 ) {
          if( cdinfo.firsttrack ) {
            for( i = cdinfo.firsttrack; i <= cdinfo.lasttrack; i++ ) {
              sprintf( track,"Track %02d", i );
              WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_INSERTITEM,
                                 MPFROMSHORT( LIT_END ), MPFROMP( track ));
            }
            if( options & TRK_ADD_TO_LIST ) {
              for( i = cdinfo.firsttrack; i <= cdinfo.lasttrack; i++ ) {
                WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_SELECTITEM,
                                   MPFROMSHORT( i - cdinfo.firsttrack ),
                                   MPFROMLONG( TRUE ));
              }
            } else {
              WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_SELECTITEM,
                                 MPFROMSHORT( 0 ), MPFROMLONG( TRUE ));
            }
          }
        } else {
          amp_error( hwnd, "Cannot find decoder that supports CD tracks." );
        }

        return 0;
      }
  }
  return WinDefDlgProc( hwnd, msg, mp1, mp2 );
}

/* Adds CD tracks to the playlist or load one to the player. */
void
amp_add_tracks( HWND owner, int options )
{
  HFILE hcdrom;
  ULONG action;
  HWND  hwnd =
        WinLoadDlg( HWND_DESKTOP, owner, amp_add_tracks_dlg_proc, NULLHANDLE, DLG_TRACK, 0 );

  if( hwnd == NULLHANDLE ) {
    return;
  }

  WinSetWindowULong( hwnd, QWL_USER, options );
  do_warpsans( hwnd );

  if( options & TRK_ADD_TO_LIST ) {
    WinSetWindowBits( WinWindowFromID( hwnd, LB_TRACKS ), QWL_STYLE, 1, LS_MULTIPLESEL );
    WinSetWindowBits( WinWindowFromID( hwnd, LB_TRACKS ), QWL_STYLE, 1, LS_EXTENDEDSEL );
    WinSetWindowText( hwnd, "Add Tracks" );
  } else {
    WinSetWindowText( hwnd, "Load Track" );
  }

  if( DosOpen( "\\DEV\\CD-ROM2$", &hcdrom, &action, 0,
               FILE_NORMAL, OPEN_ACTION_OPEN_IF_EXISTS,
               OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, NULL ) == NO_ERROR )
  {
    struct {
      USHORT count;
      USHORT first;
    } cd_info;

    char  drive[3] = "X:";
    ULONG len = sizeof( cd_info );
    ULONG i;

    if( DosDevIOCtl( hcdrom, 0x82, 0x60, NULL, 0, NULL,
                             &cd_info, len, &len ) == NO_ERROR )
    {
      for( i = 0; i < cd_info.count; i++ ) {
        drive[0] = 'A' + cd_info.first + i;
        WinSendDlgItemMsg( hwnd, CB_DRIVE, LM_INSERTITEM,
                           MPFROMSHORT( LIT_END ), MPFROMP( drive ));
      }
    }
    DosClose( hcdrom );

    if( *cfg.cddrive ) {
      WinSetDlgItemText( hwnd, CB_DRIVE, cfg.cddrive );
    } else {
      WinSendDlgItemMsg( hwnd, CB_DRIVE, LM_SELECTITEM,
                         MPFROMSHORT( 0 ), MPFROMSHORT( TRUE ));
    }
  }

  if( WinProcessDlg( hwnd ) == DID_OK ) {
    SHORT selected =
          SHORT1FROMMR( WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_QUERYSELECTION,
                        MPFROMSHORT( LIT_FIRST ), 0 ));

    WinQueryDlgItemText( hwnd, CB_DRIVE, sizeof( cfg.cddrive ), cfg.cddrive );

    if( options & TRK_ADD_TO_LIST ) {
      char track[32];
      char cdurl[64];

      while( selected != LIT_NONE ) {
        WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_QUERYITEMTEXT,
                           MPFROM2SHORT( selected, sizeof(track)), MPFROMP(track));
        sprintf( cdurl, "cd:///%s\\%s", cfg.cddrive, track );
        pl_add_file( cdurl, NULL, 0 );
        selected = SHORT1FROMMR( WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_QUERYSELECTION,
                                 MPFROMSHORT( selected ), 0 ));
      }
      pl_completed();
    } else {
      char track[32];
      char cdurl[64];

      WinSendDlgItemMsg( hwnd, LB_TRACKS, LM_QUERYITEMTEXT,
                         MPFROM2SHORT( selected, sizeof(track)), MPFROMP(track));
      sprintf( cdurl, "cd:///%s\\%s", cfg.cddrive, track );
      amp_load_singlefile( cdurl, 0 );
    }
  }
  WinDestroyWindow( hwnd );
}

/* Adds user selected files or directory to the playlist. */
void
amp_add_files( HWND hwnd )
{
  FILEDLG filedialog;

  int    i = 0;
  struct stat fi;
  char*  file;

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_OPEN_DIALOG | FDS_CUSTOM | FDS_MULTIPLESEL;
  filedialog.ulUser     = FDU_DIR_ENABLE | FDU_RECURSEBTN;
  filedialog.pszTitle   = "Add file(s) to playlist";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  strcat( filedialog.szFullFile, cfg.filedir );
  strcat( filedialog.szFullFile, "*.*" );

  WinFileDlg( HWND_DESKTOP, hwnd, &filedialog );

  if( filedialog.lReturn == DID_OK ) {
    if( filedialog.ulFQFCount > 1 ) {
      file = (*filedialog.papszFQFilename)[i];
    } else {
      file = filedialog.szFullFile;
    }

    while( *file ) {
      if( stat( file, &fi ) == 0 && fi.st_mode & S_IFDIR ) {
        pl_add_directory( file, filedialog.ulUser & FDU_RECURSE_ON ? PL_DIR_RECURSIVE : 0 );
        strcpy( cfg.filedir, file );
        if( !is_root( file )) {
          strcat( cfg.filedir, "\\" );
        }
      } else if( is_playlist( file )) {
        pl_load( file, PL_LOAD_CLEAR );
      } else {
        pl_add_file( file, NULL, 0 );
        sdrivedir( cfg.filedir, file );
      }

      if( ++i >= filedialog.ulFQFCount ) {
        break;
      } else {
        file = (*filedialog.papszFQFilename)[i];
      }
    }
    pl_completed();
  }

  WinFreeFileDlgList( filedialog.papszFQFilename );
}

/* If the window can be docked to the player, changes the pswp
   data and docks the specified window to the player. */
void
amp_dock( HWND hwnd, PSWP pswp, LONG margin )
{
  SWP swp_player;

  if( WinQueryWindowPos( hframe, &swp_player )) {
    if( dockWindow( pswp, &swp_player, margin )) {
      WinPostMsg( hframe, WM_DOCKWINDOW,   MPFROMHWND( hwnd ), 0 );
    } else {
      WinPostMsg( hframe, WM_UNDOCKWINDOW, MPFROMHWND( hwnd ), 0 );
    }
  }
}

/* Prepares the player to the drop operation. */
static MRESULT
amp_drag_over( HWND hwnd, PDRAGINFO pdinfo )
{
  PDRAGITEM pditem;
  int       i;
  USHORT    drag_op;
  USHORT    drag;

  if( !DrgAccessDraginfo( pdinfo )) {
    return MRFROM2SHORT( DOR_NEVERDROP, 0 );
  }

  for( i = 0; i < pdinfo->cditem; i++ )
  {
    pditem = DrgQueryDragitemPtr( pdinfo, i );

    if( DrgVerifyRMF( pditem, "DRM_123FILE", NULL ) &&
      ( pdinfo->cditem > 1 && pdinfo->hwndSource == hplaylist )) {

      drag    = DOR_NEVERDROP;
      drag_op = DO_UNKNOWN;
      break;

    } else if( DrgVerifyRMF( pditem, "DRM_OS2FILE", NULL ) ||
               DrgVerifyRMF( pditem, "DRM_123FILE", NULL )) {

      drag    = DOR_DROP;
      drag_op = DO_COPY;

    } else {

      drag    = DOR_NEVERDROP;
      drag_op = DO_UNKNOWN;
      break;
    }
  }

  DrgFreeDraginfo( pdinfo );
  return MPFROM2SHORT( drag, drag_op );
}

/* Receives dropped files or playlist records. */
static MRESULT
amp_drag_drop( HWND hwnd, PDRAGINFO pdinfo )
{
  PDRAGITEM pditem;

  char pathname[_MAX_PATH];
  char filename[_MAX_PATH];
  char fullname[_MAX_PATH];
  int  i;

  if( !DrgAccessDraginfo( pdinfo )) {
    return 0;
  }

  for( i = 0; i < pdinfo->cditem; i++ )
  {
    pditem = DrgQueryDragitemPtr( pdinfo, i );

    DrgQueryStrName( pditem->hstrSourceName,    sizeof( filename ), filename );
    DrgQueryStrName( pditem->hstrContainerName, sizeof( pathname ), pathname );
    strcpy( fullname, pathname );
    strcat( fullname, filename );

    if( DrgVerifyRMF( pditem, "DRM_OS2FILE", NULL ))
    {
      struct stat fi;

      if( stat( fullname, &fi ) == 0 && fi.st_mode & S_IFDIR ) {
        pl_add_directory( fullname, PL_DIR_RECURSIVE );
      } else if( is_playlist( fullname )) {
        pl_load( fullname, 0 );
      } else if( pdinfo->cditem == 1 ) {
        amp_load_singlefile( fullname, 0 );
      } else {
        pl_add_file( fullname, NULL, 0 );
      }
    }
    else if( DrgVerifyRMF( pditem, "DRM_123FILE", NULL ))
    {
      if( pdinfo->cditem == 1 ) {
        if( pdinfo->hwndSource == hplaylist && amp_playmode == AMP_PLAYLIST ) {
          amp_pl_play_record((PLRECORD*)pditem->ulItemID );
        } else {
          amp_load_singlefile( fullname, 0 );
        }
      } else {
        pl_add_file( fullname, NULL, 0 );
      }
    }
  }

  DrgDeleteDraginfoStrHandles( pdinfo );
  DrgFreeDraginfo( pdinfo );
  return 0;
}


/* Processes messages of the dialog of edition of ID3 tag. */
static MRESULT EXPENTRY
id3_page_dlg_proc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch( msg )
  {
    case WM_INITDLG:
    {
      int i;

      for( i = 0; i <= GENRE_LARGEST; i++) {
        WinSendDlgItemMsg( hwnd, CB_ID3_GENRE, LM_INSERTITEM,
                           MPFROMSHORT( LIT_END ), MPFROMP( genres[i] ));
      }
      break;
    }

    case WM_COMMAND:
      if( COMMANDMSG(&msg)->cmd == PB_ID3_UNDO )
      {
        tune* tag = (tune*)WinQueryWindowPtr( hwnd, 0 );

        WinSetDlgItemText( hwnd, EN_ID3_TITLE,   tag->title   );
        WinSetDlgItemText( hwnd, EN_ID3_ARTIST,  tag->artist  );
        WinSetDlgItemText( hwnd, EN_ID3_ALBUM,   tag->album   );
        WinSetDlgItemText( hwnd, EN_ID3_COMMENT, tag->comment );
        WinSetDlgItemText( hwnd, EN_ID3_YEAR,    tag->year    );

        WinSendDlgItemMsg( hwnd, CB_ID3_GENRE, LM_SELECTITEM,
                           MPFROMSHORT( tag->gennum ), MPFROMSHORT( TRUE ));
      }
      return 0;
  }
  return WinDefDlgProc( hwnd, msg, mp1, mp2 );
}

/* Processes messages of the dialog of edition of ID3 tag. */
static MRESULT EXPENTRY
id3_dlg_proc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch( msg )
  {
    case WM_INITDLG:
    case WM_WINDOWPOSCHANGED:
    {
      RECTL rect;

      WinQueryWindowRect( hwnd, &rect );
      WinCalcFrameRect( hwnd, &rect, TRUE );

      WinSetWindowPos( WinWindowFromID( hwnd, NB_ID3TAG ), 0,
                       rect.xLeft,
                       rect.yBottom,
                       rect.xRight-rect.xLeft,
                       rect.yTop-rect.yBottom, SWP_SIZE | SWP_MOVE );
      break;
    }
  }
  return WinDefDlgProc( hwnd, msg, mp1, mp2 );
}

/* Edits a ID3 tag for the specified file. */
void amp_id3_edit( HWND owner, const char* filename )
{
  char      caption[_MAX_FNAME] = "ID3 Tag Editor - ";
  HWND      hwnd;
  HWND      book;
  HWND      page01;
  MRESULT   id;
  tune      old_tag;
  tune      new_tag;
  PLRECORD* rec;

  if( !is_file( filename )) {
    DosBeep( 800, 100 );
    return;
  }

  hwnd = WinLoadDlg( HWND_DESKTOP, owner,
                     id3_dlg_proc, NULLHANDLE, DLG_ID3TAG, 0 );

  sfnameext( strchr( caption, 0 ), filename );
  WinSetWindowText( hwnd, caption );

  book = WinWindowFromID( hwnd, NB_ID3TAG );
  do_warpsans( book );

  WinSendMsg( book, BKM_SETDIMENSIONS, MPFROM2SHORT(100,25), MPFROMSHORT(BKA_MAJORTAB));
  WinSendMsg( book, BKM_SETDIMENSIONS, MPFROMLONG(0), MPFROMSHORT(BKA_MINORTAB));
  WinSendMsg( book, BKM_SETNOTEBOOKCOLORS, MPFROMLONG(SYSCLR_FIELDBACKGROUND),
              MPFROMSHORT(BKA_BACKGROUNDPAGECOLORINDEX));

  page01 = WinLoadDlg( book, book, id3_page_dlg_proc, NULLHANDLE, DLG_ID3V10, 0 );
  id     = WinSendMsg( book, BKM_INSERTPAGE, MPFROMLONG(0),
                       MPFROM2SHORT( BKA_AUTOPAGESIZE | BKA_MAJOR, BKA_LAST ));
  do_warpsans( page01 );

  WinSendMsg( book, BKM_SETPAGEWINDOWHWND, MPFROMLONG(id), MPFROMLONG( page01 ));
  WinSendMsg( book, BKM_SETTABTEXT, MPFROMLONG( id ), MPFROMP( "ID3 Tag" ));

  memset( &old_tag, 0, sizeof( old_tag ));
  memset( &new_tag, 0, sizeof( new_tag ));

  amp_gettag( filename, NULL, &old_tag );
  WinSetWindowPtr( page01, 0, &old_tag );
  WinPostMsg( page01, WM_COMMAND,
              MPFROMSHORT( PB_ID3_UNDO ), MPFROM2SHORT( CMDSRC_OTHER, FALSE ));

  WinProcessDlg( hwnd );

  WinQueryDlgItemText( page01, EN_ID3_TITLE,   sizeof( new_tag.title   ), new_tag.title   );
  WinQueryDlgItemText( page01, EN_ID3_ARTIST,  sizeof( new_tag.artist  ), new_tag.artist  );
  WinQueryDlgItemText( page01, EN_ID3_ALBUM,   sizeof( new_tag.album   ), new_tag.album   );
  WinQueryDlgItemText( page01, EN_ID3_COMMENT, sizeof( new_tag.comment ), new_tag.comment );
  WinQueryDlgItemText( page01, EN_ID3_YEAR,    sizeof( new_tag.year    ), new_tag.year    );

  new_tag.gennum =
    SHORT1FROMMR( WinSendDlgItemMsg( page01, CB_ID3_GENRE,
                                     LM_QUERYSELECTION, MPFROMSHORT( LIT_CURSOR ), 0 ));
  WinDestroyWindow( page01 );
  WinDestroyWindow( hwnd   );

  if( strcmp( old_tag.title,   new_tag.title   ) == 0 &&
      strcmp( old_tag.artist,  new_tag.artist  ) == 0 &&
      strcmp( old_tag.album,   new_tag.album   ) == 0 &&
      strcmp( old_tag.comment, new_tag.comment ) == 0 &&
      strcmp( old_tag.year,    new_tag.year    ) == 0 &&
      old_tag.gennum == new_tag.gennum )
  {
    return;
  }

  if( !amp_puttag( filename, &new_tag )) {
    amp_error( owner, "Unable write ID3 tag to file:\n%s\n%s", filename, clib_strerror(errno));
    return;
  }

  amp_gettag( filename, NULL, &new_tag );

  for( rec = pl_first_record(); rec; rec = pl_next_record( rec )) {
    if( stricmp( rec->full, filename ) == 0 ) {
      pl_set_tag( rec, &new_tag, NULL );
      pl_refresh_record( rec );
    }
  }

  if( stricmp( current_filename, filename ) == 0 ) {
    current_tune = new_tag;
    amp_display_filename();
    amp_invalidate( UPD_ALL );
  }
}

/* Wipes a ID3 tag for the specified file. */
void amp_id3_wipe( HWND owner, const char* filename )
{
  tune      tag;
  PLRECORD* rec;

  if( !is_file( filename )) {
    DosBeep( 800, 100 );
    return;
  }

  if( !amp_wipetag( filename )) {
    amp_error( owner, "Unable wipe ID3 tag to file:\n%s\n%s", filename, clib_strerror(errno));
    return;
  }

  amp_gettag( filename, NULL, &tag );

  for( rec = pl_first_record(); rec; rec = pl_next_record( rec )) {
    if( stricmp( rec->full, filename ) == 0 ) {
      pl_set_tag( rec, &tag, NULL );
      pl_refresh_record( rec );
    }
  }

  if( stricmp( current_filename, filename ) == 0 ) {
    current_tune = tag;
    amp_display_filename();
    amp_invalidate( UPD_ALL );
  }
}

/* Default dialog procedure for the file dialog. */
MRESULT EXPENTRY
amp_file_dlg_proc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  FILEDLG* filedialog =
    (FILEDLG*)WinQueryWindowULong( hwnd, QWL_USER );

  switch( msg )
  {
    case WM_INITDLG:
      if( filedialog && !(filedialog->ulUser & FDU_RECURSEBTN )) {
        WinShowWindow( WinWindowFromID( hwnd, CB_RECURSE ), FALSE );
      } else {
        WinCheckButton( hwnd, CB_RECURSE, cfg.add_recursive );
      }
      if( filedialog && !(filedialog->ulUser & FDU_RELATIVBTN )) {
        WinShowWindow( WinWindowFromID( hwnd, CB_RELATIV ), FALSE );
      } else {
        WinCheckButton( hwnd, CB_RELATIV, cfg.save_relative );
      }
      do_warpsans( hwnd );
      break;

    case WM_CONTROL:
      if( SHORT1FROMMP(mp1) == DID_FILENAME_ED && SHORT2FROMMP(mp1) == EN_CHANGE )
      {
        char file[_MAX_PATH];
        WinQueryDlgItemText( hwnd, DID_FILENAME_ED, sizeof(file), file );

        if( filedialog->ulUser & FDU_RECURSEBTN ) {
          if( !*file || strcmp( file, "*"   ) == 0 ||
                        strcmp( file, "*.*" ) == 0 )
          {
            WinEnableControl( hwnd, CB_RECURSE, TRUE  );
          } else {
            WinEnableControl( hwnd, CB_RECURSE, FALSE );
          }
        }

        // Prevents DID_OK from being greyed out.
        if( filedialog->ulUser & FDU_DIR_ENABLE ) {
          if( *file ) {
            return 0;
          }
        }
      }
      break;

    case WM_COMMAND:
      if( SHORT1FROMMP(mp1) == DID_OK )
      {
        if( filedialog->ulUser & FDU_RELATIVBTN ) {
          if( !WinQueryButtonCheckstate( hwnd, CB_RELATIV )) {
            filedialog->ulUser &= ~FDU_RELATIV_ON;
            cfg.save_relative = FALSE;
          } else {
            filedialog->ulUser |=  FDU_RELATIV_ON;
            cfg.save_relative = TRUE;
          }
        }

        if( filedialog->ulUser & FDU_DIR_ENABLE )
        {
          char file[_MAX_PATH];
          WinQueryDlgItemText( hwnd, DID_FILENAME_ED, sizeof(file), file );

          if( !*file ||
              strcmp( file, "*"   ) == 0 ||
              strcmp( file, "*.*" ) == 0 )
          {
            if( !is_root( filedialog->szFullFile )) {
              filedialog->szFullFile[strlen(filedialog->szFullFile)-1] = 0;
            }

            filedialog->lReturn    = DID_OK;
            filedialog->ulFQFCount = 1;

            if( filedialog->ulUser & FDU_RECURSEBTN ) {
              if( !WinQueryButtonCheckstate( hwnd, CB_RECURSE )) {
                filedialog->ulUser &= ~FDU_RECURSE_ON;
                cfg.add_recursive = FALSE;
              } else {
                filedialog->ulUser |=  FDU_RECURSE_ON;
                cfg.add_recursive = TRUE;
              }
            }

            WinDismissDlg( hwnd, DID_OK );
            return 0;
          }
        }
      }
      break;
  }
  return WinDefFileDlgProc( hwnd, msg, mp1, mp2 );
}

/* Create main pipe with only one instance possible since these pipe
   is almost all the time free, it wouldn't make sense having multiple
   intances. */
static BOOL
amp_pipe_create( void )
{
  ULONG rc;
  int   i = 1;

  while(( rc = DosCreateNPipe( pipename, &hpipe,
                               NP_ACCESS_DUPLEX,
                               NP_WAIT | NP_TYPE_BYTE | NP_READMODE_BYTE | 1,
                               2048,
                               2048,
                               500 )) == ERROR_PIPE_BUSY )
  {
    sprintf( pipename,"\\PIPE\\PM123_%d", ++i );
  }

  if( rc != 0 ) {
    amp_player_error( "Could not create pipe %s, rc = %d.", pipename, rc );
    return FALSE;
  } else {
    return TRUE;
  }
}

/* Writes data to the specified pipe. */
static void
amp_pipe_write( HPIPE pipe, char *buf )
{
  ULONG action;

  DosWrite( pipe, buf, strlen( buf ) + 1, &action );
  DosResetBuffer( pipe );
}

/* Dispatches requests received from the pipe. */
static void
amp_pipe_thread( void* scrap )
{
  char  buffer [2048];
  char  command[2048];
  char* zork;
  char* dork;
  ULONG bytesread;
  HAB   hab = WinInitialize( 0 );

  WinCreateMsgQueue( hab, 0 );

  for(;;)
  {
    DosDisConnectNPipe( hpipe );
    DosConnectNPipe( hpipe );

    if( DosRead( hpipe, buffer, sizeof( buffer ), &bytesread ) == NO_ERROR )
    {
      buffer[bytesread] = 0;
      blank_strip( buffer );

      if( *buffer && *buffer != '*' )
      {
        amp_load_singlefile( buffer, 0 );
      }
      else if( *buffer == '*' )
      {
        strcpy( command, buffer + 1 ); /* Strip the '*' character */
        blank_strip( command );

        zork = strtok( command, " " );
        dork = strtok( NULL,    ""  );

        if( zork )
        {
          if( stricmp( zork, "status" ) == 0 ) {
            if( dork ) {
              if( amp_playmode == AMP_NOFILE ) {
                amp_pipe_write( hpipe, "" );
              } else if( stricmp( dork, "file" ) == 0 ) {
                amp_pipe_write( hpipe, current_filename );
              } else if( stricmp( dork, "tag"  ) == 0 ) {
                char info[512];
                amp_pipe_write( hpipe, amp_construct_tag_string( info,  &current_tune ));
              } else if( stricmp( dork, "info" ) == 0 ) {
                amp_pipe_write( hpipe, current_decoder_info_string );
              }
            }
          }
          if( stricmp( zork, "size" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "regular" ) == 0 ||
                  stricmp( dork, "0"       ) == 0 ||
                  stricmp( dork, "normal"  ) == 0  )
              {
                WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_NORMAL ), 0 );
              }
              if( stricmp( dork, "small"   ) == 0 ||
                  stricmp( dork, "1"       ) == 0  )
              {
                WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_SMALL  ), 0 );
              }
              if( stricmp( dork, "tiny"    ) == 0 ||
                  stricmp( dork, "2"       ) == 0  )
              {
                WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_TINY   ), 0 );
              }
            }
          }
          if( stricmp( zork, "rdir" ) == 0 ) {
            if( dork ) {
              pl_add_directory( dork, PL_DIR_RECURSIVE );
            }
          }
          if( stricmp( zork, "dir"  ) == 0 ) {
            if( dork ) {
              pl_add_directory( dork, 0 );
            }
          }
          if( stricmp( zork, "font" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "1" ) == 0 ) {
                WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_FONT1 ), 0 );
              }
              if( stricmp( dork, "2" ) == 0 ) {
                WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_FONT2 ), 0 );
              }
            }
          }
          if( stricmp( zork, "add" ) == 0 )
          {
            char* file;

            if( dork ) {
              while( *dork ) {
                file = dork;
                while( *dork && *dork != ';' ) {
                  ++dork;
                }
                if( *dork == ';' ) {
                  *dork++ = 0;
                }
                pl_add_file( file, NULL, 0 );
              }
            }
          }
          if( stricmp( zork, "load" ) == 0 ) {
            if( dork ) {
              amp_load_singlefile( dork, 0 );
            }
          }
          if( stricmp( zork, "hide"  ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_MINIMIZE ), 0 );
          }
          if( stricmp( zork, "float" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "off" ) == 0 || stricmp( dork, "0" ) == 0 ) {
                if( cfg.floatontop ) {
                  WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_FLOAT ), 0 );
                }
              }
              if( stricmp( dork, "on"  ) == 0 || stricmp( dork, "1" ) == 0 ) {
                if( !cfg.floatontop ) {
                  WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( IDM_M_FLOAT ), 0 );
                }
              }
            }
          }
          if( stricmp( zork, "use" ) == 0 ) {
            amp_pl_use();
          }
          if( stricmp( zork, "clear" ) == 0 ) {
            pl_clear();
          }
          if( stricmp( zork, "next" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_NEXT ), 0 );
          }
          if( stricmp( zork, "previous" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_PREV ), 0 );
          }
          if( stricmp( zork, "forward" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_FWD  ), 0 );
          }
          if( stricmp( zork, "rewind" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_REW  ), 0 );
          }
          if( stricmp( zork, "stop" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_STOP ), 0 );
          }
          if( stricmp( zork, "jump" ) == 0 ) {
            if( dork ) {
              int i = atoi( dork ) * 1000;
              amp_msg( MSG_JUMP, &i, 0 );
            }
          }
          if( stricmp( zork, "play" ) == 0 ) {
            if( dork ) {
              amp_load_singlefile( dork, AMP_LOAD_NOT_PLAY );
              WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_PLAY ), 0 );
            } else if( !decoder_playing()) {
              WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_PLAY ), 0 );
            }
          }
          if( stricmp( zork, "pause" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "off" ) == 0 || stricmp( dork, "0" ) == 0 ) {
                if( is_paused ) {
                  WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_PAUSE ), 0 );
                }
              }
              if( stricmp( dork, "on"  ) == 0 || stricmp( dork, "1" ) == 0 ) {
                if( !is_paused ) {
                  WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_PAUSE ), 0 );
                }
              }
            }
          }
          if( stricmp( zork, "playonload" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "off" ) == 0 || stricmp( dork, "0" ) == 0 ) {
                cfg.playonload = FALSE;
              }
              if( stricmp( dork, "on"  ) == 0 || stricmp( dork, "1" ) == 0 ) {
                cfg.playonload = TRUE;
              }
            }
          }
          if( stricmp( zork, "autouse" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "off" ) == 0 || stricmp( dork, "0" ) == 0 ) {
                cfg.autouse = FALSE;
              }
              if( stricmp( dork, "on"  ) == 0 || stricmp( dork, "1" ) == 0 ) {
                cfg.autouse = TRUE;
              }
            }
          }
          if( stricmp( zork, "playonuse" ) == 0 ) {
            if( dork ) {
              if( stricmp( dork, "off" ) == 0 || stricmp( dork, "0" ) == 0 ) {
                cfg.playonuse = FALSE;
              }
              if( stricmp( dork, "on"  ) == 0 || stricmp( dork, "1" ) == 0 ) {
                cfg.playonuse = TRUE;
              }
            }
          }
          if( stricmp( zork, "repeat"  ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_REPEAT  ), 0 );
          }
          if( stricmp( zork, "shuffle" ) == 0 ) {
            WinSendMsg( hplayer, WM_COMMAND, MPFROMSHORT( BMP_SHUFFLE ), 0 );
          }
          if( stricmp( zork, "volume" ) == 0 )
          {
            char buf[64];

            if( dork )
            {
              HPS hps = WinGetPS( hplayer );
              bmp_draw_volume( hps, cfg.defaultvol = atoi( dork ));
              WinReleasePS( hps );
              amp_volume_adjust();
            }
            amp_pipe_write( hpipe, _itoa( cfg.defaultvol, buf, 10 ));
          }
        }
      }
    }
  }
}

/* Loads a file selected by the user to the player. */
void
amp_load_file( HWND owner )
{
  FILEDLG filedialog;

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_OPEN_DIALOG | FDS_CUSTOM;
  filedialog.pszTitle   = "Load file";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  strcpy( filedialog.szFullFile, cfg.filedir );
  strcat( filedialog.szFullFile, "*.*" );

  WinFileDlg( HWND_DESKTOP, owner, &filedialog );

  if( filedialog.lReturn == DID_OK ) {
    sdrivedir( cfg.filedir, filedialog.szFullFile );
    amp_load_singlefile( filedialog.szFullFile, 0 );
  }
}

/* Loads a playlist selected by the user to the player. */
void
amp_load_list( HWND owner )
{
  FILEDLG filedialog;

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_OPEN_DIALOG | FDS_CUSTOM;
  filedialog.pszTitle   = "Load playlist";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  strcpy( filedialog.szFullFile, cfg.listdir );
  strcat( filedialog.szFullFile, "*.LST;*.MPL;*.M3U;*.PLS" );

  WinFileDlg( HWND_DESKTOP, owner, &filedialog );

  if( filedialog.lReturn == DID_OK )
  {
    sdrivedir( cfg.listdir, filedialog.szFullFile );
    if( is_playlist( filedialog.szFullFile )) {
      pl_load( filedialog.szFullFile, PL_LOAD_CLEAR );
    }
  }
}

/* Saves current playlist to the file specified by user. */
void
amp_save_list_as( HWND owner, int options )
{
  FILEDLG filedialog;
  char filez[_MAX_PATH];
  char ext  [_MAX_EXT ];

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_SAVEAS_DIALOG | FDS_CUSTOM | FDS_ENABLEFILELB;
  filedialog.pszTitle   = "Save playlist";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;
  filedialog.ulUser     = FDU_RELATIVBTN;

  strcpy( filedialog.szFullFile, cfg.listdir );
  if( options & SAV_M3U_PLAYLIST ) {
    strcat( filedialog.szFullFile, "*.M3U" );
  } else {
    strcat( filedialog.szFullFile, "*.LST" );
  }

  WinFileDlg( HWND_DESKTOP, owner, &filedialog );

  if( filedialog.lReturn == DID_OK )
  {
    sdrivedir( cfg.listdir, filedialog.szFullFile );
    strcpy( filez, filedialog.szFullFile );
    if( strcmp( sext( ext, filez ), "" ) == 0 ) {
      if( options & SAV_M3U_PLAYLIST ) {
        strcat( filez, ".m3u" );
      } else {
        strcat( filez, ".lst" );
      }
    }
    if( amp_warn_if_overwrite( owner, filez )) {
      pl_save( filez, ( filedialog.ulUser & FDU_RELATIV_ON ? PL_SAVE_RELATIVE : 0 )
                    | ( options & SAV_M3U_PLAYLIST ? PL_SAVE_M3U : PL_SAVE_PLS    ));
    }
  }
}

void amp_loadskin( HAB hab, HWND hwnd, HPS hps)
{
  FILEDLG filedialog;

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_OPEN_DIALOG | FDS_CUSTOM;
  filedialog.pszTitle   = "Load PM123 skin";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  sdrivedir( filedialog.szFullFile, cfg.defskin );
  strcat( filedialog.szFullFile, "*.skn" );

  WinFileDlg( HWND_DESKTOP, HWND_DESKTOP, &filedialog );

  if( filedialog.lReturn == DID_OK ) {
    bmp_load_skin( filedialog.szFullFile, hab, hplayer, hps );
    strcpy( cfg.defskin, filedialog.szFullFile );
  }
}

BOOL
amp_save_eq( HWND owner, float *gains, BOOL *mutes, float preamp )
{
  FILEDLG filedialog;
  FILE*   file;
  int     i;
  char    ext[_MAX_EXT ];

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_SAVEAS_DIALOG | FDS_CUSTOM;
  filedialog.pszTitle   = "Save equalizer";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  if( cfg.lasteq[0] == 0 ) {
    strcpy( filedialog.szFullFile, "*.EQ" );
  } else {
    strcpy( filedialog.szFullFile, cfg.lasteq );
  }

  WinFileDlg( HWND_DESKTOP, owner, &filedialog );

  if( filedialog.lReturn == DID_OK )
  {
    if( strcmp( sext( ext, filedialog.szFullFile ), "" ) == 0 ) {
      strcat( filedialog.szFullFile, ".eq" );
    }

    if( amp_warn_if_overwrite( owner, filedialog.szFullFile ))
    {
      strcpy( cfg.lasteq, filedialog.szFullFile );
      file = fopen( filedialog.szFullFile, "w" );

      if( file == NULL ) {
        return FALSE;
      }

      fprintf( file, "#\n# Equalizer created with %s\n# Do not modify!\n#\n", VERSION );
      fprintf( file, "# Band gains\n" );
      for( i = 0; i < 20; i++ ) {
        fprintf( file, "%g\n", gains[i] );
      }
      fprintf( file, "# Mutes\n" );
      for( i = 0; i < 20; i++ ) {
        fprintf( file, "%u\n", mutes[i] );
      }
      fprintf( file, "# Preamplifier\n" );
      fprintf( file, "%g\n", preamp );

      fprintf( file, "# End of equalizer\n" );
      fclose( file );
      return TRUE;
    }
  }
  return FALSE;
}

BOOL amp_load_eq_file(char *filename, float *gains, BOOL *mutes, float *preamp)
{
   int i;
   FILE *file;
   char vz[CCHMAXPATH];

   file = fopen(filename, "r");
   if (file == NULL)
      return FALSE;
   i = 0;
   while (!feof(file))
   {
      fgets(vz, sizeof(vz), file);
      blank_strip(vz);
      if (*vz && vz[0] != '#' && vz[0] != ';' && i < 41)
      {
         if (i < 20) gains[i] = atof(vz);
         if (i > 19 && i < 40) { mutes[i-20] = atoi(vz);  }
         if (i == 40) { *preamp = atof(vz); }
         i++;
      }
   }
   fclose(file);
   return TRUE;
}


BOOL amp_load_eq(HWND hwnd, float *gains, BOOL *mutes, float *preamp)
{
  FILEDLG filedialog;

  memset( &filedialog, 0, sizeof( FILEDLG ));
  filedialog.cbSize     = sizeof( FILEDLG );
  filedialog.fl         = FDS_CENTER | FDS_OPEN_DIALOG | FDS_CUSTOM;
  filedialog.pszTitle   = "Load equalizer";
  filedialog.hMod       = NULLHANDLE;
  filedialog.usDlgId    = DLG_FILE;
  filedialog.pfnDlgProc = amp_file_dlg_proc;

  sdrivedir( filedialog.szFullFile, cfg.lasteq );
  strcat( filedialog.szFullFile, "*.EQ" );

  WinFileDlg( HWND_DESKTOP, HWND_DESKTOP, &filedialog );

  if( filedialog.lReturn == DID_OK ) {
    strcpy( cfg.lasteq, filedialog.szFullFile );
    return amp_load_eq_file( filedialog.szFullFile, gains, mutes, preamp );
  }
  return FALSE;
}

// returns if the save stream feature has been enabled
BOOL amp_save_stream( HWND hwnd, BOOL enable )
{
  if( enable )
  {
    FILEDLG filedialog;

    memset( &filedialog, 0, sizeof( FILEDLG ));
    filedialog.cbSize     = sizeof( FILEDLG );
    filedialog.fl         = FDS_CENTER | FDS_SAVEAS_DIALOG | FDS_CUSTOM;
    filedialog.pszTitle   = "Save stream as";
    filedialog.hMod       = NULLHANDLE;
    filedialog.usDlgId    = DLG_FILE;
    filedialog.pfnDlgProc = amp_file_dlg_proc;

    strcpy( filedialog.szFullFile, cfg.savedir );
    strcat( filedialog.szFullFile, "*.*" );

    WinFileDlg( HWND_DESKTOP, hwnd, &filedialog );

    if( filedialog.lReturn == DID_OK ) {
      if( amp_warn_if_overwrite( hwnd, filedialog.szFullFile ))
      {
        amp_msg( MSG_SAVE, filedialog.szFullFile, 0 );
        sdrivedir( cfg.savedir, filedialog.szFullFile );
        return TRUE;
      }
    }
  } else {
    amp_msg( MSG_SAVE, NULL, 0 );
  }

  return FALSE;
}

/* Starts playing a next file or stops the player if all files
   already played. */
static void
amp_playstop( HWND hwnd )
{
  amp_stop();

  if( amp_playmode == AMP_SINGLE && cfg.rpt ) {
    amp_play();
  }

  if( amp_playmode == AMP_PLAYLIST )
  {
    PLRECORD* rec = amp_pl_next_record();

    if( rec ) {
      amp_pl_load_record( rec );
      amp_play();
    } else {
      rec = amp_pl_first_record();
      if( rec ) {
        pl_clean_shuffle();
        amp_pl_load_record( rec );
      }
    }
  }
}

// equalizer stuff
float gains[20];
BOOL  mutes[20];
float preamp;

MRESULT EXPENTRY wpEqualizer(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   int i;
   static ULONG slider_range;
   static BOOL nottheuser = FALSE;

   switch (msg)
   {
      case WM_CLOSE:
        heq = NULLHANDLE;
        WinDestroyWindow( hwnd );
        return 0;

      case WM_DESTROY:
      {
        SWP swp;

        if( WinQueryWindowPos( hwnd, &swp )) {
          cfg.Equalizer = swp;
        }
        return WinDefDlgProc(hwnd, msg, mp1, mp2);
      }

      case AMP_REFRESH_CONTROLS:
      {
         float db;
         ULONG value;

         nottheuser = TRUE;

         _control87(EM_INVALID | EM_DENORMAL | EM_ZERODIVIDE | EM_OVERFLOW | EM_UNDERFLOW | EM_INEXACT, MCW_EM);

         for (i = 0; i < 10; i++)
         {
            db = 20*log10(gains[i]);
            value = (db * slider_range/2)/12 + slider_range/2;

            WinSendMsg(WinWindowFromID(hwnd, 101 + i),
                    SLM_SETSLIDERINFO,
                    MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE),
                    MPFROMSHORT(value));

            WinSendMsg(WinWindowFromID(hwnd, 125 + i),
                    BM_SETCHECK,
                    MPFROMSHORT(mutes[i]),
                    0);
         }
         db = 20*log10(preamp);
         value = (db * slider_range/2)/12 + slider_range/2;

         WinSendMsg(WinWindowFromID(hwnd, 139),
                 SLM_SETSLIDERINFO,
                 MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE),
                 MPFROMSHORT(value));

         WinSendMsg(WinWindowFromID(hwnd, 121),
                 BM_SETCHECK,
                 MPFROMSHORT(cfg.eq_enabled),
                 0);

         nottheuser = FALSE;

         break;
      }

      case WM_COMMAND:
         switch (COMMANDMSG(&msg) -> cmd)
         {
            case 123:
               if(amp_load_eq(hwnd, gains, mutes, &preamp))
                  WinSendMsg(hwnd,AMP_REFRESH_CONTROLS,0,0);

               if(WinQueryButtonCheckstate( hwnd, 121 ))
                  equalize_sound(gains, mutes, preamp, 1);

               break;

            case 124:
               amp_save_eq(hwnd, gains, mutes, preamp);
               break;

            case 122:
               for (i = 0; i < 20; i++) gains[i] = 1.0;
               for (i = 0; i < 20; i++) mutes[i] = 0;

               for (i = 0; i < 10; i++)
                  WinSendMsg(WinWindowFromID(hwnd, 101 + i), SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG(slider_range/2));
               for (i = 0; i < 10; i++)
                  WinSendMsg(WinWindowFromID(hwnd, 125 + i), BM_SETCHECK, MPFROMSHORT(FALSE), 0);

               WinSendMsg(WinWindowFromID(hwnd, 139), SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG(slider_range/2));

               if(WinQueryButtonCheckstate( hwnd, 121 ))
                  equalize_sound(gains, mutes, preamp, 1);
               else
                  equalize_sound(gains, mutes, preamp, 0);
               break;
         }
         break;

      case WM_CONTROL:
      {
         int id = SHORT1FROMMP(mp1);
         if(nottheuser) break;

         if (SHORT2FROMMP(mp1) == BN_CLICKED)
         {
            if (id > 124 && id < 135) /* Mute */
            {
               mutes[id - 125] = WinQueryButtonCheckstate(hwnd, id);      // left
               mutes[id - 125 + 10] = WinQueryButtonCheckstate(hwnd, id); // right

               if(WinQueryButtonCheckstate( hwnd, 121 ))
                  equalize_sound(gains, mutes, preamp, 1);
               break;
            }

            if (id == 121)
            {
               cfg.eq_enabled = WinQueryButtonCheckstate( hwnd, 121 );
               equalize_sound(gains, mutes, preamp, cfg.eq_enabled);
               break;
            }
         }

         switch (SHORT2FROMMP(mp1))
         {
            case SLN_CHANGE:
               /* Slider adjust */
               if ((id > 100 && id < 111) || id == 139)
               {
                  float g2;

                  _control87(EM_INVALID | EM_DENORMAL | EM_ZERODIVIDE | EM_OVERFLOW | EM_UNDERFLOW | EM_INEXACT, MCW_EM);
                  g2 = ((float)LONGFROMMP(mp2) - slider_range/2)/ (slider_range/2) *12; // -12 to 12 dB
                  g2 = pow(10.0, g2/20.0); // transforming into voltage gain

                  if(id == 139)
                     preamp = g2;
                  else
                  {
                     gains[SHORT1FROMMP(mp1) - 101] = g2;      // left
                     gains[SHORT1FROMMP(mp1) - 101 + 10] = g2; // right
                  }

                  if( WinQueryButtonCheckstate( hwnd, 121 ))
                     equalize_sound(gains, mutes, preamp, 1);
               }
               break;
         }
         break;
      }

      case WM_INITDLG:
         nottheuser = TRUE;

         do_warpsans(hwnd);

         if (cfg.Equalizer.x > 0 && cfg.Equalizer.y > 0)
            WinSetWindowPos(hwnd, HWND_TOP, cfg.Equalizer.x, cfg.Equalizer.y, 0, 0,
                            SWP_ACTIVATE | SWP_MOVE | SWP_SHOW | SWP_ZORDER);

         slider_range = HIUSHORT(WinSendMsg(WinWindowFromID(hwnd, 139), SLM_QUERYSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG(0)))-1;

         for (i = 101; i < 111; i++)
         {
            WinSendMsg(WinWindowFromID(hwnd, i), SLM_ADDDETENT, MPFROMSHORT(0), 0);
            WinSendMsg(WinWindowFromID(hwnd, i), SLM_ADDDETENT, MPFROMSHORT(slider_range/4), 0);
            WinSendMsg(WinWindowFromID(hwnd, i), SLM_ADDDETENT, MPFROMSHORT(slider_range/2), 0);
            WinSendMsg(WinWindowFromID(hwnd, i), SLM_ADDDETENT, MPFROMSHORT(3*slider_range/4), 0);
            WinSendMsg(WinWindowFromID(hwnd, i), SLM_ADDDETENT, MPFROMSHORT(slider_range), 0);

            WinSendMsg(WinWindowFromID(hwnd, i), SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG(slider_range/2));
         }
         WinSendMsg(WinWindowFromID(hwnd, 110), SLM_SETTICKSIZE, MPFROM2SHORT(SMA_SETALLTICKS, 0), 0);

         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_ADDDETENT, MPFROMSHORT(0), 0);
         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_ADDDETENT, MPFROMSHORT(slider_range/4), 0);
         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_ADDDETENT, MPFROMSHORT(slider_range/2), 0);
         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_ADDDETENT, MPFROMSHORT(3*slider_range/4), 0);
         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_ADDDETENT, MPFROMSHORT(slider_range), 0);
         WinSendMsg(WinWindowFromID(hwnd, 139), SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_RANGEVALUE), MPFROMLONG(slider_range/2));

         WinSendMsg(hwnd,AMP_REFRESH_CONTROLS,0,0);

         nottheuser = FALSE;

         break;

      default:
         return WinDefDlgProc(hwnd, msg, mp1, mp2);
   }
   return 0;
}

MRESULT EXPENTRY MyWinProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
  int          i;
  ULONG        o;
  static BOOL  decoder_finished = FALSE;

  switch (msg)
   {
   case WM_FOCUSCHANGE:
   {
     HPS hps = WinGetPS( hwnd );
     is_have_focus = SHORT1FROMMP( mp2 );
     bmp_draw_led( hps, is_have_focus  );
     WinReleasePS( hps );

     return WinDefWindowProc(hwnd, msg, mp1, mp2);
   }

   case 0x041e:
   {
     HPS hps = WinGetPS( hwnd );
     bmp_draw_led( hps, TRUE);
     WinReleasePS( hps );
     break;
   }
   case 0x041f:
   {
     if( !is_have_focus )
     {
       HPS hps = WinGetPS( hwnd );
       bmp_draw_led( hps, FALSE );
       WinReleasePS( hps );
     }
     break;
   }

   case WM_PLAYERROR:
     if (dec_status() == DECODER_STOPPED || !out_playing_data())
     {
        amp_stop();
        if (last_error[0] != 0)
        {
           WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, last_error, "Error detected", 0, MB_ERROR | MB_OK);
           last_error[0] = 0;
        }
     }
     break;

   case AMP_STOP:
      // The stop of the player always should be initiated from
      // the main thread. Otherwise we can receive the unnecessary
      // WM_PLAYSTOP message.
      amp_stop_playing();
      return 0;

   case AMP_DISPLAYMSG:
      WinMessageBox( HWND_DESKTOP, HWND_DESKTOP, PVOIDFROMMP(mp1), "Information from plug-in", 0, MB_INFORMATION | MB_OK );
      free( mp1 );
      break;

   case WM_CONTEXTMENU:
     amp_show_context_menu( hwnd );
     break;

   case WM_SEEKSTOP:
     is_seeking = FALSE;
     break;

   case WM_CHANGEBR:

      current_bitrate = LONGFROMMP(mp1);
      hps = WinGetPS(hwnd);
      bmp_draw_rate( hps, current_bitrate );
      WinReleasePS( hps );
      break;

   case WM_METADATA:
   {
     char *metadata = PVOIDFROMMP(mp1);
     if(metadata != NULL)
     {
        char *titlepos = strstr(metadata,"StreamTitle='")+13;
        if(titlepos != (char*)NULL+13)
        {
           char *sep;
           current_tune.title[sizeof(current_tune.title)-1] = 0;
           strncpy(current_tune.title,titlepos,sizeof(current_tune.title)-1);
           sep = strchr(current_tune.title,';');
           if(sep != NULL)
              *(sep-1) = 0; // the stupid '
           amp_display_filename();
           amp_invalidate( UPD_FILEINFO );
        }

        if(currentf != NULL)
        {
           char buf[512] = "";

           if (strcmp(current_tune.artist, "") != 0 && strcmp(current_tune.title, "") != 0)
              sprintf(buf, "%s: %s", current_tune.artist, current_tune.title);
           if (strcmp(current_tune.artist, "") == 0 && strcmp(current_tune.title, "") != 0)
              sprintf(buf, "%s", current_tune.title);
           if (strcmp(current_tune.artist, "") != 0 && strcmp(current_tune.title, "") == 0)
              sprintf(buf, "%s", current_tune.artist);

           free(currentf->songname);
           currentf->songname = strdup(buf);
           pl_refresh_record( currentf );
        }
     }
     break;
   }

   // Posted by decoder
   case WM_PLAYSTOP:
     // The decoder has finished the work, but we should wait until output
     // buffers will become empty.
     decoder_finished = TRUE;

     // If output is always hungry, WM_OUTPUT_OUTOFDATA will not be posted again
     // so we go there by our selves
     if(!out_params.always_hungry)
        break;

   // Posted by output
   case WM_OUTPUT_OUTOFDATA:
     if( decoder_finished ) {
       amp_playstop( hwnd );
     }
     decoder_finished = FALSE;
     break;

   case DM_DRAGOVER:
     return amp_drag_over( hwnd, (PDRAGINFO)mp1 );
   case DM_DROP:
     return amp_drag_drop( hwnd, (PDRAGINFO)mp1 );

   case WM_TIMER:
     if( LONGFROMMP(mp1) == TID_ONTOP ) {
        WinSetWindowPos( hframe, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER );
     }

     if( LONGFROMMP(mp1) == TID_UPDATE_PLAYER )
     {
       HPS hps = WinGetPS( hwnd );

       if( bmp_scroll_text()) {
         bmp_draw_text( hps );
       }

       if( upd_options ) {
         WinPostMsg( hwnd, AMP_PAINT, MPFROMLONG( upd_options ), 0 );
         upd_options = 0;
       }

       WinReleasePS( hps );
     }

     if( LONGFROMMP(mp1) == TID_UPDATE_TIMERS && decoder_playing()) {
       if( time_played() && cfg.mode == CFG_MODE_REGULAR )
       {
         HPS hps = WinGetPS( hwnd );
         amp_paint_timers( hps );
         WinReleasePS( hps );
       }
     }
     return (MRESULT)FALSE;

   case WM_COMMAND:
     if( COMMANDMSG(&msg)->source == CMDSRC_MENU )
     {
       if( COMMANDMSG(&msg)->cmd > IDM_M_BOOKMARKS &&
           COMMANDMSG(&msg)->cmd < IDM_M_PLUG )
       {
          process_possible_bookmark( COMMANDMSG(&msg)->cmd );
       }
       if( COMMANDMSG(&msg)->cmd > IDM_M_PLUG )
       {
          process_possible_plugin( hwnd, COMMANDMSG(&msg)->cmd );
       }
     }

     switch (COMMANDMSG(&msg) -> cmd)
   {
   case IDM_M_MANAGER:
     if (!o_manager)
      {
        o_manager = 1;
        WinDlgBox(HWND_DESKTOP, HWND_DESKTOP, PmDlgProc, NULLHANDLE, PMDLG, NULL);
        o_manager = 0;
      }
      else
      {
       if (manager_open == 2)
        {
#ifdef HIDEWINDOWS
         WinSetWindowPos(hwndManager, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_ACTIVATE);
#else
         WinSetWindowPos(hwndManager, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_RESTORE | SWP_ACTIVATE);
#endif
         manager_open = 1;
         mgrSdata.uchVisibility = SWL_VISIBLE;
         WinChangeSwitchEntry(mgrSwits, &mgrSdata);
        }
        else if(manager_open == 1)
           WinSetFocus(HWND_DESKTOP, hwndManager);
      }

     break;
   case IDM_M_LAST + 1:
   case IDM_M_LAST + 2:
   case IDM_M_LAST + 3:
   case IDM_M_LAST + 4:
   case IDM_M_LAST + 5:
   case IDM_M_LAST + 6:
   case IDM_M_LAST + 7:
   case IDM_M_LAST + 8:
   case IDM_M_LAST + 9:
   case IDM_M_LAST + 10:
      i = (COMMANDMSG(&msg)->cmd - IDM_M_LAST - 1);
      strcpy(buf, cfg.last[i]);
      for (o = i+1; o < MAX_RECALL; o++)
       strcpy(cfg.last[o-1], cfg.last[o]);
      amp_load_singlefile(buf, 0);
      break;

   case IDM_M_ADDBOOK:
      bm_add_bookmark( hwnd );
      break;

   case IDM_M_EDITBOOK:
     bm_show( TRUE );
     break;
   case IDM_M_TAG:
     amp_id3_edit( hwnd, current_filename );
     break;

   case IDM_M_SAVE:
   {
     save_stream = amp_save_stream( hwnd, !save_stream );
     break;
   }

   case IDM_M_EQUALIZE:
     if( heq == NULLHANDLE ) {
       heq = WinLoadDlg( HWND_DESKTOP, HWND_DESKTOP, wpEqualizer, NULLHANDLE, EQUALIZER, NULL );

       WinSetWindowPos( heq, HWND_TOP, cfg.Equalizer.x ,
                                       cfg.Equalizer.y ,
                                       cfg.Equalizer.cx,
                                       cfg.Equalizer.cy, SWP_SIZE | SWP_MOVE );
     }

     WinSetWindowPos( heq, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_ACTIVATE );
     WinSetFocus( HWND_DESKTOP, heq );
     break;

   case IDM_M_FLOAT:
    cfg.floatontop = !cfg.floatontop;

    if( !cfg.floatontop ) {
      WinStopTimer ( hab, hwnd, TID_ONTOP );
    } else {
      WinStartTimer( hab, hwnd, TID_ONTOP, 100 );
    }

    break;
   case BMP_PL:
    pl_show( !WinIsWindowVisible( hplaylist ));
    break;
   case IDM_M_URL:
    amp_add_url( hwnd, URL_ADD_TO_PLAYER );
    break;
   case IDM_M_TRACK:
    amp_add_tracks( hwnd, TRK_ADD_TO_PLAYER );
    break;

   case IDM_M_FONT1:
     cfg.font = 0;
     amp_invalidate( UPD_ALL );
     break;
   case IDM_M_FONT2:
     cfg.font = 1;
     amp_invalidate( UPD_ALL );
     break;
   case IDM_M_TINY:
     cfg.mode = CFG_MODE_TINY;
     bmp_reflow_and_resize(hframe);
     break;
   case IDM_M_NORMAL:
     cfg.mode = CFG_MODE_REGULAR;
     bmp_reflow_and_resize(hframe);
     break;
   case IDM_M_SMALL:
     cfg.mode = CFG_MODE_SMALL;
     bmp_reflow_and_resize(hframe);
     break;
   case IDM_M_MINIMIZE:
     WinSetWindowPos(hframe, HWND_DESKTOP, 0, 0, 0, 0, SWP_HIDE);
     WinSetActiveWindow(HWND_DESKTOP, WinQueryWindow(hwnd, QW_NEXTTOP));
     break;
   case IDM_M_SKINLOAD:
     hps = WinGetPS( hwnd );
     amp_loadskin( hab, hwnd, hps );
     WinReleasePS( hps );
     amp_invalidate( UPD_ALL );
     break;
   case IDM_M_LOADFILE:
     amp_load_file( hwnd );
     break;
   case IDM_M_CFG:
     cfg_properties( hwnd );
     break;

   case BMP_REPEAT:
     cfg.rpt = !cfg.rpt;
     if( cfg.rpt ) {
       WinSendMsg( WinWindowFromID( hwnd, BMP_REPEAT ), WM_PRESS, 0, 0 );
     } else {
       WinSendMsg( WinWindowFromID( hwnd, BMP_REPEAT ), WM_DEPRESS, 0, 0 );
     }

     amp_invalidate( UPD_FILEINFO );
     break;

   case BMP_SHUFFLE:
     if (!cfg.shf)
     {
       if (pl_size() > 0 && amp_playmode == AMP_PLAYLIST)
          pl_clean_shuffle();
       cfg.shf = TRUE;
       WinSendMsg(WinWindowFromID(hwnd, BMP_SHUFFLE), WM_PRESS, 0, 0);
     }
     else
     {
       if (amp_playmode == AMP_PLAYLIST)
          pl_clean_shuffle();
       cfg.shf = FALSE;
       WinSendMsg(WinWindowFromID(hwnd, BMP_SHUFFLE), WM_DEPRESS, 0, 0);
     }
     break;
   case IDM_M_PLAYLIST:
     pl_show( TRUE );
     break;

   case BMP_POWER:
      WinPostMsg( hwnd, WM_QUIT, 0, 0 );
      break;
   case BMP_PLAY:
     if( !decoder_playing()) {
       amp_play();
     } else {
       WinSendMsg( hwnd, WM_COMMAND, MPFROMSHORT( BMP_STOP ), mp2 );
     }
     break;

   case BMP_PAUSE:
     if( decoder_playing())
     {
       amp_msg( MSG_PAUSE, 0, 0 );
       is_paused = !is_paused;
       if( is_paused ) {
         WinSendMsg( WinWindowFromID( hwnd, BMP_PAUSE ), WM_PRESS, 0, 0 );
         break;
       }
     }

     WinSendMsg( WinWindowFromID( hwnd, BMP_PAUSE ), WM_DEPRESS, 0, 0 );
     break;

   case BMP_FLOAD:
   case BMP_N_FLOAD:
     amp_load_file(hwnd);
     break;
   case BMP_N_STOP:
   case BMP_STOP:
     amp_stop();
     pl_clean_shuffle();
     break;

   case BMP_NEXT:
      if( amp_playmode == AMP_PLAYLIST )
      {
         BOOL decoder_was_playing = decoder_playing();
         PLRECORD* rec = amp_pl_next_record();

         if( rec ) {
           if( decoder_was_playing ) {
             amp_stop();
           }
           amp_pl_load_record( rec );
           if( decoder_was_playing ) {
             amp_play();
           }
         }
      }
      break;
   case BMP_PREV:
      if (amp_playmode == AMP_PLAYLIST)
      {
         BOOL decoder_was_playing = decoder_playing();
         PLRECORD* rec = amp_pl_prev_record();

         if( rec ) {
           if( decoder_was_playing ) {
             amp_stop();
           }
           amp_pl_load_record( rec );
           if( decoder_was_playing ) {
             amp_play();
           }
         }
      }
      break;
   case BMP_FWD:
      if( decoder_playing() && !is_fast_backward && !is_paused )
      {
        amp_msg( MSG_FWD, 0, 0 );
        is_fast_forward = !is_fast_forward;
        if( is_fast_forward ) {
           WinSendMsg( WinWindowFromID( hwnd, BMP_FWD ), WM_PRESS, 0, 0 );
           amp_volume_to_lower();
        } else {
           WinSendMsg( WinWindowFromID( hwnd, BMP_FWD ), WM_DEPRESS, 0, 0 );
           amp_volume_to_normal();
        }
      } else {
        WinSendMsg( WinWindowFromID( hwnd, BMP_FWD ), WM_DEPRESS, 0, 0 );
      }
      break;

   case BMP_REW:
     if( decoder_playing() && !is_fast_forward && !is_paused )
     {
       amp_msg( MSG_REW, 0, 0 );
       is_fast_backward = !is_fast_backward;
       if( is_fast_backward ) {
         WinSendMsg( WinWindowFromID( hwnd, BMP_REW ), WM_PRESS,   0, 0 );
         amp_volume_to_lower();
       } else {
         WinSendMsg( WinWindowFromID( hwnd, BMP_REW ), WM_DEPRESS, 0, 0 );
         amp_volume_to_normal();
       }
     } else {
       WinSendMsg( WinWindowFromID( hwnd, BMP_REW ), WM_DEPRESS, 0, 0 );
     }
     break;

   default:
     break;
   }
     break;

   case WM_CREATE:
   {
     HPS hps = WinGetPS( hwnd );
     bmp_load_skin( cfg.defskin, hab, hwnd, hps );
     WinReleasePS( hps );

     if( cfg.floatontop ) {
       WinStartTimer( hab, hwnd, TID_ONTOP, 100 );
     }

     if( cfg.rpt ) {
        WinSendMsg( WinWindowFromID( hwnd, BMP_REPEAT ), WM_PRESS, 0, 0 );
     }
     if( cfg.shf || initial_shuffle ) {
       WinSendMsg( WinWindowFromID( hwnd, BMP_SHUFFLE ), WM_PRESS, 0, 0 );
       cfg.shf = TRUE;
     }

     WinStartTimer( hab, hwnd, TID_UPDATE_TIMERS, 100 );
     WinStartTimer( hab, hwnd, TID_UPDATE_PLAYER, 300 );
     break;
   }

   case WM_ERASEBACKGROUND:
     return ((MRESULT) 0);
     break;
   case WM_REALIZEPALETTE:
     vis_broadcast(msg, mp1, mp2);
     break;

   case AMP_PAINT:
   {
     HPS hps     = WinGetPS( hwnd );
     int options = LONGFROMMP( mp1 );

     if( options & UPD_TIMERS   ) {
       amp_paint_timers( hps );
     }
     if( options & UPD_FILEINFO ) {
       amp_paint_fileinfo( hps );
     }

     WinReleasePS( hps );
     return 0;
   }

   case WM_PAINT:
   {
     HPS hps = WinBeginPaint( hwnd, NULLHANDLE, NULL );

     bmp_draw_background( hps, hwnd );
     amp_paint_fileinfo ( hps );
     bmp_draw_led       ( hps, is_have_focus  );
     bmp_draw_volume    ( hps, cfg.defaultvol );

     WinEndPaint( hps );
     return 0;
   }

    case WM_BUTTON1DBLCLK:
    case WM_BUTTON1CLICK:
    {
      HPS    hps = WinGetPS( hwnd );
      POINTL pos = { SHORT1FROMMP(mp1), SHORT2FROMMP(mp1) };

      if( bmp_pt_in_volume( pos ))
      {
        bmp_draw_volume( hps, cfg.defaultvol = bmp_calc_volume( pos ));
        amp_volume_adjust();
      }
      else
      {
        if( amp_playmode != AMP_NOFILE && bmp_pt_in_text( pos )) {
          amp_display_next_mode();
          amp_invalidate( UPD_FILEINFO );
        }
      }

     WinReleasePS( hps );
     break;
   }
   case WM_MOUSEMOVE:
     if( is_volume_drag )
     {
       HPS    hps = WinGetPS( hwnd );
       POINTL pos = { SHORT1FROMMP(mp1), SHORT2FROMMP(mp1) };

       bmp_draw_volume( hps, cfg.defaultvol = bmp_calc_volume( pos ));
       WinReleasePS( hps );
       amp_volume_adjust();
     }

     if( is_slider_drag )
     {
       HPS     hps = WinGetPS( hwnd );
       POINTL  pos = { SHORT1FROMMP(mp1), SHORT2FROMMP(mp1) };
       seeking_pos = bmp_calc_time( pos, time_total());

       bmp_draw_slider( hps, seeking_pos, time_total());
       bmp_draw_timer ( hps, seeking_pos );
       WinReleasePS( hps );
     }

     WinSetPointer( HWND_DESKTOP, WinQuerySysPointer( HWND_DESKTOP, SPTR_ARROW, FALSE ));
     break;

   case WM_BUTTON1MOTIONEND:
     if( is_volume_drag ) {
       is_volume_drag = FALSE;
     }
     if( is_slider_drag )
     {
       POINTL pos = { SHORT1FROMMP(mp1), SHORT2FROMMP(mp1) };
       ULONG  ms;

       seeking_pos = bmp_calc_time( pos, time_total());
       ms = seeking_pos * 1000;

       amp_msg( MSG_JUMP, &ms, 0 );
       is_slider_drag = FALSE;
     }
     break;

   case WM_BUTTON1MOTIONSTART:
   {
      POINTL pos = { SHORT1FROMMP(mp1), SHORT2FROMMP(mp1) };

      if( bmp_pt_in_volume( pos )) {
        is_volume_drag = TRUE;
      } else if( bmp_pt_in_slider( pos ) && decoder_playing()) {
        is_slider_drag = TRUE;
        is_seeking     = TRUE;
      }
      break;
   }

   case WM_BUTTON2MOTIONSTART:
     amp_movewindow(hframe);
     WinQueryWindowPos(hframe, &cfg.Main);
     break;

   case WM_CHAR:
      if (SHORT1FROMMP(mp1) & KC_KEYUP || SHORT2FROMMP(mp2) == VK_F4) return ((MRESULT)0);

      if (!(SHORT1FROMMP(mp1) & KC_KEYUP))
      {
      if (!(SHORT1FROMMP(mp1) & KC_VIRTUALKEY) && (SHORT1FROMMP(mp1) & KC_SCANCODE) && !(SHORT1FROMMP(mp1) & KC_CHAR) && !(SHORT1FROMMP(mp1) & KC_ALT))
       {
        switch (SHORT2FROMMP(mp1) >> 8)
         {
         // 5 from the numpad
         case 76:
           WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_PAUSE, 0), 0);
           break;
         default:
           return WinDefWindowProc(hwnd, msg, mp1, mp2); break;
         }
        return (MRESULT)TRUE;
       }

      if ((SHORT1FROMMP(mp1) & KC_VIRTUALKEY) && !(SHORT1FROMMP(mp1) & KC_ALT))
       {
        switch (SHORT2FROMMP(mp2))
        {
        case VK_ESC:
         is_slider_drag = FALSE;
         is_seeking     = FALSE;
         break;
        case VK_DOWN:
         WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_NEXT, 0), 0);
         break;
        case VK_UP:
         WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_PREV, 0), 0);
         break;
        case VK_LEFT:
         WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_REW, 0), 0);
         break;
        case VK_RIGHT:
         WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_FWD, 0), 0);
         break;
        case VK_NEWLINE:
        case VK_ENTER:
         WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(BMP_PLAY, 0), 0);
         break;
        case VK_SPACE:
         amp_show_context_menu( hwnd );
         break;
        default:
         return WinDefWindowProc(hwnd, msg, mp1, mp2);
         break;
        }
        return (MRESULT)TRUE;
       }

      if (!(SHORT1FROMMP(mp1) & KC_KEYUP) && (SHORT1FROMMP(mp1) & KC_CHAR))
       {
       switch ((char) toupper((char) SHORT1FROMMP(mp2)))
        {
        case '+':
        {
         HPS hps = WinGetPS( hwnd );
         cfg.defaultvol += 5;
         if( cfg.defaultvol > 100 ) {
          cfg.defaultvol = 100;
         }
         bmp_draw_volume( hps, cfg.defaultvol );
         WinReleasePS( hps );
         amp_volume_adjust();
         break;
        }
        case '-':
        {
         HPS hps = WinGetPS( hwnd );
         cfg.defaultvol -= 5;
         if( cfg.defaultvol < 0 ) {
           cfg.defaultvol = 0;
         }
         bmp_draw_volume( hps, cfg.defaultvol );
         WinReleasePS( hps );
         amp_volume_adjust();
         break;
        }
        default:
         return WinDefWindowProc(hwnd, msg, mp1, mp2);
         break;
        }
       return (MRESULT)TRUE;
      }
     }
     return WinDefWindowProc(hwnd, msg, mp1, mp2);
     break;
   default:
     return WinDefWindowProc(hwnd, msg, mp1, mp2);
   }
   return 0;
}

PFNWP OldpFrameWndProc;

MRESULT EXPENTRY FrameWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   switch(msg) {
      case WM_DOCKWINDOW:
         if(list_search(dock_list,mp1) == NULL)
            list_add(dock_list, mp1);
         break;

      case WM_UNDOCKWINDOW:
         list_remove(dock_list, mp1);
         break;

      case WM_WINDOWPOSCHANGED:
      {
         PSWP pswpnew = (PSWP)mp1;
         SWP  swplist;

         // user does not want docking
         if(cfg.dock_windows == 0)
            break;

         // try to see if we should dock in the playlist window.
         // it seems bit 19 is set when the user is finished dragging around.
         if( LONGFROMMP(mp2) & AWP_ACTIVATE )
         {
            if( WinQueryWindowPos( hplaylist, &swplist )) {
              if( dockWindow( pswpnew, &swplist, cfg.dock_margin )) {
                WinPostMsg( hwnd, WM_DOCKWINDOW,   MPFROMHWND( hplaylist ), 0 );
              } else {
                WinPostMsg( hwnd, WM_UNDOCKWINDOW, MPFROMHWND( hplaylist ), 0 );
              }
            }

            if( WinQueryWindowPos( hbookmarks, &swplist )) {
              if( dockWindow( pswpnew, &swplist, cfg.dock_margin )) {
                WinPostMsg( hwnd, WM_DOCKWINDOW,   MPFROMHWND( hbookmarks ), 0 );
              } else {
                WinPostMsg( hwnd, WM_UNDOCKWINDOW, MPFROMHWND( hbookmarks ), 0 );
              }
            }
         }
      }

      case WM_ADJUSTWINDOWPOS:
      {
         PSWP pswpnext = (PSWP) mp1;
         SWP  swpnow;

         // user does not want docking
         if(cfg.dock_windows == 0)
            break;

         // we only want to process the MOVE messages
         if(!(pswpnext->fl & SWP_MOVE))
            break;

         if(!WinQueryWindowPos(hwnd,&swpnow))
         {
            //DosBeep(600,200);
            break;
         }

         // try to see if we are approaching the playlist window
         // if it not already docked
         if( hplaylist != NULLHANDLE && list_search( dock_list, (void*)hplaylist  ) == NULL )
         {
            SWP swplist;

            if( WinQueryWindowPos( hplaylist, &swplist )) {
              // "dock" with the playlist, but don't post WM_DOCKWINDOW
              // just yet.  do that only when the user has finished dragging
              // the window around, in WM_WINDOWPOSCHANGED
              dockWindow( pswpnext, &swplist, cfg.dock_margin );
            }
         }

         // try to see if we are approaching the playlist window
         // if it not already docked
         if( hbookmarks != NULLHANDLE && list_search( dock_list, (void*)hbookmarks ) == NULL )
         {
            SWP swplist;

            if( WinQueryWindowPos( hbookmarks, &swplist )) {
              // "dock" with the playlist, but don't post WM_DOCKWINDOW
              // just yet.  do that only when the user has finished dragging
              // the window around, in WM_WINDOWPOSCHANGED
              dockWindow( pswpnext, &swplist, cfg.dock_margin );
            }
         }

         // try to see if we have any docked windows we need to move with
         // the main frame window
         if(!list_isempty(dock_list))
         {
            SWP swpdock;
            LIST_NODE *docked_window = list_getnext(dock_list);

            while(docked_window != NULL)
            {
               if( (!WinQueryWindowPos((HWND)list_getdata(docked_window),&swpdock)) ||
                   (!WinSetWindowPos((HWND)list_getdata(docked_window),NULLHANDLE,
                                     swpdock.x+pswpnext->x-swpnow.x,
                                     swpdock.y+pswpnext->y-swpnow.y,0,0,SWP_MOVE | SWP_NOADJUST)))
               {
                  LIST_NODE *temp = list_getnext(docked_window);
                  list_remove(dock_list, list_getdata(docked_window));
                  docked_window = temp;
               }
               else
                  docked_window = list_getnext(docked_window);
            }
         }
         break;
      }
   }

   return OldpFrameWndProc( hwnd, msg, mp1, mp2 );
}

/* Stops playing and resets the player to its default state. */
void
amp_reset( void )
{
  if( decoder_playing()) {
    amp_stop();
  }

  amp_playmode     = AMP_NOFILE;
  currentf         = NULL;
  current_bitrate  = 0;
  current_channels = -1;
  current_length   = 0;
  current_freq     = 0;
  current_track    = 0;

  current_cd_drive[0] = 0;
  current_decoder_info_string[0] = 0;
  current_filename[0] = 0;

  amp_display_filename();
  amp_invalidate( UPD_ALL );
}

static USHORT
amp_message_box( HWND owner, const char* title,
                             const char* message, ULONG style  )
{
  char padded_title[60];
  sprintf( padded_title, "%-59s", title );

  if( owner == NULLHANDLE )
  {
    owner  =  HWND_DESKTOP;
    style &= ~MB_APPLMODAL;
  } else {
    style |=  MB_APPLMODAL;
  }

  return WinMessageBox( HWND_DESKTOP, owner, (PSZ)message,
                                      padded_title, 0, style );
}

/* Creates and displays a error message window.
   Use the player window as message window owner. */
void
amp_player_error( const char* format, ... )
{
  char message[4096];
  va_list args;

  va_start( args, format );
  vsprintf( message, format, args );

  amp_message_box( hframe, "PM123 Error", message,
                   MB_ERROR | MB_OK | MB_MOVEABLE );
}

/* Creates and displays a error message window.
   The specified owner window is disabled. */
void
amp_error( HWND owner, const char* format, ... )
{
  char message[4096];
  va_list args;

  va_start( args, format );
  vsprintf( message, format, args );

  amp_message_box( owner, "PM123 Error", message,
                   MB_ERROR | MB_OK | MB_MOVEABLE );
}

/* Creates and displays a message window. */
void
amp_info( HWND owner, const char* format, ... )
{
  char message[4096];
  va_list args;

  va_start( args, format );
  vsprintf( message, format, args );

  amp_message_box( owner, "PM123 Information", message,
                   MB_INFORMATION | MB_OK | MB_MOVEABLE );
}

/* Requests the user about specified action. Returns
   TRUE at confirmation or FALSE in other case. */
BOOL
amp_query( HWND owner, const char* format, ... )
{
  char message[4096];
  va_list args;

  va_start( args, format );
  vsprintf( message, format, args );

  return amp_message_box( owner, "PM123 Query", message,
                          MB_QUERY | MB_YESNO | MB_MOVEABLE ) == MBID_YES;
}

/* Requests the user about overwriting a file. Returns
   TRUE at confirmation or at absence of a file. */
BOOL
amp_warn_if_overwrite( HWND owner, const char* filename )
{
  char message[4096];
  struct stat fi;

  sprintf( message, "File %s already exists. Overwrite it?", filename );

  if( stat( filename, &fi ) == 0 ) {
    return amp_query( owner, "File %s already exists. Overwrite it?", filename );
  } else {
    return TRUE;
  }
}

int main(int argc, char *argv[])
{
  ULONG flCtlData = FCF_TASKLIST | FCF_NOBYTEALIGN | FCF_ACCELTABLE | FCF_ICON;
  char bjuf[1024];
  int i, files, o;
  APIRET rc;
  char exename[_MAX_PATH];
  char bundle [_MAX_PATH];

  // used for debug printf()s
  setvbuf(stdout,NULL,_IONBF, 0);

  hab = WinInitialize(0);
  hmq = WinCreateMsgQueue(hab, 0);

  files = count_files(argc, argv);

  strcpy( http_proxy, "" );
  strcpy( http_auth, "" );

  proxyurl = cfg.proxy;
  httpauth = cfg.auth;

  strcpy(pipename, "\\PIPE\\PM123");

  for (o = 1; o < argc; o++)
  {
   strcpy(buf, argv[o]);

   if (stricmp(buf, "-shuffle") == 0 || stricmp(buf, "/shuffle") == 0)
     initial_shuffle = 1;
   else if (stricmp(buf, "-smooth") == 0 || stricmp(buf, "/smooth") == 0)
     smooth_scroller = 1;
   else if (stricmp(buf, "-cmd") == 0 || stricmp(buf, "/cmd") == 0)
   {
    o++;
    if(strncmp(argv[o], "\\\\", 2) == 0)
    {
       strcpy(pipename, argv[o]);  // machine name
       strcat(pipename, "\\PIPE\\PM123");
       o++;
    }
    strcpy(bjuf, "*");
    for (i = o; i < argc; i++)
    {
       strcat(bjuf, argv[i]);
       strcat(bjuf, " ");
    }

    rc = DosOpen(pipename,
              &hpipe,
              &action,
              0,
              FILE_NORMAL,
              OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
              OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE | OPEN_FLAGS_FAIL_ON_ERROR,
              NULL);

    if(rc == 0)
    {
       DosWrite((HFILE) hpipe,
              bjuf,
              strlen(bjuf) + 1,
              &action);
       DosDisConnectNPipe(hpipe);
       DosClose(hpipe);
    }

    exit(0);
   }
  }


  /* If we have files in argument, try to open \\pipe\pm123 and write to it */
  if (files > 0 && DosOpen(pipename,
        &hpipe,
        &action,
        0,
        FILE_NORMAL,
        OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
        OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE | OPEN_FLAGS_FAIL_ON_ERROR,
        NULL) == 0)
  {
     // this only takes the last argument we hope is the filename
     // this should be changed
     DosWrite((HFILE) hpipe,
                argv[argc - 1],
                strlen(argv[argc - 1]) + 1,
                &action);
     DosDisConnectNPipe(hpipe);
     DosClose(hpipe);
     exit(0);
  }
  else
  {
     if(!amp_pipe_create())
        exit(1);

     _beginthread(amp_pipe_thread, NULL, 1024*1024, NULL);
  }

  getExeName( exename, sizeof( exename ));
  sdrivedir( startpath, exename );

  /* Randomize */
  srand((unsigned long) time(NULL));

  load_ini();

  /* set default volume */
  amp_volume_to_normal();

  /* Initialize Button'95 - oOoo */
  InitButton(hab);

  dock_list = list_create();

  WinRegisterClass( hab, "PM123", MyWinProc, CS_SIZEREDRAW | CS_SYNCPAINT, 0 );

  hframe = WinCreateStdWindow( HWND_DESKTOP, 0, &flCtlData, "PM123",
                               VERSION, 0, NULLHANDLE, 1, &hplayer );

  OldpFrameWndProc = WinSubclassWindow( hframe, (PFNWP)FrameWndProc );

  do_warpsans( hframe  );
  do_warpsans( hplayer );

  mp3        = WinLoadPointer( HWND_DESKTOP, 0, ICO_MP3 );
  mp3play    = WinLoadPointer( HWND_DESKTOP, 0, ICO_MP3PLAY );
  hplaylist  = pl_create();
  hbookmarks = bm_create();

  strcpy( bundle, startpath   );
  strcat( bundle, "pm123.lst" );

  if( files == 1 ) {
    amp_load_singlefile( argv[argc - 1], 0 );
  } else if( files > 1 ) {
    for( i = 1; i < argc; i++ ) {
      if( argv[i][0] != '/' && argv[i][0] != '-' ) {
        pl_add_file( argv[i], NULL, 0 );
      }
    }
    pl_completed();
  } else {
    struct stat fi;
    if( stat( bundle, &fi ) == 0 ) {
      pl_load_bundle( bundle, 0 );
    }
  }

  WinSetWindowPos( hframe, HWND_TOP,
                   cfg.Main.x, cfg.Main.y, 0, 0, SWP_ACTIVATE | SWP_MOVE | SWP_SHOW );

  for( i = 0; i < num_visuals; i++ ) {
    if( !visuals[i].skin ) {
      vis_init( hplayer, i );
    }
  }

  bm_load( hplayer );

  while( WinGetMsg( hab, &qmsg, (HWND)0, 0, 0 ))
    WinDispatchMsg( hab, &qmsg );

  amp_stop();
  pl_save_bundle( bundle, 0 );
  bm_save( hplayer );

  for( i = 0; i < num_visuals; i++ ) {
    vis_deinit( i );
  }

  if( heq != NULLHANDLE ) {
    WinDestroyWindow( heq );
  }

  save_ini();

  WinDestroyWindow( hplaylist  );
  WinDestroyWindow( hbookmarks );

  list_destroy( dock_list );
  bmp_clean_skin();
  return 0;
}
