/*
 * Copyright 1997-2003 Samuel Audet <guardia@step.polymtl.ca>
 *                     Taneli Lepp <rosmo@sektori.com>
 *
 * Copyright 2004-2005 Dmitry A.Steklenev <glass@ptv.ru>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define  INCL_WIN
#define  INCL_GPI
#define  INCL_DOS
#define  INCL_OS2MM
#include <os2.h>
#include <os2me.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <direct.h>

#include "utilfct.h"
#include "format.h"
#include "decoder_plug.h"
#include "output_plug.h"
#include "filter_plug.h"
#include "plugin.h"

#include "pm123.h"
#include "plugman.h"
#include "httpget.h"

extern OUTPUT_PARAMS out_params;

/* Returns the drive letter followed by a colon (:)
   if a drive is specified in pathname. */
char*
sdrive( char* result, const char *pathname )
{
  _splitpath((char*)pathname, result, NULL, NULL, NULL );
  return result;
}

/* Returns the path of subdirectories, if any, including
   the trailing slash.  Slashes (/), backslashes (\), or both
   may be present in pathname. */
char*
sdir( char* result, const char *pathname )
{
  _splitpath((char*)pathname, NULL, result, NULL, NULL );
  return result;
}

/* Returns the base file name without any extensions. */
char*
sfname( char* result, const char *pathname )
{
  _splitpath((char*)pathname, NULL, NULL, result, NULL );
  return result;
}

/* Returns the file name extension, if any,
   including the leading period (.). */
char*
sext( char* result, const char *pathname )
{
  _splitpath((char*)pathname, NULL, NULL, NULL, result );
  return result;
}

/* Returns the base file name with file extension. */
char*
sfnameext( char *result, const char *pathname )
{
  char fname[_MAX_FNAME];
  char ext  [_MAX_EXT  ];

  _splitpath((char*)pathname, NULL, NULL, fname, ext );
  strcpy( result, fname );
  strcat( result, ext   );
  return result;
}

/* Returns the drive letter and the path of
   subdirectories, if any, including the trailing slash.
   Slashes (/), backslashes (\), or both  may be present
   in pathname. */
char*
sdrivedir( char *result, const char *pathname )
{
  char drive[_MAX_DRIVE];
  char path [_MAX_PATH ];
  _splitpath((char*)pathname, drive, path, NULL, NULL );
  strcpy( result, drive );
  strcat( result, path  );
  return result;
}

/* Returns TRUE if the specified file is a HTTP URL. */
BOOL
is_http( const char* filename )
{
  return ( strnicmp( filename, "http://", 7 ) == 0 );
}

/* Returns TRUE if the specified file is a CD track. */
BOOL
is_track( const char* filename )
{
  return ( strnicmp( filename, "cd://"  , 5 ) == 0 );
}

/* Returns TRUE if the specified file is a regular file. */
BOOL
is_file( const char* filename )
{
  return *filename             &&
         !is_http ( filename ) &&
         !is_track( filename );
}

/* Returns TRUE if the specified directory is a root sirectory. */
BOOL
is_root( const char* path )
{
  size_t size = strlen( path );

  if( size == 3 && path[1] == ':'  && path[2] == '\\' ) {
    return TRUE;
  }
  if( size  > 3 && path[1] == '\\' && path[2] == '\\' ) {
    if( strchr( path + 2, '\\' ) == path + size - 1 ) {
      return TRUE;
    }
  }
  return FALSE;
}

/* Reads ID3 tag from the specified file. */
BOOL
amp_gettag( const char* filename, DECODER_INFO* info, tune* tag )
{
  int  handle;
  BOOL rc = FALSE;

  memset( tag, 0, sizeof( *tag ));

  if( filename != NULL && *filename && is_file( filename )) {
    handle = open( filename, O_RDONLY | O_BINARY );
    if( handle != -1 ) {
      rc = gettag( handle, tag );
      close( handle );
    }
  }

  if( !rc && info ) {
    strcpy( tag->title,   info->title   );
    strcpy( tag->artist,  info->artist  );
    strcpy( tag->album,   info->album   );
    strcpy( tag->year,    info->year    );
    strcpy( tag->comment, info->comment );
    strcpy( tag->genre,   info->genre   );
    return TRUE;
  }

  return rc;
}

/* Wipes ID3 tag from the specified file. */
BOOL
amp_wipetag( const char* filename )
{
  int  handle;
  BOOL rc = FALSE;
  struct stat fi;

  handle = open( filename, O_RDWR | O_BINARY );
  if( handle != -1 && fstat( handle, &fi ) == 0 ) {
    rc = wipetag( handle, fi.st_size );
    close( handle );
  }

  return rc;
}

/* Writes ID3 tag to the specified file. */
BOOL
amp_puttag( const char* filename, tune* tag )
{
  int  handle;
  BOOL rc = 0;

  handle = open( filename, O_RDWR | O_BINARY );
  if( handle != -1 ) {
    rc = settag( handle, tag );
    close( handle );
  }

  return rc;
}

/* Constructs a string of the displayable text from the ID3 tag. */
char*
amp_construct_tag_string( char* result, const tune* tag )
{
  *result = 0;

  if( *tag->artist ) {
    strcat( result, tag->artist );
    if( *tag->title ) {
      strcat( result, ": " );
    }
  }

  if( *tag->title ) {
    strcat( result, tag->title );
  }

  if( *tag->album && *tag->year )
  {
    strcat( result, " (" );
    strcat( result, tag->album );
    strcat( result, ", " );
    strcat( result, tag->year  );
    strcat( result, ")" );
  }
  else
  {
    if( *tag->album && !*tag->year )
    {
      strcat( result, " (" );
      strcat( result, tag->album );
      strcat( result, ")" );
    }
    if( !*tag->album && tag->year )
    {
      strcat( result, " (" );
      strcat( result, tag->year);
      strcat( result, ")" );
    }
  }

  if( tag->comment )
  {
    strncat( result, " -/- ", sizeof( result ) - strlen( result ) - 1 );
    strncat( result, tag->comment, sizeof( result ) - strlen( result ) - 1 );
  }

  return result;
}

/* Makes a menu item selectable. */
BOOL
mn_enable_item( HWND menu, SHORT id, BOOL enable )
{
  return LONGFROMMR( WinSendMsg( menu, MM_SETITEMATTR,
                                 MPFROM2SHORT( id,  TRUE ),
                                 MPFROM2SHORT( MIA_DISABLED, enable ? 0 : MIA_DISABLED )));
}

/* Places a a check mark to the left of the item. */
BOOL
mn_check_item( HWND menu, SHORT id, BOOL check )
{
  return LONGFROMMR( WinSendMsg( menu, MM_SETITEMATTR,
                                 MPFROM2SHORT( id,  TRUE ),
                                 MPFROM2SHORT( MIA_CHECKED, check ? MIA_CHECKED : 0 )));
}

/* Delete all the items in the list box. */
BOOL
lb_remove_all( HWND hwnd, SHORT id )
{
  return LONGFROMMR( WinSendDlgItemMsg( hwnd, id, LM_DELETEALL, 0, 0 ));
}

/* Deletes an item from the list box control. Returns the number of
   items in the list after the item is deleted. */
SHORT
lb_remove_item( HWND hwnd, SHORT id, SHORT i )
{
  return SHORT1FROMMR( WinSendDlgItemMsg( hwnd, id,  LM_DELETEITEM,
                                          MPFROMSHORT( i ), 0 ));
}

/* Adds an item into a list box control. */
SHORT
lb_add_item( HWND hwnd, SHORT id, const char* item )
{
  return SHORT1FROMMR( WinSendDlgItemMsg( hwnd, id, LM_INSERTITEM,
                       MPFROMSHORT( LIT_END ), MPFROMP( item )));
}

/* Sets the selection state of an item in a list box. */
BOOL
lb_select( HWND hwnd, SHORT id, SHORT i )
{
  return LONGFROMMR( WinSendDlgItemMsg( hwnd, id,  LM_SELECTITEM,
                     MPFROMSHORT( i ), MPFROMSHORT( TRUE )));
}

/* Returns the current selected item. */
SHORT
lb_selected( HWND hwnd, SHORT id )
{
  return SHORT1FROMMR( WinSendDlgItemMsg( hwnd, id, LM_QUERYSELECTION,
                                          MPFROMSHORT( LIT_CURSOR ), 0 ));
}

/* Returns a count of the number of items in the list box control. */
SHORT
lb_size( HWND hwnd, SHORT id )
{
  return SHORT1FROMMR( WinSendDlgItemMsg( hwnd, id, LM_QUERYITEMCOUNT, 0, 0 ));
}

/* Constructs a information text for currently loaded file
   and selects it for displaying. */
void
amp_display_filename( void )
{
  char display[512];

  if( amp_playmode == AMP_NOFILE ) {
    bmp_set_text( "No file loaded" );
    return;
  }

  switch( cfg.viewmode )
  {
    case CFG_DISP_ID3TAG:
      amp_construct_tag_string( display, &current_tune );

      if( *display ) {
        bmp_set_text( display );
      } else if( currentf && *currentf->songname ) {
        if( *currentf->comment ) {
          sprintf( display, "%s - %s", currentf->songname, currentf->comment );
          bmp_set_text( display );
        } else {
          bmp_set_text( currentf->songname );
        }
      }
      break;

    case CFG_DISP_FILENAME:
      if( current_cd_drive[0] != 0 && current_track != 0 )
      {
        sprintf( display, "%s\\Track %02d", current_cd_drive, current_track );
        bmp_set_text( display );
      }
      else if( current_filename != NULL && *current_filename )
      {
        if( is_http( current_filename )) {
          bmp_set_text( current_filename );
        } else {
          bmp_set_text( sfname( display, current_filename ));
        }
      } else {
         bmp_set_text( "This is a bug!" );
      }
      break;

    case CFG_DISP_FILEINFO:
      bmp_set_text( current_decoder_info_string );
      break;
  }
}

/* Switches to the next text displaying mode. */
void
amp_display_next_mode( void )
{
  if( cfg.viewmode == CFG_DISP_FILEINFO ) {
    cfg.viewmode = CFG_DISP_FILENAME;
  } else {
    cfg.viewmode++;
  }

  amp_display_filename();
}

/* Converts time to two integer suitable for display by the timer. */
void
sec2num( long seconds, int* major, int* minor )
{
  int mi = seconds % 60;
  int ma = seconds / 60;

  if( ma > 99 ) {
    mi = ma % 60; // minutes
    ma = ma / 60; // hours
  }

  if( ma > 99 ) {
    mi = ma % 24; // hours
    ma = ma / 24; // days
  }

  *major = ma;
  *minor = mi;
}

void amp_movewindow(HWND hwnd) {
   TRACKINFO TrackInfo;
   SWP Position;

   memset ( &TrackInfo, 0, sizeof(TrackInfo) ) ;

   TrackInfo.cxBorder = 1 ;
   TrackInfo.cyBorder = 1 ;
   TrackInfo.cxGrid = 1 ;
   TrackInfo.cyGrid = 1 ;
   TrackInfo.cxKeyboard = 8 ;
   TrackInfo.cyKeyboard = 8 ;

   WinQueryWindowPos ( hwnd, &Position ) ;
   TrackInfo.rclTrack.xLeft   = Position.x ;
   TrackInfo.rclTrack.xRight  = Position.x + Position.cx ;
   TrackInfo.rclTrack.yBottom = Position.y ;
   TrackInfo.rclTrack.yTop   = Position.y + Position.cy ;

   WinQueryWindowPos ( HWND_DESKTOP, &Position ) ;
   TrackInfo.rclBoundary.xLeft    = Position.x ;
   TrackInfo.rclBoundary.xRight  = Position.x + Position.cx ;
   TrackInfo.rclBoundary.yBottom = Position.y ;
   TrackInfo.rclBoundary.yTop  = Position.y + Position.cy ;

   TrackInfo.ptlMinTrackSize.x = 0 ;
   TrackInfo.ptlMinTrackSize.y = 0 ;
   TrackInfo.ptlMaxTrackSize.x = Position.cx ;
   TrackInfo.ptlMaxTrackSize.y = Position.cy ;

   TrackInfo.fs = TF_MOVE | TF_STANDARD /*| TF_ALLINBOUNDARY*/;

   WinSendMsg(hwnd, WM_TRACKFRAME, MPFROMSHORT(TrackInfo.fs), 0);
/*  if ( WinTrackRect ( HWND_DESKTOP, NULLHANDLE, &TrackInfo ) )
   {
    WinSetWindowPos ( hwnd, NULLHANDLE,
     (SHORT) TrackInfo.rclTrack.xLeft,
     (SHORT) TrackInfo.rclTrack.yBottom,
     0, 0, SWP_MOVE | SWP_ACTIVATE | SWP_SHOW) ;
   } */

}

int count_files(int argc, char *argv[]) {
 int i, o = 0;

 for (i = 1; i < argc; i++)
    if (argv[i][0] != '/' && argv[i][0] != '-') o++;

 return(o);
}

ULONG handle_dfi_error(ULONG rc, const char *file)
{
 char buf[256];

 if (rc == 0) return 0;

 *buf = '\0';

 if (rc == 100)
   sprintf(buf, "The file %s could not be read.", file);

 if (rc == 200)
   sprintf(buf, "The file %s cannot be played by PM123. The file might be corrupted or the necessary plug-in not loaded or enabled.", file);

 if (rc == 1001)
  {
   amp_stop(); /* baa */
   sprintf(buf, "%s: HTTP error occurred: %s", file, http_strerror());
  }

 if (strcmp(file, "") == 0)
  {
   sprintf(buf, "%s: Error occurred: %s", file, strerror(errno));
  }

 WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, buf, "Error", 0, MB_ERROR | MB_OK);
 return 1;
}

void _System pm123_control(int index, void *param)
{
 switch (index)
  {
   case CONTROL_NEXTMODE:
    amp_display_next_mode();
    break;
  }
}

int _System pm123_getstring(int index, int subindex, size_t bufsize, char *buf)
{
 switch (index)
  {
   case STR_NULL: *buf = '\0'; break;
   case STR_VERSION:
     strncpy(buf, VERSION, bufsize);
     break;
   case STR_DISPLAY_TEXT:
     strncpy( buf, bmp_query_text(), bufsize );
     break;
   case STR_FILENAME:
     strncpy(buf, current_filename, bufsize);
     break;
   default: break;
  }
 return(0);
}


