/**************************************
 * shell.c : pty shells for pcucp
 *
 * Copyright (C) 1992 Jouni Lepp{j{rvi
 **************************************/

#include <sys/types.h>
#include <fcntl.h>
#include <pwd.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <utmp.h>
#include <sys/wait.h>

#define SYS_INCL_STDLIB
#include "system.h"
#include "usrmsg.h"
#include "ptyopen.h"
#include "cttyopen.h"
#include "tty.h"
#include "utmpmod.h"
#include "shell.h"

/*
 * Any flavor of non-blocking
 * io will do, as long as it 
 * just is available.
 *
 * The primary option should
 * actually be O_NONBLOCK 
 * (posix), but since this
 * does not work under Ultrix
 * in the master-pty context,
 * O_NDELAY is used.
 */

#ifndef O_NDELAY
#ifdef  FNDELAY
#define O_NDELAY FNDELAY
#else
#ifdef  O_NONBLOCK
#define O_NDELAY O_NONBLOCK
#endif
#endif
#endif

#ifndef SIGCHLD
#ifdef  SIGCLD /* sysV native form */
#define SIGCHLD SIGCLD
#endif
#endif

#define SHELL_SHELLS 32 /* arbitary limit */

#define SHELL_SFREE   0
#define SHELL_SACTIVE 1
#define SHELL_SDEAD   2

typedef struct  
{
  int state;
  int pid;
  int rd, wr;
  int ch;
  char tty[sizeof(((struct utmp *) NULL) -> ut_line)];
} SHELL;

SHELL _shellTable[SHELL_SHELLS];

#if SYS_PROTOS
static int ShellCreate(SHELL sdata);
static void SigChld(void);
#endif

static int ShellCreate(sdata)
SHELL *sdata;
{
  int i, fd, pid;
  char *tty, *shell;

  if (PtyOpen(&fd,&tty))
    {
      MsgError("ShellCreate","pty open failed");
      return(-1);
    }

  if (pid = fork())
    {
      if (pid < 0)
    {
      MsgError("ShellCreate","fork failed");
      return(-1);
    }

      sdata -> pid = pid;
      sdata -> rd = sdata -> wr = fd;
      strncpy(sdata -> tty,&tty[5],sizeof(sdata -> tty) - 1);
      sdata -> tty[sizeof(sdata -> tty) - 1] = '\0';

      i = fcntl(fd,F_GETFL);
      fcntl(fd,F_SETFL,i | O_NDELAY);
    
      return(0);
    }
  
  /* what is the elegant way ? */
   
  for (fd = 3; fd < 256 ;fd++)
    close(fd);

  /* open the slave tty as control tty, ensure sane mode */

  if (CttyOpen(tty))
    {
      MsgDisplay(MSG_ERROR,"unexpected : cannot open as control tty : %s",tty);
      _exit(0);
    }

  TtySane();

  /* use the shell defined in the environment, if not possible use /bin/sh */

  shell = getenv("SHELL");
  if (shell == NULL || access(shell,1))
    shell = "/bin/sh";

  execlp(shell,shell,NULL);

  /* cannot display error here - the error fd has been closed */

  _exit(0); /* exit() includes c-lib cleanups (not wanted here) */
}

static void SigChld()
{
  int st, pid, i;

  /*
   * waitpid() or wait3() would be more clever here, but
   * wait() is more portable and should work most of the time.
   * In the rare case when a child chooses to die while this
   * handler is executing this event might be missed.
   */

  pid = wait(&st);

  for (i = 0; i < SHELL_SHELLS ;i++)
    if (_shellTable[i].state == SHELL_SACTIVE && _shellTable[i].pid == pid)
      {
    /*
     * logic : if a child dies due to channel close it
     * has been marked inactive already. If it dies
     * due to exit/signal it is still active and marked
     * dead and will be closed next time the read function
     * is called.
     */

    _shellTable[i].state = SHELL_SDEAD;
    break;
      }
  signal(SIGCHLD,SigChld);
}

int ShellOpen(ch)
int ch;
{
  static int initFlg = 0;
  int i;
  struct passwd *pwd;

  if (!initFlg)
    {
      initFlg = 1;
      signal(SIGCHLD,SigChld);
    }

  for (i = 0; i < SHELL_SHELLS && _shellTable[i].state != SHELL_SFREE ;i++);
    
  if (i < SHELL_SHELLS)
    {
      if (!ShellCreate(&_shellTable[i]))
    {
      _shellTable[i].state = SHELL_SACTIVE;
      _shellTable[i].ch = ch;

      pwd = getpwuid(getuid());
      if (pwd == NULL || !pwd -> pw_name[0] || 
          UtmpSetEntry(_shellTable[i].tty,pwd -> pw_name,"",time(NULL)))
        MsgError("ShellCreate","/etc/utmp modification failed");
      
      return(i);
    }
    }
  return(-1);
}

void ShellClose(desc)
int desc;
{
  if (UtmpSetEntry(_shellTable[desc].tty,"","",0L))
    MsgError("ShellCreate","/etc/utmp modification failed");
 
  close(_shellTable[desc].rd); /* -> SIGHUP */
  close(_shellTable[desc].wr);

  _shellTable[desc].state = SHELL_SFREE;
}

int ShellRead(desc,buf,len)
int desc;
char *buf;
int len;
{
  int r, t;

  if (_shellTable[desc].state != SHELL_SACTIVE)
    {
      if (_shellTable[desc].state == SHELL_SDEAD)
    ChanClose(_shellTable[desc].ch);
      else
    MsgDisplay("ShellRead","unexpected : invalid state");
      return(0);
    }

  /* The following loop is required by Ultrix (under load?) :
   *
   * Each time a process write()s even one character to
   * the slave pty, a task switch occurs. Hence, if a
   * process outputs a string a character at a time,
   * each character ends up to be transfered as a packet
   * of its own -> severe performance hit. Re-reading the
   * pty seems to cause another task switch and this way
   * the entire string ends up in one packet which is
   * more economical.
   */

  for (t = 0; t < len ;t += r)
    {
      r = read(_shellTable[desc].rd,&buf[t],len - t);
      if (r <= 0)
    break; 
    }

  return(t);
}

void ShellWrite(desc,buf,len)
int desc;
char *buf;
int len;
{
  write(_shellTable[desc].wr,buf,len);
}

