unit Scrlcapt;

interface

uses
  {$IfDef Win32} Windows, {$Else} WinTypes, WinProcs, {$EndIf}
  Classes, Controls, Forms, Messages;

type
  TScrollDirection = (dLeft, dRight);
  TScrollingCaption = class(TComponent)
  private
    FCaption, FSpace: String;
    ParentForm: TForm;
    TmpCount: Integer;
    FIsMainWindow: Boolean;
    FDirection: TScrollDirection;
    FEnabled: Boolean;
    FInterval: Word;
    FWindowHandle: HWND;
    FOnProcessCaption: TNotifyEvent;
    procedure UpdateProcessCaption;
    procedure SetCaption(Value: String);
    procedure SetEnabled(Value: Boolean);
    procedure SetInterval(Value: Word);
    procedure SetOnProcessCaption(Value: TNotifyEvent);
    procedure WndProc(var Msg: TMessage);
  protected
    procedure ProcessCaption; dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Caption: String read FCaption write SetCaption;
    property Direction: TScrollDirection read FDirection write FDirection;
    property IsMainWindow: Boolean read FIsMainWindow write FIsMainWindow;
    property Space: String read FSpace write FSpace;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Interval: Word read FInterval write SetInterval default 200;
    property OnProcessCaption: TNotifyEvent read FOnProcessCaption write SetOnProcessCaption;
  end;

procedure Register;

implementation

constructor TScrollingCaption.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FEnabled := True;
  FInterval := 200;
  FWindowHandle := AllocateHWnd(WndProc);
  FSpace := '     ';

  ParentForm := TForm(aOwner);
  try
   FCaption := ParentForm.Caption;
  except
  end;
  TmpCount := 1;
  UpdateProcessCaption;
end;

destructor TScrollingCaption.Destroy;
begin
  Enabled := False;
  UpdateProcessCaption;
  DeallocateHWnd(FWindowHandle);
  inherited Destroy;
end;

procedure TScrollingCaption.WndProc(var Msg: TMessage);
begin
  with Msg do
    if Msg = WM_TIMER then
      try
        ProcessCaption;
      except
        Application.HandleException(Self);
      end
    else
      Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

procedure TScrollingCaption.UpdateProcessCaption;
begin
  KillTimer(FWindowHandle, 1);
  if (FInterval <> 0) and FEnabled then
    if SetTimer(FWindowHandle, 1, FInterval, nil) = 0 then
      raise EOutOfResources.Create('No timers');
end;

procedure TScrollingCaption.SetCaption(Value: String);
begin
  if FCaption <> Value then
   begin
    FCaption := Value;
    ParentForm.Caption := Value;
    if FIsMainWindow then
     Application.Title := Value;
    TmpCount := 1;
    UpdateProcessCaption;
   end;
end;

procedure TScrollingCaption.SetEnabled(Value: Boolean);
begin
  if Value <> FEnabled then
  begin
    FEnabled := Value;
    ParentForm.Caption := FCaption;
    if FIsMainWindow then
     Application.Title := FCaption;
    TmpCount := 1;
    UpdateProcessCaption;
  end;
end;

procedure TScrollingCaption.SetInterval(Value: Word);
begin
  if Value <> FInterval then
  begin
    FInterval := Value;
    UpdateProcessCaption;
  end;
end;

procedure TScrollingCaption.SetOnProcessCaption(Value: TNotifyEvent);
begin
  FOnProcessCaption := Value;
  UpdateProcessCaption;
end;

procedure TScrollingCaption.ProcessCaption;
var
  St: String;
begin
  try
   St := FCaption + FSpace;
   ParentForm.Caption := Copy(St, TmpCount, Length(St) - TmpCount + 1) + Copy(St, 1, TmpCount - 1);
   if FIsMainWindow then Application.Title := ParentForm.Caption;
   if Direction = dLeft then
    begin
     inc(TmpCount);
     if TmpCount > Length(St) then TmpCount := 1;
    end
   else
    begin
     dec(TmpCount);
     if TmpCount = 0 then TmpCount := Length(St);
    end;
  except
  end;
  if Assigned(FOnProcessCaption) then FOnProcessCaption(Self);
end;

procedure Register;
begin
  RegisterComponents('UtilMind', [TScrollingCaption]);
end;

end.
