/* mputz4p.c  -  Decoding image in Z4Plane format to HRAM buffer,
 *               in form of 4 planes. (for UNCHAINED mode).
 *
 * Description:
 *
 *   Uses the following coding method:
 *   Each image contains 4 arrays (one per plane).
 *   Each array has non-determined length, and terminated by row counter.
 *   Line ends by a 0, 0.
 *   Image is coded by the groups:
 *
 *     n pixels literally       (n = 0 - 127)
 *     +-----+---------------------------+
 *     | +n  |  n bytes of pixel values  |
 *     +-----+---------------------------+
 *
 *     n pixels of specified color (n = 3 - 255)
 *     +-----+-----+--------+
 *     |  0  |  n  | color  |
 *     +-----+-----+--------+
 *
 *     skipping n pixels  (n = 1 - 127)
 *     +-----+
 *     | -n  |
 *     +-----+
 *
 *     EOL
 *     +-----+
 *     | 128 |
 *     +-----+
 *
 *   Decoding is directed into the animation buffer, specified by a
 *   segment address (starting offset is assumed to be 0).
 *   Plane interleaving for each buffer is specified by the argument,
 *   given in paragraphs.
 *   Width in bytes of each plane in buffer also must be specified, to
 *   provide the proper increment in case of EOL.
 *
 * Functions:
 *      MputZ4Planes().
 * Portability: BORLANDC
 *                                      (c) erdy 1992
 * $Header: $
 */
#pragma inline
#include "far.h"
#include "vgaprefx.h"
#include "vgadrv.h"
#include "screen.h"

#ifndef Z4P_RAMWIDTH
void MputZ4Planes(char far *image,    /* image data ptr */
                  int rows,           /* rows to process */
                  unsigned segment,   /* address segment of planes buffer */
                  unsigned offset,    /* starting offset */
                  unsigned planewidth,/* width of plane buffer,
                                       * MUST BE equal to Npixels/4 */
                  unsigned char planeintl/* plane buffer interleaving, in paragraphs */
                  )
#else   Z4P_RAMWIDTH
void MputZ4Planes(char far *image,    /* image data ptr */
                  int rows,           /* rows to process */
                  unsigned segment,   /* address segment of planes buffer */
                  unsigned offset     /* starting offset */
                  )
#       ifndef Z4P_RAMHEIGHT
#               error Z4P_RAMHEIGHT must be defined
#       endif  Z4P_RAMHEIGHT
#endif  Z4P_RAMWIDTH
{
        /* Charge the registers */
        _ES = segment;
        _DI = offset;                   /* Offset of this image */
#       ifndef Z4P_RAMWIDTH
                _BX = rows;
                _DX = planewidth;
#       else   Z4P_RAMWIDTH
                _DX = rows;
#       endif  Z4P_RAMWIDTH

#       ifndef Z4P_RAMHEIGHT
                _AX = planeintl;
#       endif  Z4P_RAMHEIGHT

	asm push ds;
	asm lds si, image;
	asm cld;
        asm push bp;
        asm mov cx, BITPLANES;
PlaneLoop:
        /*
         * Planes loop. CX contains plane counter.
         */
        asm push cx;                    /* Save plane counter */
        asm push di;                    /* Save the initial offset */

#       ifndef Z4P_RAMWIDTH
                asm mov  cx, bx;        /* And store row counter in cx */
                asm push bx;            /* Save initial row counter */
                asm push ax;            /* Save plane interleaving */
#       else   Z4P_RAMWIDTH
                asm mov  cx, dx;        /* And store row counter in cx */
#       endif  Z4P_RAMWIDTH

                /*++++same+++as++uputz4p.c+++*/
RowLoop:
                /*
                 * Scanlines loop. CX contains nrows.
                 * Scratch registers:
                 *      bx - saving the ram index (di)
                 *      bp - saving the loop counter (cx)
                 */
                asm mov bx, di;                 /* Save ram index */
                asm mov bp, cx;                 /* Save loop counter */
                asm xor ch, ch;                 /* Zero high byte, ch always zero */
                /* Loop until EOL */
                asm jmp short LineLoop;         /* Jump over the literal-dealing code */
                                                /* This tricky is for optimizing jumps */
Literal:
                        asm rep movsb;          /* Copy cx bytes */
LineLoop:
			asm lodsb;              /* al <- ds:[si++]; get next byte */
                        asm mov cl, al;         /* Move code */

                        asm or al, al;          /* Query flags */
                        asm jg Literal;         /* al > 0, so literally */
                        asm je short String;    /* al == 0, string or EOL */
Skip:                                           /* Otherwise skipping */
                        asm neg cl;             /* Invert counter */
                        asm jo Eol;             /* Overflow, i.e. neg 128 */
                        asm add di, cx;         /* Advance the index */
                        asm jmp short LineLoop;
String:
                        asm lodsb;              /* Get counter byte */
			asm mov cl, al;		/* Put counter to cl */
                        asm lodsb;              /* Get color byte into al */
                        asm rep stosb;          /* Store cx bytes of al */
                        asm jmp short LineLoop;
Eol:
                /*--same--as--uputz4p.c---*/
                asm mov di, bx;                 /* Restore ram index */
#               ifndef Z4P_RAMWIDTH
                        asm add di, dx;         /* Advance ram index to next row */
#               else   Z4P_RAMWIDTH
                        asm add di, Z4P_RAMWIDTH SHR 2;/* Advance ram index to next row */
#               endif  Z4P_RAMWIDTH

                asm mov cx, bp;                 /* Restore row counter */
                asm loop RowLoop;

#       ifndef Z4P_RAMHEIGHT
                asm pop ax;
                asm mov bx, es;
                asm add bx, ax;
#       else   Z4P_RAMHEIGHT
                asm mov bx, es;
                asm add bx, (Z4P_RAMHEIGHT*(Z4P_RAMWIDTH SHR 2)) SHR 4;
#       endif  Z4P_RAMHEIGHT
        asm mov es, bx;         /* Advance the ram buffer segment register */

#       ifndef Z4P_RAMWIDTH
                asm pop bx;     /* Restore initial row counter */
#       endif  Z4P_RAMWIDTH

        asm pop di;             /* Restore the initial offset */

        asm pop cx;
        asm loop PlaneLoop;

        asm pop bp;
	asm pop ds;
	return;
}