(*****************************************************************************

  Drives
    version 1.1

  This unit contains a collection of drive accessing routines.

  Features:
    Use of characters as drive identifiers instead of the harder to remember
      drive number values.
    Use of the operating system where-ever possible instead of relying on
      lower level BIOS routines.
    Routines ensure the preservation of current system drive status.

  Limitations:
    Somewhat limited error checking.

  Versions:
    1.0  -  Created as a help unit for the install program.
    1.01  -  Updated code to remember result of Check_System_Drives.
    1.1  -  Updated code to compile via Speed Pascal/2 for OS/2.

  Copyright 1995, 1996, P. Renaud.
    All rights reserved.

  Compilers:
    Turbo Pascal versions 4.0 through 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2.

*****************************************************************************)

Unit Drives;

  Interface

    Uses
      DOS;

(***********************************************************

  Drive_Set_Type - Defines the result type of the system
    census.

***********************************************************)

    Type
      Drive_Set_Type = Set of 'A' .. 'Z';

(***********************************************************

  Procedure:  Check system drives.

    This function scans the entire system for existing
    drives and returns a set containing letters only for the
    drives that exist.  The procedure makes sure to return
    the system drive back to it's origional directory.

***********************************************************)

    Procedure Check_System_Drives( Var Drives: Drive_Set_Type );

(***********************************************************

  Function: Disk fixed.

    This function returns true if the specified disk is a
    fixed disk as opposed to removeable medium.
    ( Currently not supported with Speed Pascal/2 )

***********************************************************)

    Function Disk_Fixed( Drive: Char ): Boolean;

(***********************************************************

  Function:  Get current drive.

    This function returns the current system disk drive.

***********************************************************)

    Function Get_Current_Drive: Char;

(***********************************************************

  Procedure:  Set current drive.

    This procedure sets the default system drive to the
    given character value.

***********************************************************)

    Procedure Set_Current_Drive( New_Drive: Char );

(***********************************************************

  Procedure: Call DOS about drive.

    This procedure inquires to the disk operating system
    about the existance about a particular disk drive in two
    steps.  First it attempts to set the default drive to
    the particular drive, then it polls the system to see if
    the operation was successful.  Exist is returned as true
    if it exists, otherwise it is set to false.

***********************************************************)

    Procedure Call_DOS_About_Drive( Drive: Char; Var Exist: Boolean );

(***********************************************************

  Function: Does this drive exist?

    This function returns true if the drive of the given
    character exists, otherwise it returns false.

***********************************************************)

    Function Does_This_Drive_Exist( Drive: Char ): Boolean;

(***********************************************************

  Function:  Convert drive to number

    This function returns a byte value indicating the
    appropriate drive to the given character.

***********************************************************)

    Function Drive_Number( Drive: Char ): Byte;

{----------------------------------------------------------------------------}

  Implementation

    Var
     { Used to hold the results of the system poll. }
      Hold_System_Drives: Drive_Set_Type;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Convert drive to number
    As previously defined.

*************************************************)

    Function Drive_Number( Drive: Char ): Byte;
      Begin
        Drive := UpCase( Drive );
        Drive_Number := ( Ord( Drive ) - Ord( 'A' ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Convert drive to letter
    This function takes a drive as a number and
    converts it into a letter.

*************************************************)

    Function Drive_Letter( Drive: Byte ): Char;
      Begin
        Drive_Letter := Chr( Drive + Ord( 'A' ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Get current drive.
    As previously defined.

*************************************************)

    Function Get_Current_Drive: Char;
     {$IFNDEF OS2}
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $19;
        MSDOS( The_Registers );
        Get_Current_Drive := Drive_Letter( The_Registers.Al );
      End;
     {$ELSE}
      Var
        Data: String;
      Begin
        GetDir( 0, Data );
        Get_Current_Drive := Data[ 1 ];
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Set current drive.
    As previously defined.

*************************************************)

    Procedure Set_Current_Drive( New_Drive: Char );
     {$IFNDEF OS2}
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $E;
        The_Registers.Dl := Drive_Number( New_Drive );
        MSDOS( The_Registers );
      End;
     {$ELSE}
      Var
        Directory: String;
      Begin
        GetDir( Drive_Number( New_Drive ), Directory );
        ChDir( Directory );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Call DOS about drive.
    As previously defined.

*************************************************)

    Procedure Call_DOS_About_Drive( Drive: Char; Var Exist: Boolean );
     {$IFNDEF OS2}
      Var
        Hold: Byte;
        The_Registers: Registers;
      Begin
        Hold := Drive_Number( Drive );
        The_Registers.Ah := $E;
        The_Registers.DL := Hold;
        MSDOS( The_Registers );
        The_Registers.Ah := $19;
        MSDOS( The_Registers );
        Exist := ( The_Registers.AL = Hold );
      End;
     {$ELSE}
      Begin
        Exist := ( DiskSize( Drive_Number( Drive ) ) <> -1 );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Disk fixed.
    As previously defined.

*************************************************)

    Function Disk_Fixed( Drive: Char ): Boolean;
     {$IFNDEF OS2}
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AH := $15;
        The_Registers.DL := Drive_Number( Drive );
        Intr( $13, The_Registers );
        Disk_Fixed := ( The_Registers.AH = 3 );
      End;
     {$ELSE}
      Begin
        Disk_Fixed := ( Drive in [ 'C'..'Z' ] );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Does this drive exist?
    As previously defined.

*************************************************)

    Function Does_This_Drive_Exist( Drive: Char ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( ( Drive >= 'A' ) and ( Drive <= 'Z' ) );
        If Okay
          then
            Call_DOS_About_Drive( Drive, Okay );
        Does_This_Drive_Exist := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Check system drives.
    As previously defined.

*************************************************)

    Procedure Check_System_Drives( Var Drives: Drive_Set_Type );
      Var
        Search_Drive: 'A' .. 'Z';
        Current_Drive: Char;
      Begin
        If ( Hold_System_Drives = [] )
          then
            Begin
              Current_Drive := Get_Current_Drive;
              Drives := [];
              For Search_Drive := 'A' to 'Z' do
                If Does_This_Drive_Exist( Search_Drive )
                  then
                    Drives := Drives + [ Search_Drive ];
              Set_Current_Drive( Current_Drive );
              Hold_System_Drives := Drives;
            End
          else
            Drives := Hold_System_Drives;
      End;

{----------------------------------------------------------------------------}

  Begin
    Hold_System_Drives := [];
  End.

