/*
 *  Routine for processing the keyboard input.
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "freq.h"
#include "fft.h"
#include "extern.h"
#include "display.h"

int display_peak=0;     /* Flag for displaying the peak information */
int decay_mode=0;       /* Flag indicating decay mode on/off */
int help_mode=0;        /* Flag indicating help mode 0 (off), 1, or 2 */
int bw_mode=0;          /* Flag indicating a Black/White palette */
int freeze=0;           /* Flag indicating a paused display */
float freq_scalefactor=1.0;  /* Scalefactor for increasing the horizonal scale */
float freq_base=0.0;    /* Base frequency for the display */
int bin=0;              /* Bin number for cursor display */
int mic_level=0;        /* Storage for mic input mixer level (0-100) */
int ext_level=0;        /* Storage for ext input mixer level (0-100) */
int int_level=0;        /* Storage for int input mixer level (0-100) */

#ifdef DOS
void cleanup_vga(void)
{
   /*
    *  Clean up VGA registers so Borland libraries work again.
    */
   outportb(0x3c4,2);
   outportb(0x3c5,0x0f);   /* Use all bit planes */
}

void setup_vga(void)
{
   /*
    *  Set up VGA registers for fast graphics display
    */
   outportb(0x3ce,1);
   outportb(0x3cf,0);   /* Disable set/reset for all planes */
   outportb(0x3ce,3);
   outportb(0x3cf,0);   /* No rotate, just write data */
   outportb(0x3ce,4);
   outportb(0x3cf,3);   /* Read from bit plane 3 */
   outportb(0x3ce,5);
   outportb(0x3cf,inportb(0x3cf)&0x70); /* Read and write direct to memory */
   outportb(0x3ce,8);
   outportb(0x3cf,0xff);  /* Allow modification of all bits */
   outportb(0x3c4,2);
   outportb(0x3c5,12);   /* Use bit planes 2 and 3 */
}
#endif


void input_text(char *text,int maxlen,int x,int y)
{
   /*
    *  Input a line of text using graphical display at (x,y)
    *  Maximum length allowed for input is given by maxlen
    *  The input begins with the current contents of text, so this must
    *  be initialized before calling the routine.
    */
   int c=0;
   int count;
   char ach[3];
   ach[1]='_';
   ach[2]=0;

   draw_text_left(x,y,text);
   count=strlen(text);
   x+=count*_font_width;
   draw_text_left(x,y,"_");

   while((c != 0x0d) && (c != 0x0a))
   {
      c=draw_getch();

      if(((c==0x08)||(c==0x7f)) && count)
      {
	 count--;
	 x-=_font_width;
	 draw_bar(x,y-1,x+2*_font_width,y+_font_height,0);
	 draw_text_left(x,y,"_");
      }
      else if(count<(maxlen-1))
      {
	 if(((c>='0') && (c<='9')) || ((c>='A') && (c<='Z')) || ((c>='a') && (c<='z')) || (c=='.') || (c=='\\') || (c=='/') || (c==':'))
	 {
	    draw_bar(x,y-1,x+_font_width,y+_font_height,0);

	    ach[0]=c;
	    text[count]=c;
	    draw_text_left(x,y,ach);
	    count++;
	    x+=_font_width;
	 }
      }
      if(c==0x1b)
      {
	 count=0;
	 break;
      }
   }
   text[count]=0;
}

void update_display(void)
{
   /*  Update the frequency and amplitude information displayed while */
   /*  the input is frozen. */

   char ach[100];
   double re,im;
   double amp,db,phase;

   if(decay_mode)
   {
      re=displayval[bin]/16.0;
      im=0;
   }
   else
   {
      int bri=BitReversed[bin];
      re=fftdata[bri];
      im=fftdata[bri+1];
   }

   amp=sqrt(re*re+im*im)/32768.0;

   if(gain3db)
      amp*=sqrt((double)bin*(double)SampleRate/fftlen/(double)ref_freq);

   if(deriv==1)
      amp*=(double)SampleRate/(2*M_PI*(double)ref_freq);

   if(deriv==2)
      amp*= (double)SampleRate/(2*M_PI*(double)ref_freq)
	   *(double)SampleRate/(2*M_PI*(double)ref_freq);

   if(amp!=0)
   {
      phase=atan2(im,re)*2*M_PI;
      db=20*log10(amp);
   }
   else
   {
      phase=0;
      db=-100;
   }

   draw_bar(0,21,160,71,0);
   sprintf(ach," Freq: %7.5g Hz ",(double)bin*SampleRate/fftlen);
   draw_text_left(0,22,ach);
   sprintf(ach,"  Amp: %7.5g   ",amp);
   draw_text_left(0,34,ach);
   sprintf(ach,"       %7.5g db ",db);
   draw_text_left(0,46,ach);
   sprintf(ach,"Phase: %7.4g deg. ",phase);
   draw_text_left(0,58,ach);
}


void highlight(int on)
{
   /*  Turn highlighting of a specified bin on the display on (1) or off (0) */

   int pos;
   double freq=(double)bin*(double)SampleRate/fftlen;
   if(logfreq)
      pos=floor(log(freq/freq_base)/log(fftlen/2)*freq_scalefactor*(double)(WINDOW_RIGHT-WINDOW_LEFT)+0.5);
   else
      pos=floor((freq-freq_base)*(double)(WINDOW_RIGHT-WINDOW_LEFT)/(double)SampleRate*freq_scalefactor*2.0+0.5);

   if(on)
      draw_line(WINDOW_LEFT+pos,WINDOW_TOP,WINDOW_LEFT+pos,lasty[pos],DARK_HIGHLIGHT);
   else
      draw_line(WINDOW_LEFT+pos,WINDOW_TOP,WINDOW_LEFT+pos,lasty[pos],0);
   if(on)
      draw_line(WINDOW_LEFT+pos,lasty[pos],WINDOW_LEFT+pos,WINDOW_BOTTOM,LIGHT_HIGHLIGHT);
   else
   {
      if(barmode)
         draw_line(WINDOW_LEFT+pos,lasty[pos],WINDOW_LEFT+pos,WINDOW_BOTTOM,GRAPH_COLOR);
      else
      {
         draw_line(WINDOW_LEFT+pos,lasty[pos],WINDOW_LEFT+pos,WINDOW_BOTTOM,0);
         draw_pixel(WINDOW_LEFT+pos,lasty[pos],GRAPH_COLOR);
      }
   }
}

void reset_decay_buffers(void)
{
   int i,j;
   decay_ptr=0;
   decay_count_actual=0; /* Restart the count of number of samples taken */
   memset(lastval,0,fftlen/2*sizeof(long));
   memset(lastval_f,0,fftlen/2*sizeof(float));
   for(i=0;i<max_decay_count;i++)
   {
      for(j=0;j<fftlen/2;j++)
	 databuf[i][j]=0;
   }
}

int process_input(int c,int repetitions)
{
   /*  Process any keyboard input received by the program */
   char ach[50];
   int i;

   if(c==0) return(0);

   if(freeze) highlight(0);

   if(c==' ')  /* Toggle freeze mode on/off */
   {
      freeze=!freeze;
      if(freeze)
      {
	 if(help_mode)
	 {
	    help_mode=0;
	    update_header();
	 }
	 if((double)bin<freq_base*fftlen/SampleRate)
	    bin=ceil(freq_base*fftlen/SampleRate);
	 if((double)bin>maxfreq*fftlen/SampleRate)
	    bin=floor(maxfreq*fftlen/SampleRate);

	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");

	 update_display();
      }
      else
      {
	 draw_bar(0,21,160,71,0);
	 draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	 reset_decay_buffers();
      }
   }
   if(freeze)
   {
      /* Keys only valid in freeze-display */
      if((c==0x0d)||(c==0x0a))  /* <CR> saves the spectrum data to a file */
      {
	 char filename[30];
	 draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	 draw_text_left(MGX-156,MGY,"File in which to save this spectrum:");

	 filename[0]=0;
	 input_text(filename,sizeof(filename),MGX+140,MGY);
	 if(filename[0]!=0)
	 {
	    clock_t clk;
	    FILE *fp=fopen(filename,"w");
	    if(fp)
	    {
	       for(i=0;i<fftlen/2;i++)
	       {
		  double re,im;
		  if(!decay_mode)
		  {
		     re=fftdata[BitReversed[i]]/32768.0;
		     im=fftdata[BitReversed[i]+1]/32768.0;
		  }
		  else
		  {
		     re=displayval[i]/524388.0;
		     im=0;
		  }
		  if(gain3db)
		  {
		     re*=sqrt((double)bin*(double)SampleRate
                             /(double)fftlen/(double)ref_freq);
		     im*=sqrt((double)bin*(double)SampleRate
                             /(double)fftlen/(double)ref_freq);
		  }
		  if(deriv==1)
		  {
		     re*=(double)SampleRate/(2*M_PI*(double)ref_freq);
		     im*=(double)SampleRate/(2*M_PI*(double)ref_freq);
		  }
		  if(deriv==2)
		  {
		     re*= (double)SampleRate/(2*M_PI*(double)ref_freq)
		         *(double)SampleRate/(2*M_PI*(double)ref_freq);
		     im*= (double)SampleRate/(2*M_PI*(double)ref_freq)
		         *(double)SampleRate/(2*M_PI*(double)ref_freq);
		  }
		  fprintf(fp,"%g\t%g\t%g\n",(double)i/(double)fftlen
                                     *(double)SampleRate,re,im);
	       }
	       fclose(fp);
	       sprintf(ach,"Spectrum data saved in %s",filename);
	    }
	    else
	       sprintf(ach,"Unable to open (%s)",filename);

	    draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	    draw_text_centered(MGX,MGY,ach);

	    clk=clock();
	    while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
	 }
	 draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");
      }
      if(c=='J' || c=='j')  /* Move the cursor one bin left */
      {
	 double current;
         bin-=repetitions;
         if(bin<=0)
         {
            if(logfreq)
               bin=1;
            else
               bin=0;
         }
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current<freq_base)
	 {
	    freq_base=current;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='K' || c=='k') /* Move the cursor one bin right */
      {
	 double current;
         bin+=repetitions;
	 if(bin>=fftlen/2)
	    bin=fftlen/2-1;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current>maxfreq)
	 {
	    if(logfreq)
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	    else
	       freq_base=current-(double)SampleRate/(freq_scalefactor*2.0);
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='H' || c=='h') /* Move the cursor 10 bins left */
      {
	 double current;
	 bin-=10*repetitions;
         if(bin<=0)
         {
            if(logfreq)
               bin=1;
            else
               bin=0;
         }
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current<freq_base)
	 {
	    freq_base=current;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='L' || c=='l') /* Move the cursor 10 bins right */
      {
	 double current;
	 bin+=10*repetitions;
	 if(bin>=fftlen/2)
	    bin=fftlen/2-1;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current>maxfreq)
	 {
	    if(logfreq)
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	    else
	       freq_base=current-(double)SampleRate/(freq_scalefactor*2.0);
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c==HOME) /* Move the cursor to the lowest frequency */
      {
	 if(logfreq)
	    bin=1;
	 else
	    bin=0;
	 update_display();
	 if((logfreq && (freq_base>((double)SampleRate/(double)fftlen)))
           || (!logfreq && (freq_base>0)))
	 {
	    freq_base=0;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
      else if(c==END) /* Move the cursor to the highest frequency */
      {
	 bin=fftlen/2-1;
	 update_display();
	 if(maxfreq<(double)SampleRate/2)
	 {
	    freq_base=SampleRate/2;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
   }
   else
   {
      /* Keys only valid in non-freeze display */
      if((c=='l') || (c=='L'))  /* Log the FFT data to a file */
      {
         extern FILE *fp_fftlog;  /* Defined in passdata.c */
         clock_t clk;

	 draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
         if(fp_fftlog)
         {
            fclose(fp_fftlog);
            fp_fftlog=NULL;
      	    draw_text_centered(MGX,MGY,"FFT logging stopped");
	    clk=clock();
	    while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
   	    draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
         }
         else
         {
            char filename[30];
	    draw_text_left(MGX-156,MGY,"File in which to log the FFT data:");
  	    filename[0]=0;
	    input_text(filename,sizeof(filename),MGX+116,MGY);
            if(filename[0]!=0)
	    {
	       clock_t clk;
	       fp_fftlog=fopen(filename,"w");
	       if(fp_fftlog)
               {
	          sprintf(ach,"FFT data being logged to %s",filename);
              	  draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	          draw_text_centered(MGX,MGY,ach);
                  fprintf(fp_fftlog,"Samping rate:\t%lu\n",SampleRate);
                  fprintf(fp_fftlog,"FFT length:\t%d\n",fftlen);
                  fprintf(fp_fftlog,"Window:\t%s\n",window_name[windfunc]);
                  fprintf(fp_fftlog,"Gain:\t%d dB/octave\n",deriv*6);
                  fprintf(fp_fftlog,"Ext:%d\tCD:%d\tMic:%d\n\n",ext_level,int_level,mic_level);
               }
  	       else
               {
	          sprintf(ach,"Unable to open (%s)",filename);
                  clk=clock();
	          while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
 	          draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
               }
            }
            else
               draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
         }
      }
      else if(c==HOME) /* Move the cursor to the lowest frequency */
      {
	 if(logfreq && (freq_base>((double)SampleRate/(double)fftlen)))
	 {
	    /* Re-initialize the display */
	    freq_base=0;
	    setup_xscale();
	 }
	 else if(!logfreq && (freq_base>0))
	 {
	    /* Re-initialize the display */
	    freq_base=0;
	    setup_xscale();
	 }
      }
      else if(c==END) /* Move the cursor to the highest frequency */
      {
	 if(maxfreq<(double)SampleRate/2)
	 {
	    freq_base=SampleRate/2;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
      else if(c=='H' || c=='h' || c=='?' || c=='/') /* Toggle help mode */
      {
	 help_mode++;
	 if(help_mode>2) help_mode=0;

	 if(help_mode)
	    show_help();
	 else
	    update_header();
      }
      else if(c=='F' || c== 'f') /* Change the FFT length */
      {
	 halt_soundcard();
	 EndFFT();
	 if(c=='f')
	 {
            for(i=0;i<repetitions;i++)
            {
  	       fftlen*=2;
	       if(fftlen>MAX_LEN) fftlen=8;
            }
	 }
	 else
	 {
            for(i=0;i<repetitions;i++)
            {
	       fftlen/=2;
	       if(fftlen<8) fftlen=MAX_LEN;
            }
	 }
	 InitializeFFT(fftlen);
	 compute_window_function();
	 /* Flush the buffers */
	 reset_decay_buffers();
	 for(i=0;i<fftlen/2;i++)
	    displayval[i]=0;
	 for(i=0;i<fftlen;i++)
	    fftdata[i]=0;

	 /* Re-initialize the display */
	 setup_xscale();
	 if(help_mode)
	    show_help();
	 else
	    update_header();
	 reset_soundcard();
      }
      else if(c=='R' || c=='r') /* Change the sampling rate */
      {
	 char in[6];
	 draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	 draw_text_left(MGX-80,MGY,"Sampling rate: ");

	 in[0]=0;
	 input_text(in,sizeof(in),MGX+40,MGY);
	 if(in[0]!=0)
	 {
	    halt_soundcard();

	    SampleRate=atol(in);
	    if(SampleRate<5000L) SampleRate=5000L;
	    if(SampleRate>88200L) SampleRate=88200L;

	    reset_soundcard();

	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 if(help_mode)
	    show_help();
	 else
	    update_header();
	 reset_decay_buffers();
      }
      else if(c=='Y' || c=='y') /* Toggle linear/logarithmic amplitude axis */
      {
	 /*
	  *  Toggle between linear and logarithmic amplitude scale
	  */
	 logamp=!logamp;

	 if(logamp)
	    setup_logscales();
	 else
	    setup_linscales();

	 amplitude_scale();
      }
   }

   /* Keys valid in both freeze and non-freeze mode */

   if((c==UP_ARROW) || (c=='8'))
   {  /* Increase the vertical scale factor */
      if(logamp)
      {
	 if(log_base<10)
	 {
            logs+=repetitions;
            log_base+=repetitions;
            if(log_base>10)
            {
               logs-=(log_base-10);
               log_base=10;
            }
	    setup_logscales();
	 }
      }
      else
      {
         double last_ys=ys;
         for(i=0;i<repetitions;i++)
         {
            if(ys>0.15)
            {
               ys-=0.1;
            }
            else if(ys>0.01)
            {
               ys-=0.01;
               if(ys<0.01) ys=0.01;
            }
         }
         if(ys!=last_ys)
	    setup_linscales();
      }
      amplitude_scale();
   }
   else if((c==DOWN_ARROW) || (c=='2'))
   {  /* Decrease the vertical scale factor */
      if(logamp)
      {
	 if(logs>0)
	 {
	    logs-=repetitions;
	    log_base-=repetitions;
            if(logs<0)
            {
               log_base-=logs;
               logs=0;
            }
	    setup_logscales();
	 }
      }
      else
      {
         double last_ys=ys;
         for(i=0;i<repetitions;i++)
         {
            if(ys<0.095)
            {
               ys+=0.01;
            }
            else if(ys<2.00)
            {
               ys+=0.1;
               if(ys>2.0) ys=2.0;
            }
         }
         if(ys!=last_ys)
            setup_linscales();
      }

      amplitude_scale();
   }
   else if((c==CTRL_UP) || (c==ALT_UP) || (c==PG_UP) || (c=='9'))
   {  /* Shift the vertical scales up */
      if(logamp)
      {
	 if(log_base<10)
	 {
	    log_base+=repetitions;
            if(log_base>10) log_base=10;
	    setup_logscales();
	 }
	 else if(logs>0)
	 {
	    logs-=repetitions;
            if(logs<0) logs=0;
	    setup_logscales();
	 }
      }
      else
      {
         double last_ys=ys;
         for(i=0;i<repetitions;i++)
         {
            if(ys>0.15)
            {
               ys-=0.1;
            }
            else if(ys>0.01)
            {
               ys-=0.01;
               if(ys<0.01) ys=0.01;
            }
         }
         if(ys!=last_ys)
            setup_linscales();
      }
      amplitude_scale();
   }
   else if((c==CTRL_DOWN) || (c==ALT_DOWN) || (c==PG_DN) || (c=='3'))
   {  /* Shift the vertical scales down */
      if(logamp)
      {
	 if(log_base>logs+3)
	 {
	    log_base-=repetitions;
            if(log_base<=logs+3) log_base=logs+3;
	    setup_logscales();
	 }
      }
      else
      {
         double last_ys=ys;
         for(i=0;i<repetitions;i++)
         {
            if(ys<0.095)
            {
               ys+=0.01;
            }
            else if(ys<2.00)
            {
               ys+=0.1;
               if(ys>2.0) ys=2.0;
            }
         }
         if(ys!=last_ys)
            setup_linscales();
      }
      amplitude_scale();
   }
   else if((c=='<') || (c==CTRL_LEFT) || (c==ALT_LEFT))
   {  /* Decrease the horizontal scale factor */
      if(logfreq<2)
      {
	 if(freq_scalefactor>1.0)
	 {
            for(i=0;(freq_scalefactor>1.0)&&(i<repetitions);i++)
            {
               freq_scalefactor*=0.84089641525371;  /* 1/(2^0.25) */
               if(logfreq)
                  freq_base*=exp(log(fftlen/2)*0.07955179237314/freq_scalefactor);
               else
                  freq_base-=(double)SampleRate*0.03977589618657/freq_scalefactor;
            }
	    /* Re-initialize the display */
	    if(freeze)
	    {
	       double current=(double)bin*(double)SampleRate/(double)fftlen;
	       xrange_check();
	       if(current<freq_base)
	       {
		  freq_base=current;
	       }
	       if(current>maxfreq)
	       {
		  if(logfreq)
		     freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
		  else
		     freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	       }
	    }
	    setup_xscale();
	 }
      }
   }
   else if((c=='>') || (c==CTRL_RIGHT) || (c==ALT_RIGHT))
   {  /* Increase the horizontal scale factor */
      if(logfreq<2)
      {
	 if(freq_scalefactor<16.0)
	 {
            for(i=0;(freq_scalefactor<16.0)&&(i<repetitions);i++)
            {
               if(logfreq)
                  freq_base*=exp(log(fftlen/2)*0.07955179237314/freq_scalefactor);
               else
                  freq_base+=(double)SampleRate*0.03977589618657/freq_scalefactor;
               freq_scalefactor*=1.18920711500272;  /* 2^0.25 */
            }

	    /* Re-initialize the display */
	    if(freeze)
	    {
	       double current=(double)bin*(double)SampleRate/(double)fftlen;
	       xrange_check();
	       if(current<freq_base)
	       {
		  freq_base=current;
	       }
	       if(current>maxfreq)
	       {
		  if(logfreq)
		     freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
		  else
		     freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	       }
	    }
	    setup_xscale();
	 }
      }
   }
   else if((c==',') || (c==LEFT_ARROW) || (c=='4'))
   {  /* Shift the horizontal display to the right */
      if(logfreq<2)
      {
	 if(maxfreq<(double)SampleRate/2)
	 {
            for(i=0;(i<repetitions)&&(maxfreq<(double)SampleRate/2);i++)
            {
               if(logfreq)
                  freq_base*=exp(log(fftlen/2)/(8.0*freq_scalefactor));
               else
                  freq_base+=(double)SampleRate/(16.0*freq_scalefactor);
               xrange_check();
            }

	    /* Re-initialize the display */
	    if(freeze)
	    {
	       double current=(double)bin*(double)SampleRate/(double)fftlen;
	       if(current<freq_base)
	       {
		  bin=ceil(freq_base/(double)SampleRate*(double)fftlen);
		  update_display();
	       }
	    }
	    setup_xscale();
	 }
      }
   }
   else if((c=='.') || (c==RIGHT_ARROW) || (c=='6'))
   {  /* Shift the horizontal display to the left */
      if(logfreq<2)
      {
	 if((logfreq && (freq_base>((double)SampleRate/(double)fftlen)))
	    || (!logfreq && (freq_base>0)))
	 {
            for(i=0;i<repetitions;i++)
            {
               if((logfreq && (freq_base>((double)SampleRate/(double)fftlen)))
                  || (!logfreq && (freq_base>0)))
               {
                  if(logfreq)
                     freq_base/=exp(log(fftlen/2)/(8.0*freq_scalefactor));
                  else
                     freq_base-=(double)SampleRate/(16.0*freq_scalefactor);
               }
	       xrange_check();
            }
	    /* Re-initialize the display */
	    if(freeze)
	    {
	       double current=(double)bin*(double)SampleRate/(double)fftlen;
	       if(current>maxfreq)
	       {
		  bin=floor(maxfreq/(double)SampleRate*(double)fftlen);
		  if(bin>=fftlen/2) bin=fftlen/2-1;
		  update_display();
	       }
	    }
	    setup_xscale();
	 }
      }
   }
   else if(c=='P' || c=='p') /* Toggle peak display mode on/off */
   {
      display_peak=!display_peak;
      if(display_peak)
      {
	 draw_text_left(PKX-64,PKY,"Peak at         Hz");
      }
      else
      {
	 draw_bar(PKX-64,PKY-1,PKX+79,PKY+_font_height,0);
      }
   }
   else if(c=='V' || c=='v') /* Redraw the video display */
   {
      draw_bar(0,0,639,479,0);

      /* Refresh the video display */

      draw_rectangle(WINDOW_LEFT-2,WINDOW_TOP-2,WINDOW_RIGHT+2,WINDOW_BOTTOM+2,BORDER_COLOR);
      setup_xscale();
      amplitude_scale();
      if(help_mode)
	 show_help();
      else
	 update_header();
      if(display_peak)
      {
	 draw_text_left(PKX-64,PKY,"Peak at         Hz");
      }
      if(freeze)
      {
	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");
	 update_display();
      }
      /* Reset the last-value contents */
      for(i=0;i<WINDOW_RIGHT-WINDOW_LEFT;i++)
	 lasty[i]=WINDOW_BOTTOM;
   }
   else if(c=='B' || c=='b') /* Toggle between line and bar display */
   {
      barmode=!barmode;

      draw_bar(WINDOW_LEFT,WINDOW_TOP,WINDOW_RIGHT,WINDOW_BOTTOM,0);
      /* Reset the last-value contents */
      for(i=0;i<WINDOW_RIGHT-WINDOW_LEFT;i++)
	 lasty[i]=WINDOW_BOTTOM;
   }
   else if(c=='S' || c=='s') /* Save the current state to an INI file */
   {
      draw_bar(WINDOW_LEFT,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
      draw_text_left(MGX-130,MGY,"Save the current state to: ");
      strncpy(ach,ini_file,20);
      input_text(ach,20,MGX+86,MGY);
      if(ach[0]!=0)
      {
	 FILE *fp;
	 strcpy(ini_file,ach);
	 fp=fopen(ini_file,"w");
	 if(fp)
	 {
	    fprintf(fp,"Soundcard: %d\n",Soundcard);
            #ifdef UNIX
	       fprintf(fp,"Audio device: %s\n",audio_device);
            #endif /* UNIX */
            #ifdef LINUX_MIXER
	       fprintf(fp,"Mixer device: %s\n",mixer_device);
            #endif /* LINUX_MIXER */
	    fprintf(fp,"Sample rate: %ld\n",SampleRate);
            fprintf(fp,"Rate fudge factor: %g\n",fudgefactor);
	    fprintf(fp,"FFT length: %d\n",fftlen);
	    fprintf(fp,"Window function: %d\n",windfunc);
            fprintf(fp,"Bar display: %d\n",barmode);
	    fprintf(fp,"Log freq scale: %d\n",logfreq);
	    fprintf(fp,"Log amp scale: %d\n",logamp);
	    fprintf(fp,"Base db: %d\n",log_base);
	    fprintf(fp,"Top db: %d\n",logs);
	    fprintf(fp,"Max amp: %g\n",ys);
	    fprintf(fp,"DB/octave gain: %d\n",gain);
	    fprintf(fp,"Reference frequency: %ld\n",ref_freq);
	    fprintf(fp,"Base frequency: %g\n",freq_base);
	    fprintf(fp,"Frequency factor: %g\n",freq_scalefactor);
	    fprintf(fp,"Decay mode: %d\n",decay_mode);
	    fprintf(fp,"Decay factor: %g\n",decay_factor);
	    fprintf(fp,"Averaging count: %d\n\n",decay_count);
	    fprintf(fp,"Background color: %d,%d,%d\n",background.red,background.green,background.blue);
	    fprintf(fp,"Clipping warning color: %d,%d,%d\n",warn.red,warn.green,warn.blue);
	    fprintf(fp,"Graph color: %d,%d,%d\n",graph.red,graph.green,graph.blue);
	    fprintf(fp,"Tick mark color: %d,%d,%d\n",tick.red,tick.green,tick.blue);
	    fprintf(fp,"Axis label color: %d,%d,%d\n",label.red,label.green,label.blue);
	    fprintf(fp,"Border color: %d,%d,%d\n",border.red,border.green,border.blue);
	    fprintf(fp,"Text color: %d,%d,%d\n",text.red,text.green,text.blue);
	    fprintf(fp,"Cursor upper color: %d,%d,%d\n",darkhl.red,darkhl.green,darkhl.blue);
	    fprintf(fp,"Cursor lower color: %d,%d,%d\n",lighthl.red,lighthl.green,lighthl.blue);
	    fclose(fp);
	 }
	 else
	 {
	    clock_t clk=clock();
	    sprintf(ach,"Unable to open %s",ini_file);
	    draw_bar(WINDOW_LEFT-5,MGY-1,WINDOW_RIGHT+5,MGY+_font_height,0);
	    draw_text_centered(MGX,MGY,ach);
	    while(!draw_getkey() && ((clock()-clk)<3*CLOCKS_PER_SEC));
	 }
      }
      if(help_mode)
	 show_help();
      else
	 update_header();
      if(freeze)
      {
	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");
	 update_display();
      }
      else
	 reset_decay_buffers();
   }
   else if((c=='-') || (c=='_')) /* Increase decay rate */
   {
      if((decay_mode==1) || (decay_mode==2))
      {
         for(i=0;i<repetitions;i++)
   	    decay_factor=decay_factor*decay_factor;
	 if(decay_factor<0.0001) decay_factor=0.0001;
      }
      else if(decay_mode==3)
      {
         for(i=0;i<repetitions;i++)
         {
            if(!decay_count)
               decay_count=MAX_DECAY_COUNT;
            else
               decay_count/=2;
            if(decay_count<1) decay_count=1;
         }
	 reset_decay_buffers();
      }
      if(!help_mode)
      {
	 draw_bar(DCX-150,DCY-1,DCX+150,DCY+_font_height,0);
	 if(decay_mode==0)
	    sprintf(ach,"No averaging");
	 else if(decay_mode==1)
	    sprintf(ach,"Step up/decay averaging (%g)",decay_factor);
	 else if(decay_mode==2)
	    sprintf(ach,"Exponential averaging (%g)",decay_factor);
	 else
         {
            if(decay_count)
               sprintf(ach,"Uniform averaging (%2d/%2d)",decay_count_actual,decay_count);
            else
               sprintf(ach,"Uniform averaging (%5d)",decay_count_actual);
         }
	 draw_text_centered(DCX,DCY,ach);
      }
   }
   else if((c=='+') || (c=='=')) /* Decrease decay rate */
   {
      if((decay_mode==1) || (decay_mode==2))
      {
         for(i=0;i<repetitions;i++)
   	    decay_factor=sqrt(decay_factor);
	 if(decay_factor>0.99999) decay_factor=0.99999;
      }
      else if(decay_mode==3)
      {
         for(i=0;i<repetitions;i++)
            decay_count*=2;
         if(decay_count>max_decay_count) decay_count=0;
	 reset_decay_buffers();
      }
      if(!help_mode)
      {
	 draw_bar(DCX-150,DCY-1,DCX+150,DCY+_font_height,0);
	 if(decay_mode==0)
	    sprintf(ach,"No averaging");
	 else if(decay_mode==1)
	    sprintf(ach,"Step up/decay averaging (%g)",decay_factor);
	 else if(decay_mode==2)
	    sprintf(ach,"Exponential averaging (%g)",decay_factor);
	 else
         {
            if(decay_count)
               sprintf(ach,"Uniform averaging (%2d/%2d)",decay_count_actual,decay_count);
            else
               sprintf(ach,"Uniform averaging (%5d)",decay_count_actual);
         }
	 draw_text_centered(DCX,DCY,ach);
      }
   }
   else if(c=='A' || c=='a') /* Select averaging mode */
   {
      if(c=='A')
      {
	 decay_mode--;
	 if(decay_mode<0) decay_mode=3;
      }
      else
      {
	 decay_mode++;
	 if(decay_mode>3) decay_mode=0;
      }

      /* Re-initialize the buffers */
      reset_decay_buffers();

      if(!help_mode)
      {
	 draw_bar(DCX-150,DCY-1,DCX+150,DCY+_font_height,0);
	 if(decay_mode==0)
	    sprintf(ach,"No averaging");
	 else if(decay_mode==1)
	    sprintf(ach,"Step up/decay averaging (%g)",decay_factor);
	 else if(decay_mode==2)
	    sprintf(ach,"Exponential averaging (%g)",decay_factor);
	 else
         {
            if(decay_count)
               sprintf(ach,"Uniform averaging (%2d/%2d)",decay_count_actual,decay_count);
            else
               sprintf(ach,"Uniform averaging (%5d)",decay_count_actual);
         }
	 draw_text_centered(DCX,DCY,ach);
      }
   }
   else if(c=='C' || c=='c') /* Toggle Black/White palette on/off */
   {
      bw_mode=!bw_mode;
      if(bw_mode)
	 setbwpalette();
      else
	 setnormalpalette();
   }
   else if(c=='X' || c=='x') /* Toggle linear/logarithmic frequency axis */
   {
      /*
       *  Toggle between linear and logarithmic and equalizer frequency scale
       */
      logfreq++;
      if(logfreq>2) logfreq=0;
      if(logfreq==2)
      {
	 /* In equalizer mode, force shift to full display */
	 freq_base=(double)SampleRate/(double)fftlen;
	 freq_scalefactor=1.0;
      }
      if(freeze)
      {
	 double current;
	 if(logfreq && (bin==0)) bin=1;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 xrange_check();
	 if(current<freq_base)
	 {
	    freq_base=current;
	 }
	 if(current>maxfreq)
	 {
	    if(!logfreq)
	       freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	    else
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	 }
      }
      setup_xscale();
   }
   else if(c=='W' || c=='w') /* Advance to the next windowing function */
   {
      /*
       *  Change the windowing function
       */
      if(c=='w')
      {
	 windfunc+=repetitions;
	 while(windfunc>6) windfunc-=7;
      }
      else
      {
	 windfunc-=repetitions;
	 while(windfunc<0) windfunc+=7;
      }
      compute_window_function();

      if(!help_mode)
      {
	 sprintf(ach,"%s window function",window_name[windfunc]);
	 draw_bar(WFX-150,WFY-1,WFX+150,WFY+_font_height,0);
	 draw_text_centered(WFX,WFY,ach);
      }
   }
   else if((c=='G') || (c=='g')) /* Change the db/octave gain */
   {
      if(c=='G')
      {
	 gain-=3*repetitions;
	 while(gain<0) gain+=15;
      }
      else
      {
	 gain+=3*repetitions;
	 while(gain>12) gain-=15;
      }
      deriv=gain/6;
      gain3db=(gain/3)-deriv*2;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      /* Re-initialize the buffers */
      reset_decay_buffers();

      if(!help_mode)
      {
	 draw_bar(DGX-160,DGY-1,DGX+160,DGY+_font_height,0);
	 sprintf(ach,"Gain of %d dB per octave (ref: %ld Hz)",gain,ref_freq);
	 draw_text_centered(DGX,DGY,ach);
      }
   }
   else if(mixers)
   {
      if(c=='[') /* External jack down */
      {
	 ext_level-=2*repetitions;
	 if(ext_level<0) ext_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",ext_level);
	    draw_bar(LVX+104,LVY-1,LVX+127,LVY+_font_height,0);
	    draw_text_left(LVX+104,LVY,ach);
	 }
	 set_mixer(MIXER_EXT,ext_level);
      }
      else if(c==']') /* External jack up */
      {
	 ext_level+=2*repetitions;
	 if(ext_level>100) ext_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",ext_level);
	    draw_bar(LVX+104,LVY-1,LVX+127,LVY+_font_height,0);
	    draw_text_left(LVX+104,LVY,ach);
	 }
	 set_mixer(MIXER_EXT,ext_level);
      }
      else if(c=='{') /* CD input down */
      {
	 int_level-=2*repetitions;
	 if(int_level<0) int_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY-1,LVX+191,LVY+_font_height,0);
	    draw_text_left(LVX+168,LVY,ach);
	 }
	 set_mixer(MIXER_INT,int_level);
      }
      else if(c=='}') /* CD input up */
      {
	 int_level+=2*repetitions;
	 if(int_level>100) int_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY-1,LVX+191,LVY+_font_height,0);
	    draw_text_left(LVX+168,LVY,ach);
	 }
	 set_mixer(MIXER_INT,int_level);
      }
      else if(c=='(') /* Mic input down */
      {
	 mic_level-=2*repetitions;
	 if(mic_level<0) mic_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY-1,LVX+55,LVY+_font_height,0);
	    draw_text_left(LVX+32,LVY,ach);
	 }
	 set_mixer(MIXER_MIC,mic_level);
      }
      else if(c==')') /* Mic input up */
      {
	 mic_level+=2*repetitions;
	 if(mic_level>100) mic_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY-1,LVX+55,LVY+_font_height,0);
	    draw_text_left(LVX+32,LVY,ach);
	 }
	 set_mixer(MIXER_MIC,mic_level);
      }
   }

   if(freeze) highlight(1);

   return(c=='E' || c=='e' || c=='Q' || c=='q' || c==0x03);
}
