/*
 *  A standard set of graphics calls, defined on different machines
 *  to make the proper calls for that machine
 *
 *  The following functions are required:
 *     draw_init(void)  
 *       Inititialize system, called immediately after program starts
 *     setup_graphics(void)
 *       Enter the graphics mode
 *     cleanup_graphics(void)
 *       Exit the graphics mode, and return to text mode
 *     draw_fontcolor(int c)
 *       Set the font color to 'c'
 *     draw_text_left(int x,int y,char *string)
 *       Write the text string with its upper left corner = (x,y)
 *     draw_text_right(int x,int y,char *string)
 *       Write the text string with its upper right corner = (x,y)
 *     draw_text_centered(int x,int y,char *string)
 *       Write the text string with its upper edge centered on (x,y)
 *     draw_rectangle(int x1,int y1,int x2,int y2,int c)
 *       Draw a rectangle from (x1,y1) to (x2,y2), lines having color 'c'
 *     draw_line(int x1,int y1,int x2,int y2,int c)
 *       Draw a line from (x1,y1) to (x2,y2), having color 'c'
 *     draw_bar(int x1,int y1,int x2,int y2,int c)
 *       Draw a filled rectangle from (x1,y1) to (x2,y2) with color 'c'
 *     draw_setpalette(int n,int r,int g,int b)
 *       Set the paletted color number n to (r,g,b) where max(r)=63
 *     int draw_getch(void)
 *       Wait for a key press, and return the key value
 *     int draw_getkey(void)
 *       Check for key press, and return the key value if there, else return 0
 */

#ifdef UNIX

 #ifdef SRGP
  /*
   *  Definitions for the SRGP library for Linux (X windows)
   */
  #include "srgp.h"

  void draw_init(void);
  void setup_graphics(void);
  void cleanup_graphics(void);
  void draw_fontcolor(int c);
  void draw_text_left(int x,int y,char *string);
  void draw_text_centered(int x,int y,char *string);
  void draw_text_right(int x,int y,char *string);
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  void draw_pixel(int x,int y,int c);
  void draw_line(int x1,int y1,int x2,int y2,int c);
  void draw_bar(int x1,int y1,int x2,int y2,int c);
  void draw_setpalette(int n,int r,int g,int b);
  int draw_getch(void);
  int draw_getkey(void);

 #else

  /*
   *  Definitions for the SVGALIB library for Linux
   */ 
  #include <vga.h>
  #include <vgagl.h>

  #define GMODE G640x480x256

  #define draw_init() vga_init()
  void setup_graphics(void);
  #define cleanup_graphics() vga_setmode(TEXT)
  void draw_fontcolor(int c);
  #define draw_text_left(x,y,string) gl_write(x,y,string)
  #define draw_text_centered(x,y,string) \
	  gl_write((x)-strlen(string)*_font_width/2,y,string)
  #define draw_text_right(x,y,string) \
	  gl_write((x)-strlen(string)*_font_width,y,string)
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  #define draw_pixel(x,y,c) gl_setpixel(x,y,c)
  #define draw_line(x1,y1,x2,y2,c) gl_line(x1,y1,x2,y2,c)
  #define draw_bar(x1,y1,x2,y2,c) gl_fillbox(x1,y1,(x2)-(x1)+1,(y2)-(y1)+1,c)
  #define draw_setpalette(n,r,g,b) gl_setpalettecolor(n,r,g,b)
  #define draw_getch() vga_getch()
  int draw_getkey();
 #endif

#else

  /*
   *  Definitions for the Borland BGI library for DOS
	*/
  #include <conio.h>
  #include <graphics.h>

  int ctrlcfunc(void);
  #define draw_init() ctrlbrk(ctrlcfunc)
  void setup_graphics(void);
  #define cleanup_graphics() closegraph()
  void draw_fontcolor(int c);
  void draw_text_left(int x,int y,char *string);
  void draw_text_centered(int x,int y,char *string);
  void draw_text_right(int x,int y,char *string);
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  #define draw_pixel(x,y,color) putpixel(x,y,color)
  void draw_line(int x1,int y1,int x2,int y2,int c);
  void draw_bar(int x1,int y1,int x2,int y2,int c);
  #define draw_setpalette(n,r,g,b) setrgbpalette(n,r,g,b)
  int draw_getch(void);
  int draw_getkey(void);
#endif
