/*  8086.H

    Author:      Patrick Whittle
    Date:        July, 1994

    Revisions:   October 3, 1994
                 February 24, 1995

    Various 8086 CPU assembler calls, including mouse calls (int 33h),
    DOS int 21h calls, and direct video functions are included in this
    file.  Borland's inline assembly is used in some functions.

*/

#if !defined( __DOS_H )
#include <dos.h>                   /* int86 uses this file */
#endif
#if !defined( __STDARG_H )
#include <stdarg.h>
#endif
#if !defined(__CONIO_H)
#include <conio.h>
#endif
#if !defined(__STRING_H)
#include <string.h>
#endif

#define VIDEO          0x10
#define DOSCALL        0x21
#define GETVEC         0x35  /* Get interrupt vector function of int 21h */
#define PRINTCHR       0x2
#define MOUSE          0x33
#define RESET          0x21
#define SHOW           0x1
#define HIDE           0x2
#define MOUSEPOS       0x4
#define PRESS          0x5
#define RELEASE        0x6
#define MONO           0xB000
#define COLOUR         0xB800
#if !defined(TRUE)
#define TRUE           -1
#endif
#if !defined(FALSE)
#define FALSE          0
#endif

/* Function prototypes are next */
int videoAddress(void);
int scanvector(char *str, int segval, int offval);

/***************************************************************************
Function  : sendpkt
Parameters: Input is a pointer to the string to be sent on the ethernet
Returns   : A TRUE or FALSE condition for success or failure is returned

***************************************************************************/
int sendpkt(char *packet, int len, int intno) {

    int stat = TRUE;
    union REGS regs;

    regs.h.ah = 4;
    regs.x.cx = len;
    regs.x.si = (int)packet;

    int86(intno, &regs, &regs);

    if(regs.x.cflag)    /* If there's an error sending, return FALSE.  */
	stat = FALSE;

    return(stat);
}
/***************************************************************************
Function  : drvrchk
Parameters: No input required
Returns   : A TRUE or FALSE condition

      This function will search for the first available interrupt between
      between 0x60 and 0x80 and stop after first is found.  If no packet
      driver is found in memory, drvrchk returns a FALSE condition.  If
      a driver is found, drvrchk returns the interrupt it is using.

***************************************************************************/
int drvrchk(void) {

    int flag = FALSE, segment, offptr, intnum;
    char *pktstr = "PKT DRVR";

    asm	push   es      /* current interrupt handler returned in es:bx */
    asm	push   bx

    for(intnum = 0x60; intnum <= 0x80; intnum++) {
	asm {
	    mov    ah, GETVEC
	    mov    al, BYTE PTR intnum      /* get interupt vector */
	    int    DOSCALL
	    mov    WORD PTR segment, es
	    mov    WORD PTR offptr, bx
	}
	offptr += 3;                        /* move 3 bytes into segment */

	flag = scanvector(pktstr, segment, offptr);

	if(flag) {          /* If TRUE condition resulted - driver found. */
	    flag = intnum;  /* flag is returned to be used as int number */
	    intnum = 0x81;  /* We want to exit loop now */
	}
    }
    asm pop    bx
    asm pop    es

    return(flag);
}
/***************************************************************************
Function  : scanvector
Parameters: assumes ES segment as the destination for compare
Returns   : TRUE or FALSE condition

***************************************************************************/
int scanvector(char *str, int destseg, int offval) {

    int f = TRUE, strl;
    strl = strlen(str);

    asm push   es
    asm mov    es, WORD PTR destseg
    asm mov    si, WORD PTR str
    asm	mov    di, WORD PTR offval
    asm mov    cx, WORD PTR strl
    asm cld
    asm	repe   cmpsb
    asm	je     scandone  // IF result of compare was true, then do not
    f = FALSE;           // change f and exit routine (i.e. match found).

  scandone:
    asm pop    es
    return(f);
}
/***************************************************************************
Function  : writetovideo
Parameters: ASCII code for character to display
Returns   :

***************************************************************************/
void writetovideo(int len, int code, int color) {

    int address = ((wherex() - 1) * 2) + ((wherey() -1) * 160);
    int videoSeg = videoAddress();   /* Is a colour card on system? */

    asm {
	mov   ah, BYTE PTR color
	mov   al, BYTE PTR code      /* The character to display      */
	mov   cx, WORD PTR len       /* cx determines number of loops */
	mov   dx, WORD PTR videoSeg
	push  es
	mov   es, dx
	mov   di, WORD PTR address
	rep   stosw
	pop   es
    }
}
/***************************************************************************
Function  : videoAddress
Parameters: None
Returns   : Segment address for video RAM on a given system

***************************************************************************/
int videoAddress(void) {

    int returnReg;

    asm {
	mov   ax, 0x0F00
	int   VIDEO
	mov   BYTE PTR returnReg, al
    }
    if (returnReg == 7)
	returnReg = MONO;
    else
	returnReg = COLOUR;

    return(returnReg);
}
/***************************************************************************
Function  : initmouse
Parameters: None
Returns   :

***************************************************************************/
int initmouse(void) {

    union REGS regs;

    regs.x.ax = 0;

    int86(MOUSE, &regs, &regs);
    return(regs.x.ax);
}
/***************************************************************************
Function  : showmouse
Parameters: None
Returns   :

***************************************************************************/
void showmouse(void) {

    union REGS regs;

    regs.h.ah = 0;
    regs.h.al = SHOW;
    int86(MOUSE, &regs, &regs);
}
/***************************************************************************
Function  : hidemouse
Parameters: None
Returns   :

***************************************************************************/
void hidemouse(void) {

    union REGS regs;

    regs.h.ah = 0;
    regs.h.al = HIDE;
    int86(MOUSE, &regs, &regs);
}
/***************************************************************************
Function  : movemouse
Parameters: column and row to position to
Returns   : nothing

**************************************************************************/
void movemouse(int col, int row) {

    union REGS regs;

    regs.x.ax = 4;
    regs.x.cx = (col*8)-1;
    regs.x.dx = (row*8)-1;      /* rows are from 1 to 25 */

    int86(MOUSE, &regs, &regs);
}
/***************************************************************************
Function  : mouseoff
Parameters: None
Returns   :

**************************************************************************/
void mouseoff(void) {

    union REGS regs;

    regs.h.ah = 0;
    regs.h.al = RESET;                  /* Software reset */
    int86(MOUSE, &regs, &regs);
}