*       MAKDAT.FOR

*       Create a binary data file which
*       can be read by the spectrum routine.

*       David E. Hess
*       Fluid Flow Group - Process Measurements Division
*       Chemical Science and Technology Laboratory
*       National Institute of Standards and Technology
*       June 30, 1994

*       This routine reads an ASCII input data file and rewrites
*       the data into a binary data file which can be processed by the
*       SPECTRUM calculation program. The routine first prompts the
*       user for information necessary to create the file header and
*       then the rewriting procedure begins. Extensive error checking
*       is included in an attempt to make the transformation process as
*       painless as possible. Refer to the section in the user's manual
*       for further details.

*                       File Extensions
*                       ---------------
*       .ASC - ASCII input data file (no header, just numbers)
*       .DAT - Binary output file (with file header)

*                       Header Information
*                       ------------------
*       ICHANS  : # of channels of data.
*       IDELTMS : sampling interval in microseconds.
*       IRSIZE  : # of bytes in each record.
*       N       : # of points per record per channel.
*       NUMREC  : # of records in data file.
*       GAIN    : array of gain values for each channel

*                           Batch Mode
*                           ----------
*	This routine has been updated to allow parameters to be added
*	to the command line in order to process many files in a batch.
*	The user is encouraged to first become familiar with the
*	operation of MAKDAT and with the various parameters used.
*	Then, if desired, the following command line may be used:

*	MAKDAT I|F 1|2 N NUMREC DELT 0|1|2|3 [0|1|2|3] FNAM

*	where the parameters are as follows:

*	I|F     - The letter I or F to indicate integer
*	          or floating-point data

*	1|2     - The digit 1 or the digit 2 to indicate 1 or 2
*	          channels of data

*	N       - The number of points per channel per record

*	NUMREC  - The number of records in the Ascii file

*	DELT    - The spacing between data points of the SAME
*	          channel in seconds

*	0|1|2|3 - The conversion factor to use for channel 1

*	0|1|2|3 - The conversion factor to use for channel 2

*	FNAM    - Four character filename for FNAM.ASC

*	For example, to convert the ASCII file A001.ASC containing
*	204800 rows of one channel of floating-point data sampled
*	at a rate of 20 kHz, into the Binary file A001.DAT one
*	could use:

*	makdat f 1 2048 100 .00005 0 a001

	IMPLICIT        REAL*4 (A-H,O-Z), INTEGER*2 (I-N)
	PARAMETER       (NUMI=2,NUMO=3,NMAX=16384,NARGMX=8)
	INTEGER*2       GAIN(0:7),ILEN(NARGMX)
	INTEGER*2       NDATA[ALLOCATABLE,HUGE](:)
	INTEGER*4       IRSIZE,IDELTMS,NUMARGS,I
	REAL*4          RDATA[ALLOCATABLE,HUGE](:)
	LOGICAL*1       INTGER,FLOTNG,ONECHAN,TWOCHAN,BATCH
	CHARACTER       INSFX *4 /'.ASC'/, OUTSFX *4 /'.DAT'/
	CHARACTER*1     INP,FIRST
	CHARACTER*4     INNAM
	CHARACTER*8     INFIL,OUTFIL,BUFFER(NARGMX)
	CHARACTER*72    TXTLINE

*       Initialize gain array.

	GAIN=0

*       Code to perform command line parsing.

	NUMARGS=NARGS()-1
	IF (NUMARGS .EQ. 0) THEN
	  BATCH=.FALSE.		! No arguments on command line.
	ELSE
	  BATCH=.TRUE.		! Arguments on the command line.
	  IF (NUMARGS .GT. NARGMX) THEN	 ! Arguments within range ?
	    WRITE (*,'(/1X,A/1X,I1,A/1X,A)')
     +      'The routine is currently set to recognize up to ',
     +      NARGMX,' arguments on the command line and will',
     +      'proceed with these arguments only.'
	    NUMARGS=NARGMX
	  ENDIF
	  DO I=1,NUMARGS	! Get the arguments.
	    CALL GETARG (I,BUFFER(I),ILEN(I))
	  ENDDO
	ENDIF

*       Integer or floating point data ?

10	IF (BATCH) THEN		! Argument 1
	  INP=BUFFER(1)(1:1)
	ELSE
	  WRITE (*,'(/1X,A,A\)') '(I)nteger (2-byte) or ',
     +                        '(F)loating-point (4-byte) data : '
	  READ (*,'(A)') INP
	ENDIF

	IF (INP .EQ. 'i') INP = 'I'
	IF (INP .EQ. 'f') INP = 'F'
	INTGER=(INP .EQ. 'I')
	FLOTNG=(INP .EQ. 'F')

	IF (.NOT. INTGER .AND. .NOT. FLOTNG
     +                   .AND. .NOT. BATCH) GO TO 10

	IF (.NOT. INTGER .AND. .NOT. FLOTNG .AND. BATCH) THEN
	  WRITE (*,'(/1X,A/)') 'Argument 1 must be the letter I or F'
	  STOP 'Program aborted.'
	ENDIF

*       Get # of channels.

20	IF (BATCH) THEN		! Argument 2
	  CALL STRGTONM (BUFFER(2),X)
	  ICHANS=NINT(X)
	ELSE
	  WRITE (*,'(/1X,A\)') 'Enter # of channels (1 or 2) : '
	  READ (*,*) ICHANS
	ENDIF

	ONECHAN=(ICHANS .EQ. 1)
	TWOCHAN=(ICHANS .EQ. 2)
	IF (.NOT. ONECHAN .AND. .NOT. TWOCHAN
     +                    .AND. .NOT. BATCH) GO TO 20

	IF (.NOT. ONECHAN .AND. .NOT. TWOCHAN .AND. BATCH) THEN
	  WRITE (*,'(/1X,A/)') 'Argument 2 must be the number 1 or 2'
	  STOP 'Program aborted.'
	ENDIF

*       Get # of points per record per channel.

30	IF (BATCH) THEN		! Argument 3
	  CALL STRGTONM (BUFFER(3),X)
	  N=NINT(X)
	ELSE
	  WRITE (*,'(/1X,A,I5,A/1X,A,A,I5,A)')
     +      'One channel  : Total # points per record <= ',
     +      NMAX,'.','Two channels : Total # points per record',
     +      ' per channel <= ',NMAX/2,'.'
	  IF (ONECHAN) THEN
	    WRITE (*,'(/1X,A,A\)') 'Enter # of points per',
     +                             ' record (power of two) : '
	    READ (*,*) N
	  ELSE
	    WRITE (*,'(/1X,A,A/1X,A\)') 'Enter # of points per',
     +               ' record for each channel (power of two).',
     +      'Total # of points per record is double this number : '
	    READ (*,*) N
	  ENDIF   
	ENDIF   

*       N less than or equal to NMAX error checking.

	IF (ONECHAN) NTST=NMAX
	IF (TWOCHAN) NTST=NMAX/2
	IF (N .GT. NMAX .AND. .NOT. BATCH) THEN
	  WRITE (*,'(/1X,A,A,I5,A)') '# of points per record',
     +         ' per channel <= ',NTST,' dummy!'
	  GO TO 30
	ENDIF
	IF (N .GT. NMAX .AND. BATCH) THEN
	  WRITE (*,'(/1X,A,A,I5,A)') '# of points per record',
     +         ' per channel (arg 3) <= ',NTST,' dummy!'
	  STOP 'Program aborted.'
	ENDIF

*       Power of two error checking.

	FN=FLOAT(N)
	ITST=NINT(ALOG10(FN)/ALOG10(2.0))
	ITST2=INT(2**ITST)-N

	IF (ITST2 .NE. 0 .AND. .NOT. BATCH) THEN
	  WRITE (*,'(/1X,A,I5,A/1X,A)') 'You have entered ',
     +           N,' data points.','# data points must be a power of 2.'
	  GO TO 30
	ENDIF
	IF (ITST2 .NE. 0 .AND. BATCH) THEN
	  WRITE (*,'(/1X,A,I5,A/1X,A/)') 'You have entered ',
     +           N,' data points (arg 3).',
     +           '# data points must be a power of 2.'
	  STOP 'Program aborted.'
	ENDIF

	IF (INTGER) IRSIZE=ICHANS*N*2
	IF (FLOTNG) IRSIZE=ICHANS*N*4

*       Allocate space for NDATA and RDATA arrays.

	IF (ONECHAN .AND. INTGER) ALLOCATE (NDATA(N), STAT=IERR)
	IF (ONECHAN .AND. FLOTNG) ALLOCATE (RDATA(N), STAT=IERR)
	IF (TWOCHAN .AND. INTGER) ALLOCATE (NDATA(2*N), STAT=IERR)
	IF (TWOCHAN .AND. FLOTNG) ALLOCATE (RDATA(2*N), STAT=IERR)
	IF (IERR .NE. 0)
     +          STOP 'Not enough storage for data.  Aborting ...'

*       Get # of records in data file.

	IF (BATCH) THEN		! Argument 4
	  CALL STRGTONM (BUFFER(4),X)
	  NUMREC=NINT(X)
	ELSE
	  WRITE (*,'(/1X,A\)') 'Enter # of records in the data file : '
	  READ (*,*) NUMREC
	ENDIF

*       Get the sampling interval.

	IF (BATCH) THEN		! Argument 5
	  CALL STRGTONM (BUFFER(5),DELT)
	ELSE
	  WRITE (*,'(/1X,A/1X,A,A/1X,A,A/1X,A)')
     +      'One chan  : Delta t is spacing between data points.',
     +      'Two chans : Delta t is spacing between data pts',
     +      ' - SAME channel.',
     +      '            Delta t divided by 2 is spacing between',
     +      ' data pts','            - different channels.'

	  WRITE (*,'(/1X,A\)')
     +      'Enter sampling interval delta t (secs) : '
	  READ (*,*) DELT
	  WRITE (*,'( )')
	ENDIF

	IDELTMS=NINT(DELT*1.0E+06)

*       Set the gain for each channel.

35	IF (BATCH) THEN		! Argument 6 (and 7)
	  DO I=0,ICHANS-1
	    CALL STRGTONM (BUFFER(I+6),X)
	    GAIN(I)=NINT(X)
	  ENDDO
	ELSE
	  WRITE (*,'(14X,A,5X,A)') ' Voltage Range ','Gain'
	  WRITE (*,'(14X,A,5X,A)') ' ------------- ','----'
	  WRITE (*,'(14X,A,5X,A)') '-10.00 to 10.00','  0 '
	  WRITE (*,'(14X,A,5X,A)') '- 5.00 to  5.00','  1 '
	  WRITE (*,'(14X,A,5X,A)') '- 2.50 to  2.50','  2 '
	  WRITE (*,'(14X,A,5X,A)') '- 1.25 to  1.25','  3 '
	  WRITE (*,'( )')

	  DO I=0,ICHANS-1
	    WRITE (*,'(1X,A,I1,A\)') 'Enter gain for channel ',I,' : '
	    READ (*,*) GAIN(I)
	  ENDDO

	ENDIF

*	Test each GAIN value to see if it is within range.

	DO I=0,ICHANS-1
	  IF (GAIN(I) .LT. 0 .OR. GAIN(I) .GT. 3
     +                .AND. .NOT. BATCH) THEN
	    WRITE (*,'(/1X,2(A,I1),A/1X,A/)')
     +        'The gain for channel ',I,' = ',GAIN(I),'.',
     +        'Gain must be a number from 0 to 3.'
	    GO TO 35
	  ENDIF
	  IF (GAIN(I) .LT. 0 .OR. GAIN(I) .GT. 3
     +                .AND. BATCH) THEN
	    WRITE (*,'(/1X,3(A,I1),A/1X,A/)')
     +        'The gain for channel ',I,' (arg ',I+6,') = ',GAIN(I),'.',
     +        'Gain must be a number from 0 to 3.'
	    STOP 'Program aborted.'
	  ENDIF
	ENDDO

*       Get input file name.

40	IF (BATCH) THEN		! Argument 7 (or 8)
	  IF (ONECHAN) INNAM=BUFFER(7)
	  IF (TWOCHAN) INNAM=BUFFER(8)
	ELSE
	  WRITE (*,'(/1X,A\)')
     +      'Enter ASCII input file name (4 chars) : '
	  READ (*,'(A)') INNAM
	ENDIF

*       Convert to uppercase and check first character alphabetic.

	DO J=4,1,-1
	  FIRST=INNAM(J:J)
	  IF (ICHAR(FIRST) .GE. 97 .AND. ICHAR(FIRST) .LE. 122) THEN
	    IHOLD=ICHAR(FIRST)-32
	    FIRST=CHAR(IHOLD)
	    INNAM(J:J)=FIRST
	  ENDIF
	ENDDO
	IF (ICHAR(FIRST) .LT. 65 .OR. ICHAR(FIRST) .GT. 90) THEN
	  WRITE (*,'(/1X,A,A,A/1X,A,A,A/1X,A)') 
     +      'Filename ',INNAM,' began with',
     +      'the nonalphabetic character ',FIRST,'.',
     +      'Re-enter the filename correctly.'
	  GO TO 40
	ENDIF

45      INFIL=INNAM // INSFX
	OUTFIL=INNAM // OUTSFX

*       Put message on screen.

	WRITE (*,'(/////////////////////16X,
     +      ''D A T A   F I L E   C R E A T I O N   U T I L I T Y'')')
	WRITE (*,'(/25X,''Creating '',A,'' now.'')') OUTFIL

*       Open input ASCII file.

	OPEN (NUMI,FILE=INFIL,STATUS='OLD',ERR=100)

*       Read line of text in data file.

	READ (NUMI,'(A)') TXTLINE

*       Open output data file and write header.

	OPEN (NUMO,FILE=OUTFIL,STATUS='UNKNOWN',ACCESS='SEQUENTIAL',
     +        FORM='BINARY',ERR=110)
	WRITE (NUMO) ICHANS,IRSIZE,NUMREC,IDELTMS
	WRITE (NUMO) (GAIN(I),I=0,7)

*       Display header information.

	WRITE (*,'(/25X,A,I1)')   '# channels = ',ICHANS
	WRITE (*,'(25X,A,I5,A)')  'record size = ',IRSIZE,' bytes'
	WRITE (*,'(25X,A,I5)')    '# of records = ',NUMREC
	WRITE (*,'(25X,A,I5,A/)') 'delta t = ',IDELTMS,' microseconds'

	DO J=1,NUMREC

*         Display record count.

	  IF (J .EQ. 1) THEN
	    WRITE (*,50) J
50          FORMAT (25X,'Record ',I4.4)
	  ELSE
	    WRITE (*,60) J
60          FORMAT ('+',24X,'Record ',I4.4)
	  ENDIF

	  IF (INTGER) THEN
	    IF (ONECHAN) THEN
	      READ (NUMI,*,ERR=120,END=140) (NDATA(I), I=1,N)
	      WRITE (NUMO, ERR=130)         (NDATA(I), I=1,N)
	    ELSE
	      READ (NUMI,*,ERR=120,END=140) (NDATA(I), I=1,2*N)
	      WRITE (NUMO, ERR=130)         (NDATA(I), I=1,2*N)
	    ENDIF
	  ELSE IF (FLOTNG) THEN
	    IF (ONECHAN) THEN
	      READ (NUMI,*,ERR=120,END=140) (RDATA(I), I=1,N)
	      WRITE (NUMO, ERR=130)         (RDATA(I), I=1,N)
	    ELSE
	      READ (NUMI,*,ERR=120,END=140) (RDATA(I), I=1,2*N)
	      WRITE (NUMO, ERR=130)         (RDATA(I), I=1,2*N)
	    ENDIF
	  ENDIF
	ENDDO

	CLOSE (NUMI,STATUS='KEEP')
	CLOSE (NUMO,STATUS='KEEP')

	WRITE (*,'( )')
	STOP '                        Program terminated successfully.'

*       Problem opening input ASCII file.

100     WRITE (*,'(/25X,A/)') 'Problem opening input ASCII file.'
	STOP '                       Program terminated unsuccessfully.'

*       Problem opening output data file.

110     WRITE (*,'(/25X,A/)') 'Problem opening output data file.'
	STOP '                       Program terminated unsuccessfully.'

*       Problem reading input ASCII file.

120     WRITE (*,'(/25X,A/)') 'Problem reading input ASCII file.'
	CLOSE (NUMI,STATUS='KEEP')
	CLOSE (NUMO,STATUS='KEEP')
	STOP '                       Program terminated unsuccessfully.'

*       Problem writing output data file.

130     WRITE (*,'(/25X,A/)') 'Problem writing output data file.'
	CLOSE (NUMI,STATUS='KEEP')
	CLOSE (NUMO,STATUS='KEEP')
	STOP '                       Program terminated unsuccessfully.'

*       Problem : reached end of file marker reading input ASCII file.

140     WRITE (*,'(/25X,A/)') 'Problem : reached end of file marker',
     +                     ' reading input ASCII file.'
	CLOSE (NUMI,STATUS='KEEP')
	CLOSE (NUMO,STATUS='KEEP')
	STOP '                       Program terminated unsuccessfully.'
	END

	SUBROUTINE STRGTONM (STRG,NUMBER)

	IMPLICIT	REAL*4 (A-H,O-Z), INTEGER*2 (I-N)
	INTEGER*2	DIGIT
	REAL*4		NUMBER
	CHARACTER*(*)	STRG

*	Initialization.

	NUMBER=0.0

*	Find the length of the string and look for a decimal point.

	IEND=LEN_TRIM(STRG)
	IPD=INDEX(STRG,'.')

*	Convert digits after decimal point.

	IF (IPD .NE. 0) THEN
	  DO I=IPD+1,IEND
	    DIGIT=ICHAR(STRG(I:I))-48
	    NUMBER=NUMBER+DIGIT/10.0**(I-IPD)
	  ENDDO
	  IEND=IPD-1
	ENDIF

*	Convert digits before decimal point.

	DO I=IEND,1,-1
	  DIGIT=ICHAR(STRG(I:I))-48
	  NUMBER=NUMBER+DIGIT*10.0**(IEND-I)
	ENDDO

	RETURN
	END
