/*
 * MultiMail offline mail reader
 * QWK

 Copyright (c) 1997 John Zero <john@graphisoft.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "qwk.h"
#include "compress.h"
#include "../interfac/mysystem.h"

// ---------------------------------------------------------------------------
// The qheader method
// ---------------------------------------------------------------------------

int qheader::init(FILE *datFile)
{
	qwkmsg_header qh;
	char buf[9];

	if (!fread(&qh, 1, qmRecLen, datFile))
		return 0;

	getQfield(from, qh.from, 25);
	getQfield(to, qh.to, 25);
	getQfield(subject, qh.subject, 25);

	cropesp(from);
	cropesp(to);
	cropesp(subject);

	getQfield(date, qh.date, 8);
	date[2] = '-';
	date[5] = '-';		// To deal with some broken messages
	strcat(date, " ");

	getQfield(buf, qh.time, 5);
	strcat(date, buf);

	getQfield(buf, qh.refnum, 8);
	refnum = atoi(buf);

	getQfield(buf, qh.msgnum, 7);
	msgnum = atoi(buf);

	getQfield(buf, qh.chunks, 6);
	msglen = (atoi(buf) - 1) << 7;

	privat = (qh.status == '*');
	origArea = ((int) qh.confMSB << 8) + (int) qh.confLSB;

	return 1;
}

// ---------------------------------------------------------------------------
// The QWK methods
// ---------------------------------------------------------------------------

qwkpack::qwkpack(mmail *mmA)
{
	mm = mmA;
	init();
}

void qwkpack::init()
{
	ID = 0;
	bodyString = NULL;

	readControlDat();
	readIndices();
	initMessagesDat();
}

qwkpack::~qwkpack()
{
	cleanup();
}

void qwkpack::cleanup()
{
	while (maxConf--) {
		delete body[maxConf];
		delete areas[maxConf].name;
	}
	delete body;
	delete areas;
	delete bodyString;

	fclose(msgdatFile);
}

unsigned long qwkpack::MSBINtolong(unsigned const char *ms)
{
	return ((((unsigned long) ms[0] + ((unsigned long) ms[1] << 8) +
		  ((unsigned long) ms[2] << 16)) | 0x800000L) >>
			(24 + 0x80 - ms[3]));
}

int qwkpack::getNoOfAreas()
{
	return maxConf;
}

void qwkpack::resetAll()
{
	cleanup();
	init();
}

area_header *qwkpack::getNextArea()
{
	area_header *tmp;
	int cMsgNum = areas[ID].nummsgs;
	int x = (areas[ID].num == -1);

	tmp = new area_header(mm, ID + mm->driverList->getOffset(this),
			areas[ID].numA, areas[ID].name,
			(x ? "Letters addressed to you" : areas[ID].name),
			(x ? "QWK personal" : "QWK"), (x ? COLLECTION : 0) |
			(cMsgNum ? ACTIVE : 0), cMsgNum, 0);
	ID++;
	return tmp;
}

void qwkpack::selectArea(int area)
{
	currentArea = area;
	resetLetters();
}

void qwkpack::resetLetters()
{
	currentLetter = 0;
}

int qwkpack::getNoOfLetters()
{
	return areas[currentArea].nummsgs;
}

void qwkpack::readIndices()
{
	struct {
        	unsigned char MSB[4];
        	unsigned char confnum;
	} ndx_rec;

	FILE *idxFile;
	char fname[13];
	int cMsgNum, x, y;

	body = new struct bodytype *[maxConf];

	for (x = 0; x < maxConf; x++) {
		body[x] = NULL;
		cMsgNum = 0;

		if (areas[x].num == -1)
			sprintf(fname, "personal.ndx");
		else
			sprintf(fname, "%03d.ndx", areas[x].num);

		if ((idxFile = mm->workList->ftryopen(fname, "rb"))) {
			cMsgNum = mm->workList->getSize() / ndxRecLen;
			body[x] = new struct bodytype[cMsgNum];
			for (y = 0; y < cMsgNum; y++) {
				fread(&ndx_rec, ndxRecLen, 1, idxFile);
				body[x][y].pointer = MSBINtolong(ndx_rec.MSB);
			}
			fclose(idxFile);
		}

		areas[x].nummsgs = cMsgNum;
	}
}

int qwkpack::getXNum(int area)
{
	int c;

	for (c = 0; c < maxConf; c++)
		if (areas[c].num == area)
			break;
	return c;
}

int qwkpack::getYNum(int area, unsigned long rawpos)
{
	int c;

	for (c = 0; c < areas[area].nummsgs; c++)
		if ((unsigned) body[area][c].pointer == rawpos)
			break;
	return c;
}

letter_header *qwkpack::getNextLetter()
{
	qheader q;
	unsigned long pos, rawpos;
	int areaID, letterID;

	rawpos = body[currentArea][currentLetter].pointer;
	pos = (rawpos - 1) << 7;

	fseek(msgdatFile, pos, SEEK_SET);
	if (!q.init(msgdatFile))
		fatalError("Error reading MESSAGES.DAT");

	body[currentArea][currentLetter].msgLength = q.msglen;

	if (areas[currentArea].num == -1) {
		areaID = getXNum(q.origArea);
		letterID = getYNum(areaID, rawpos);
	} else {
		areaID = currentArea;
		letterID = currentLetter;
	}
	currentLetter++;

	return new letter_header(mm, stripre(q.subject), q.to, q.from,
			q.date, q.refnum, letterID, q.msgnum, areaID,
			q.privat, q.msglen, this, NULL);
}

// returns the body of the requested letter in the active area
char *qwkpack::getBody(int AreaID, int LetterID)
{
	unsigned char *p;
	int c, kar, lfig = 0;

	delete bodyString;
	bodyString = new char[body[AreaID][LetterID].msgLength + 1];
	fseek(msgdatFile, body[AreaID][LetterID].pointer << 7, SEEK_SET);

	for (c = 0, p = (unsigned char *) bodyString;
	     c < body[AreaID][LetterID].msgLength; c++) {
		kar = fgetc(msgdatFile);

		if (!kar)
			kar = ' ';

		// Ctrl-A line filtering (it shouldn't appear in
		// a QWK, but anyway)

		if (kar == 227) {
			*p++ = '\n';
			if (lfig)
				p--;
			lfig = 0;
		} else
			if (kar == 1)
				lfig = 1;
			else
				if (!lfig)
					*p++ = kar;
	}
	do
		p--;
	while ((*p == ' ') || (*p == '\n'));	// Strip blank lines
	*++p = '\0';

	return bodyString;
}

char *qwkpack::nextLine()
{
	static char line[128];

	fgets(line, 127, ctrdatFile);
	strtok(line, "\r\n");
	return line;
}

void qwkpack::readControlDat()
{
	char *p, *q;
	int pers;

	if (!(ctrdatFile = mm->workList->ftryopen("control.dat", "rb")))
		fatalError("Could not open CONTROL.DAT");

	mm->resourceObject->set(BBSName, nextLine());	// 1: BBS name
	nextLine();					// 2: city/state
	nextLine();					// 3: phone#

	q = nextLine();					// 4: sysop's name
	int slen = strlen(q);
	if (slen > 6) {
		p = q + slen - 5;
		if (!strcasecmp(p, "Sysop")) {
			if (*--p == ' ')
				p--;
			if (*p == ',')
				*p = '\0';
		}
	}
	mm->resourceObject->set(SysOpName, q);

	q = nextLine();					// 5: doorserno,BBSid
	p = strtok(q, ",");
	p = strtok(NULL, " ");
	mm->resourceObject->set(qwkBBSID, p);

	nextLine();					// 6: time&date
	p = nextLine();					// 7: USERNAME
	mm->resourceObject->set(LoginName, p);
	mm->resourceObject->set(AliasName, p);

	nextLine();					// 8: blank/any
	nextLine();					// 9: anyth.
	nextLine();					// 10: anyth.

	maxConf = atoi(nextLine()) + 1;			// 11: Max conf# - 1!

	pers = !(!mm->workList->exists("personal.ndx"));
	if (pers)
		maxConf++;

	areas = new AREAs[maxConf];
	if (pers) {
		areas[0].num = -1;
		strcpy(areas[0].numA, "PERS");
		areas[0].name = strdupplus("PERSONAL");
	}
	for (int c = pers; c < maxConf; c++) {
		areas[c].num = atoi(nextLine());		// conf #
		sprintf(areas[c].numA, "%d", areas[c].num);
		areas[c].name = strdupplus(nextLine());		// conf name
	}

	fclose(ctrdatFile);
}

void qwkpack::initMessagesDat()
{
	if (!(msgdatFile = mm->workList->ftryopen("messages.dat", "rb")))
		fatalError("Could not open MESSAGES.DAT");
}

// ---------------------------------------------------------------------------
// The QWK reply methods
// ---------------------------------------------------------------------------

qwkreply::qwkreply(mmail *mmA, specific_driver *baseClassA)
{
	mm = mmA;
	baseClass = (qwkpack *) baseClassA;
	init();
}

qwkreply::~qwkreply()
{
	if (replyExists)
		cleanup();
}

void qwkreply::init()
{
	replyText = NULL;
	repFileName(mm->resourceObject->get(qwkBBSID));
	mychdir(mm->resourceObject->get(ReplyDir));
	replyExists = !access(replyPacketName, R_OK | W_OK);

	if (replyExists) {
		uncompress();
		readRep();
		currentLetter = 1;
	} else {
		uplListHead = NULL;
		noOfLetters = currentLetter = 0;
	}
}

void qwkreply::cleanup()
{
	upl_list *next, *curr = uplListHead;

	while (noOfLetters--) {
		remove(curr->fname);
		next = curr->nextRecord;
		delete curr;
		curr = next;
	}
	delete replyText;
}

void qwkreply::repFileName(const char *qFile)
{
	int c;

	for (c = 0; (qFile[c] != '.') && qFile[c]; c++)
		replyPacketName[c] = tolower(qFile[c]);
	strcpy(replyPacketName + c, ".rep");
}

void qwkreply::uncompress()
{
	char fname[256];

	sprintf(fname, "%s/%s", mm->resourceObject->get(ReplyDir),
		replyPacketName);
	uncompressFile(mm->resourceObject, fname,
		mm->resourceObject->get(UpWorkDir));
}

int qwkreply::getRep1(FILE *rep, upl_list *l)
{
	FILE *replyFile;
	char *p, *replyText;

	if (!l->qHead.init(rep))
		return -1;

	mytmpnam(l->fname);

	if (!(replyFile = fopen(l->fname, "wt")))
		return -1;

	replyText = new char[l->qHead.msglen + 1];
	fread(replyText, l->qHead.msglen, 1, rep);

	for (p = &replyText[l->qHead.msglen - 1]; ((*p == ' ') ||
		(*p == (char) 227)) && (p > replyText); p--);
	*++p = '\0';

	for (p = replyText; *p; p++)
		if (*p == (char) 227)
			*p = '\n';	// PI-softcr

	l->qHead.msglen = p - replyText;	//strlen(replyText);

	fwrite(replyText, 1, l->qHead.msglen, replyFile);
	fclose(replyFile);
	delete replyText;

	return 0;
}

void qwkreply::readRep()
{
	FILE *repFile;
	upl_list baseUplList, *currUplList;
	file_list *upWorkList;
	char repName[13];

	upWorkList = new file_list(mm->resourceObject->get(UpWorkDir));
	sprintf(repName, "%s.msg", findBaseName(replyPacketName));
	if (!(repFile = upWorkList->ftryopen(repName, "rb")))
		fatalError("Error opening REP");
	fseek(repFile, 128, SEEK_SET);

	noOfLetters = 0;

	currUplList = &baseUplList;
	while (!feof(repFile)) {
		currUplList->nextRecord = new upl_list;
		currUplList = currUplList->nextRecord;
		if (getRep1(repFile, currUplList)) {	// ha eof/error
			delete currUplList;
			break;
		}
		noOfLetters++;
	}
	uplListHead = baseUplList.nextRecord;

	fclose(repFile);
	remove(upWorkList->exists(repName));
	delete upWorkList;
}

int qwkreply::getNoOfAreas()
{
	return 1;
}

void qwkreply::resetAll()
{
	cleanup();
	init();
}

area_header *qwkreply::getNextArea()
{
	return new area_header(mm, 0, "REPLY", "REPLIES",
		"Letters written by you", "QWK replies",
		(COLLECTION | REPLYAREA | ACTIVE), noOfLetters, 0);
}

void qwkreply::selectArea(int ID)
{
	if (ID == 0)
		resetLetters();
}

void qwkreply::resetLetters()
{
	currentLetter = 1;
	uplListCurrent = uplListHead;
}

int qwkreply::getNoOfLetters()
{
	return noOfLetters;
}

letter_header *qwkreply::getNextLetter()
{
	letter_header *newLetter;

	newLetter = new letter_header(mm, uplListCurrent->qHead.subject,
			uplListCurrent->qHead.to, uplListCurrent->qHead.from,
			uplListCurrent->qHead.date,
			uplListCurrent->qHead.refnum,
			currentLetter, currentLetter,
			baseClass->getXNum(uplListCurrent->qHead.msgnum) + 1,
			uplListCurrent->qHead.privat,
			uplListCurrent->qHead.msglen, this, NULL);

	currentLetter++;
	uplListCurrent = uplListCurrent->nextRecord;
	return newLetter;
}

char *qwkreply::getBody(int area, int ID)
{
	FILE *replyFile;
	upl_list *actUplList;
	int msglen;

	area = area;	// warning suppression
	delete replyText;

	actUplList = uplListHead;
	for (int c = 1; c < ID; c++)
		actUplList = actUplList->nextRecord;

	if ((replyFile = fopen(actUplList->fname, "rt"))) {
		msglen = actUplList->qHead.msglen;
		replyText = new char[msglen + 1];
		msglen = fread(replyText, 1, msglen, replyFile);
		fclose(replyFile);
		replyText[msglen] = '\0';
	} else
		replyText = NULL;

	return replyText;
}

int qwkreply::monthval(const char *abbr)
{
	static const char *month_abbr[] =
		{"Jan", "Feb", "Mar", "Apr", "Mar", "Jun",
	 	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	int c;

	for (c = 0; c < 12; c++)
		if (!strcmp(month_abbr[c], abbr))
			break;
	return c + 1;
}

void qwkreply::enterLetter(letter_header *newLetter, char *newLetterFileName,
				int length)
{
	upl_list *newList = new upl_list;
	memset(newList, 0, sizeof(upl_list));

	strncpy(newList->qHead.subject, newLetter->getSubject(), 25);
	strncpy(newList->qHead.from, newLetter->getFrom(), 25);
	strncpy(newList->qHead.to, newLetter->getTo(), 25);
	newList->qHead.msgnum = atoi(mm->areaList->getShortName());
	newList->qHead.privat = newLetter->getPrivate();
	newList->qHead.refnum = newLetter->getReplyTo();
	strcpy(newList->fname, newLetterFileName);

	time_t tt = time(NULL);
	char *dt = ctime(&tt);
	dt[7] = 0;
	dt[10] = 0;
	dt[16] = 0;
	dt[24] = 0;
	sprintf(newList->qHead.date, "%02d-%02d-%s %s", monthval(&dt[4]),
		atoi(&dt[8]), &dt[22], &dt[11]);	// 'MM-DD-YY hh:mm'

	newList->qHead.msglen = length;

	if (!noOfLetters)
		uplListHead = newList;
	else {
		upl_list *workList = uplListHead;
		for (int c = 1; c < noOfLetters; c++)	//go to last elem
			workList = workList->nextRecord;
		workList->nextRecord = newList;
	}

	noOfLetters++;
	replyExists = 1;
}

void qwkreply::killLetter(int letterNo)
{
	upl_list *actUplList, *tmpUplList;

	if (!noOfLetters || (letterNo < 1) || (letterNo > noOfLetters))
		fatalError("Internal error in qwkreply::killLetter");

	if (letterNo == 1) {
		tmpUplList = uplListHead;
		uplListHead = uplListHead->nextRecord;
	} else {
		actUplList = uplListHead;
		for (int c = 1; c < letterNo - 1; c++)
			actUplList = actUplList->nextRecord;
		tmpUplList = actUplList->nextRecord;
		actUplList->nextRecord = (letterNo == noOfLetters) ? 0 :
			    actUplList->nextRecord->nextRecord;
	}
	noOfLetters--;
	remove(tmpUplList->fname);
	delete tmpUplList;
	resetLetters();
}

area_header *qwkreply::refreshArea()
{
	return getNextArea();
}

void qwkreply::addRep1(FILE *rep, upl_list *l)
{
	FILE *replyFile;
	qwkmsg_header qh;
	char buf[10], *p, *lastsp = NULL, *replyText;
	int chunks, count = 0;

	memset(&qh, ' ', sizeof(qh));
	chunks = (l->qHead.msglen + 127) / 128;

	sprintf(buf, " %-6d", l->qHead.msgnum);
	strncpy(qh.msgnum, buf, 7);
	if (l->qHead.refnum) {
		sprintf(buf, " %-7d", l->qHead.refnum);
		strncpy(qh.refnum, buf, 8);
	}
	strncpy(qh.to, l->qHead.to, strlen(l->qHead.to));
	strncpy(qh.from, l->qHead.from, strlen(l->qHead.from));
	strncpy(qh.subject, l->qHead.subject, strlen(l->qHead.subject));

	qh.alive = (char) 0xE1;

	strncpy(qh.date, l->qHead.date, 8);
	strncpy(qh.time, &l->qHead.date[9], 5);

	sprintf(buf, "%-6d", chunks + 1);
	strncpy(qh.chunks, buf, 6);
	if (l->qHead.privat)
		qh.status = '*';

	fwrite(&qh, 1, sizeof(qh), rep);

	replyText = new char[chunks * 128 + 1];
	memset(&replyText[(chunks - 1) * 128], ' ', 128);

	replyFile = fopen(l->fname, "rt");	// !! check it !
	fread(replyText, l->qHead.msglen, 1, replyFile);
	fclose(replyFile);

	replyText[l->qHead.msglen] = 0;
	for (p = replyText; *p; p++) {
		if (*p == '\n') {
			*p = (char) 227;
			count = 0;
			lastsp = NULL;
		} else
			count++;
		if (*p == ' ')
			lastsp = p;
		if ((count >= 80) && lastsp) {
			*lastsp = (char) 227;
			count = p - lastsp;
			lastsp = NULL;
		}
	}
	replyText[l->qHead.msglen] = (char) 227;

	fwrite(replyText, 1, chunks * 128, rep);
	delete replyText;
}

void qwkreply::makeReply()
{
	FILE *repFile;
	upl_list *actUplList;
	char repFileName[13], tmp[256];

	if (mychdir(mm->resourceObject->get(UpWorkDir)))
		fatalError("Could not cd to upworkdir in qwkreply::makeReply");

	// Delete old packet
	sprintf(tmp, "%s/%s", mm->resourceObject->get(ReplyDir),
		replyPacketName);
	remove(tmp);

	if (!noOfLetters)
		return;

	sprintf(repFileName, "%s.msg", findBaseName(replyPacketName));
	repFile = fopen(repFileName, "wb");	//!! no check yet

	sprintf(tmp, "%-128s", mm->resourceObject->get(qwkBBSID));
	fwrite(tmp, 128, 1, repFile);

	actUplList = uplListHead;
	for (int c = 0; c < noOfLetters; c++) {
		addRep1(repFile, actUplList);
		actUplList = actUplList->nextRecord;
	};

	fclose(repFile);

	//pack the files
	compressAddFile(mm->resourceObject,
		mm->resourceObject->get(ReplyDir),
			replyPacketName, repFileName);

	// clean up the work area
	clearDirectory(".");
}
