/*
 *      JET PAK - HP DeskJet and LaserJet series printer utilities
 *
 *      JETPRD program - output Printer Description file
 *
 *      Version 1.1 (Public Domain)
 */

/* system include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* application include files */
#include "patchlev.h"
#include "jetfont.h"
#include "jetutil.h"

/*
 *  MODULE GLOBAL DATA
 */

/* input command being processed */
static FONT_COMMAND fc;

/* character code being processed */
static UNSIGNEDINT charcode;

/* pitch required for character widths in fixed fonts */
static UNSIGNEDINT pitch;

/* output file suffix */
static char output_suffix[SUFFIX_MAX] = ".prd";

/*
 * LOCAL FUNCTIONS
 */

static void usage_wrong()
{
    /*
     * Print usage message and exit.
     */
    fprintf(stderr, USAGE_HEADER);
    fprintf(stderr, "Usage: JETPRD [-h] [-t filetype] fontfile [fontfile...]\n\n");
    fprintf(stderr, "Output Printer Descriptor files for soft fonts\n\n");
    fprintf(stderr, "  -h            print this usage information\n");
    fprintf(stderr, "  -t filetype   change output file type (default %s)\n", output_suffix);
    fprintf(stderr, "  fontfile      soft font file name\n");
    exit(1);
}

static int printisd(fp, i)
FILE *fp;
UNSIGNEDINT i;
{
    /*
     *  Convert an integer to a string, then output each byte
     *  separately in decimal
     */
    char s[20], *sp = s;

    sprintf(s, "%d", i);

    while (*sp)
    {
        if (fprintf(fp, "%d", *sp) < 0)
            return(ERROR);
        if (*(++sp))
            if (fprintf(fp, ",") < 0)
                return(ERROR);
    }

    return(OK);
}

static int printfsd(fp, f)
FILE *fp;
float f;
{
    /*
     *  Convert a float to a string, then output each byte
     *  separately in decimal
     */
    char s[20], *sp = s;

    sprintf(s, "%.2f", f);

    while (*sp)
    {
        if (fprintf(fp, "%d", *sp) < 0)
            return(ERROR);
        if (*(++sp))
            if (fprintf(fp, ",") < 0)
                return(ERROR);
    }

    return(OK);
}

static int headerprint(fp)
FILE *fp;
{
    /*
     * Output the escape sequence required to select the font unambiguously
     */

    /* orientation */
    if (fprintf(fp, "27,38,108,") < 0)
        return(ERROR);
    if (printisd(fp, fc.data.font.orientation) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",79,") < 0)
        return(ERROR);

    /* character set */
    if (fprintf(fp, "27,40,") < 0)
        return(ERROR);
    if (printisd(fp, fc.data.font.symbol_set/32) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",%d", '@'+fc.data.font.symbol_set%32) < 0)
        return(ERROR);

    /* spacing */
    if (fprintf(fp, ",27,40,115,") < 0)
        return(ERROR);
    if (printisd(fp, fc.data.font.spacing) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",112,") < 0)
        return(ERROR);

    /* pitch required for fixed pitch fonts only */
    if (fc.data.font.spacing == FIXED)
    {
        if (printfsd(fp, quarter_dots_to_lpi(fc.data.font.pitch)) == ERROR)
            return(ERROR);
        if (fprintf(fp, ",104,") < 0)
            return(ERROR);
    }

    /* point size */
    if (printisd(fp, (UNSIGNEDINT)(0.5+quarter_dots_to_points(fc.data.font.height))) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",118,") < 0)
        return(ERROR);

    /* type style */
    if (printisd(fp, fc.data.font.style) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",115,") < 0)
        return(ERROR);

    /* stroke weight */
    if (printisd(fp, fc.data.font.stroke_weight) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",98,") < 0)
        return(ERROR);

    /* type face */
    if (printisd(fp, fc.data.font.typeface) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",116,") < 0)
        return(ERROR);

    /* quality */
    if (printisd(fp, fc.data.font.quality) == ERROR)
        return(ERROR);
    if (fprintf(fp, ",81\n") < 0)
        return(ERROR);

    /* print width of space (same as pitch) in 300dpi units */
    if (fprintf(fp, "%d=%d\n", ' ', fc.data.font.pitch / 4) < 0)
        return(ERROR);

    /* save pitch for fixed width fonts (0 indicates proportional font) */
    if (fc.data.font.spacing == FIXED)
        pitch = fc.data.font.pitch / 4;
    else
        pitch = 0;

    return(OK);
}

static int widthprint(fp)
FILE *fp;
{
    /*
     * Output the width information for a single character
     */

    switch (fc.data.character.format)
    {
    default:
        fprintf(stderr, WARNING_BAD_CHAR_FORMAT,
                    os_dir, os_file, fc.data.character.format);

        /* !! INTENTIONAL DROP-THROUGH !! */

    case LJCHARFORMAT:
        if (fprintf(fp, "%d=%d\n", charcode, (pitch > 0) ? (int)pitch :
                    (int)(fc.data.character.data.ljchar.delta_x/4)) < 0)
            return(ERROR);
        break;
    case DJCHARFORMAT:
    case DJPCHARFORMAT:
    case DJ500CHARFORMAT:
        /* only print the data in the first pass */
        if (pass_for_char_type(fc.data.character.data.djchar.char_type) == 0)
            if (fprintf(fp, "%d=%d\n", charcode, (pitch > 0) ? (int)pitch :
                    (   (int)fc.data.character.data.djchar.character_width
                      + (int)fc.data.character.data.djchar.left_offset
                      + (int)fc.data.character.data.djchar.right_offset)/2) < 0)
                return(ERROR);
        break;
    }

    return(OK);
}

static void jetprd()
{
    char inpath[OS_PATH_LEN], outpath[OS_PATH_LEN], *sp;
    FILE *infp, *outfp;
    int r, fdc_found = FALSE;

    /* build the input and output file paths */
    strcpy(inpath, os_dir);
    strcat(inpath, os_file);
    strcpy(outpath, os_file);
    if ((sp = strrchr(outpath, '.')) == NULL)
        strcat(outpath, output_suffix);
    else
        strcpy(sp, output_suffix);

    /* rudimentary check for input overwriting output */
    if (strcmp(inpath, outpath) == 0)
    {
        fprintf(stderr, ERROR_OVERWRITE, os_dir, os_file);
        return;
    }

    if (!(infp = fopen(inpath, "rb")))
    {
        fprintf(stderr, ERROR_OPEN_READ_FAILED, os_dir, os_file);
        return;
    }
    if (!(outfp = fopen(outpath, "w")))
    {
        fprintf(stderr, ERROR_OPEN_WRITE_FAILED, os_dir, os_file, outpath);
        fclose(infp);
        return;
    }

    while ((r = font_command_read(infp, &fc)) == OK)
    {
        switch(fc.command_type)
        {
        case FDC:
            if (headerprint(outfp) == ERROR)
            {
                fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                fclose(infp);
                fclose(outfp);
                return;
            }

            fdc_found = TRUE;

            break;
        case CCC:
            charcode = fc.number;
            break;
        case CDC:
            if (widthprint(outfp) == ERROR)
            {
                fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                fclose(infp);
                fclose(outfp);
                return;
            }
            break;
        }
    }


    /* error scenarios: (!fdc_found) means no font descriptor found,
       probably processing a text file; (r != EOF) means a font
       command escape sequence wasn't read in correctly, probably
       processing a binary file or a truncated soft font file */
    if (!fdc_found || r != EOF)
        fprintf(stderr, ERROR_FILE_READ_FAILED, os_dir, os_file);
    else
        fprintf(stderr, OK_JETPRD, os_dir, os_file, outpath);

    fclose(infp);
    fclose(outfp);
}

main(argc, argv)
int argc;
char *argv[];
{
    char c;

    /* stop getopt() printing errors */
    opterr = FALSE;
    while ((c = getopt(argc, argv, "t:h")) != EOF)
    {
        switch (c)
        {
        case 't':
            strncpy(output_suffix+1, optarg, SUFFIX_MAX-2);
            output_suffix[SUFFIX_MAX-1] = '\0';
            break;
        case 'h':
        case '?':
            /* help required, or invalid option specified */
            usage_wrong();
        }
    }

    /* must specify at least one file */
    if (optind >= argc)
        usage_wrong();

    /* process file arguments */
    if (os_findfiles((argc - optind), &argv[optind]) == ERROR)
        fprintf(stderr, ERROR_OUT_OF_HEAP);

    while (os_getfile() != EOF)
        jetprd();

    return(0);
}
