                                                                             
/*
**  Cassette And Cartridge Emulation Functions.
**
**  Copyright 1993-97 by Paul D. Burgin. All rights reserved.
*/

#include "stdio.h"
#include "dos.h"
#include "alloc.h"
#include "string.h"
#include "ctype.h"
#include "dir.h"
#include "conio.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"

extern outfile_status		open_outfile(unsigned char *fname,
								FILE **fptr, unsigned char *dev);

/* Cassette status. */
f_m				file_mode			= CLOSED;
boolean 		cassette_disabled	= FALSE;
FILE			*fp					= NULL;

/* Cassette filenames. */
boolean			ask_filenames = TRUE;
unsigned char	filemem[9];
unsigned char	*last_filename;
unsigned char	filename[MAXPATH+4+9] = "cassette\\";
unsigned char	gam_filename[22+6] = "cassette\\games\\";
unsigned char	arc_filename[22+8] = "cassette\\archive\\";
int				filename_start;

/* Cartridge/extra ROM info. */
unsigned char		cartname[9] = "None";
unsigned char		cartfname[MAXPATH+4] = "cartrige";
unsigned int		cartstart = 0xc000, cartend = 0xc000;
boolean				rom2loaded = FALSE;

/* Cassette surplus read bits buffer and count. */
signed int			buffer;
unsigned char		got_bits = 0;

/* Generic function to check whether a filename has an extension. */
unsigned char *strext(unsigned char *checkname)
{
	unsigned char *extloc = NULL;

	while (*checkname != '\0')
	{
		switch(*checkname)
		{
			case '\\':
			case '/':	extloc = NULL;
							break;

			case '.':	extloc = checkname;
							break;
		}
		checkname++;
	}
	return(extloc);
}

/* Read .DGN files into memory. */
boolean load_dgn_file(unsigned char *file_name, unsigned char dgn_dest)
{

	/* Values given to dgn_dest indicate: 0 - Main ROM file image. */
	/*                                    1 - D64 second ROM file. */
	/*                                    2 - Cartridge image.     */

	FILE			*ifp;
	unsigned char	fn[MAXPATH+4+9] = "cartrige\\";
	unsigned int	addrs, start;
	signed int		chr = 0;
	boolean			rom_format = FALSE;

	/* Append .DGN extension to name stub. */
	strcat(&fn[9], file_name);
	if (strext(&fn[9]) == NULL)
		strcat(&fn[9],".dgn");

	/* Open file - try both current and CARTRIGE directories. */
	if ((ifp = fopen(&fn[9], "rb")) == NULL)
	{
		if ((ifp = fopen(fn, "rb")) == NULL)
		{
			if (dgn_dest == 0)
			{
				/* Look for .ROM format file. */
				sprintf(&fn[9], "%s.rom", file_name);
				if ((ifp = fopen(&fn[9], "rb")) == NULL)
					ifp = fopen(fn, "rb");
				rom_format = TRUE;
			}
		}
	}
	if (ifp == NULL)
		return(FALSE);

	/* Check file format. */
	switch (rom_format)
	{
		case FALSE:	/* DGN format has signature 0xa1. */
					if (getc(ifp) != 0xa1)
					{
						fclose(ifp);
						return(FALSE);
					}

					/* Get file name. */
					for (addrs = 0; addrs < 8; addrs++)
					{
						if ((chr = getc(ifp)) == EOF)
						{
							fclose(ifp);
							return(FALSE);
						}
						else if (dgn_dest == 2)
							cartname[addrs] = chr;
					}

					/* Skip rest of header up to load address. */
					for (addrs = 0; addrs < 5; addrs++)
					{
						if (getc(ifp) == EOF)
						{
							fclose(ifp);
							return(FALSE);
						}
					}

					/* Get start address. */
					start = (getc(ifp) << 8) + getc(ifp);
					break;

		case TRUE:	/* ROM format begins 0x0080. */
					if (getc(ifp) == 0x00)
					{
						if (getc(ifp) == 0x80)
							break;
					}
					fclose(ifp);
					return(FALSE);
	}

	/* Note filename of cartridge files. */
	if (dgn_dest == 2)
		strcpy(cartfname, file_name);

	/* Override start address to 0x8000 for main ROM. */
	if (dgn_dest == 0)
		start = 0x8000;

	/* Load rest of .DGN file into memory. */
	for (addrs = start; chr != EOF; addrs++)
	{
		chr = getc(ifp);
		if (chr != EOF)
		{
			if (dgn_dest == 1)
				rom2[addrs-start] = chr;
			else if (addrs < 0xff00)
				memory[addrs] = chr;
		}
	}
	fclose(ifp);

	/* Remember start and end addresses for cartridge files. */
	if (dgn_dest == 2)
	{
		cartstart	= start;
		cartend		= addrs - 2;
	}
	return(TRUE);
}

/* Get a filename from the user, using a diaglogue box. */
boolean get_filename(f_m for_message, boolean first_attempt)
{
	if (first_attempt)
		strcpy(last_filename, &filename[9]);

	if (!input_box(last_filename,3,MAXPATH,"  CASSETTE FILENAME",
			((for_message == READING) ? for_reading_text : for_writing_text),""))
		return FALSE;

	strcpy(&filename[9], last_filename);

	/* Append .CAS to filename if necessary. */
	if (strext(&filename[9]) == NULL)
		strcat(filename,".cas");

	/* Use CASSETTE directory if no directory specified. */
	if ((strchr(&filename[9],'\\') == NULL)
			&& (strchr(&filename[9],'/') == NULL))
		filename_start = 0;
	else
		filename_start = 9;

	return(TRUE);
}

/* Doctor dragon filename characters for use with DOS. */
unsigned char doc(unsigned char chin)
{
	if ((chin >= '#') && (chin <= ')'))
		return(chin);
	if ((chin >= '0') && (chin <= '9'))
		return(chin);
	chin = toupper(chin);
	if ((chin >= '@') && (chin <= 'Z'))
		return(chin);
	return('_');
}

/* Assess the filename of the file that the emulated dragon is accessing. */
boolean set_filename(f_m file_mod)
{
	filename[9]  = doc(filemem[0]);
	filename[10] = doc(filemem[1]);
	filename[11] = doc(filemem[2]);
	filename[12] = doc(filemem[3]);
	filename[13] = doc(filemem[4]);
	filename[14] = doc(filemem[5]);
	filename[15] = doc(filemem[6]);
	filename[16] = doc(filemem[7]);

	/* Remove excess underscores from DOS version of filename. */
	if (filename[16] == '_')
	{
		filename[16] = 0;
		if (filename[15] == '_')
		{
			filename[15] = 0;
			if (filename[14] == '_')
			{
				filename[14] = 0;
				if (filename[13] == '_')
				{
					filename[13] = 0;
					if (filename[12] == '_')
					{
						filename[12] = 0;
						if (filename[11] == '_')
						{
							filename[11] = 0;
							if (filename[10] == '_')
							{
								filename[10] = 0;
							}
						}
					}
				}
			}
		}
	}
	else
		filename[17] = '\0'; /* Terminate 8 character filename. */

	/* Destroy stored filename to prevent overwriting. */
	strcpy(filemem,"        ");

	if (strcmp(&filename[9],"_") == 0)
		strcpy(&filename[9],"cassette");

	if (ask_filenames)
		return (get_filename(file_mod, TRUE));

	strcpy(last_filename, &filename[9]);
	strcat(filename,".cas");
	filename_start = 0;

	return(TRUE);
}

/* Request to close virtual cassette files. */
void close_files(boolean forced)
{
	if (file_mode != CLOSED)
	{
		/* Only close dos file if just written EOF block, or when forced. */
		if (((file_mode == WRITING) && (memory[0x7c] == 0xff)) || (forced))
		{
			fclose(fp);
			file_mode = CLOSED;
		}
	}
	else if (forced || (memory[0x7c] == 0xff))
		cassette_disabled = FALSE;
}

/* Open virtual cassette file for output. */
void open_out_file(void)
{
	boolean first_try = TRUE;

	if (!cassette_disabled)
	{
		file_mode = EDITING;
		while (file_mode == EDITING)
		{
			if (first_try ? set_filename(WRITING) : get_filename(WRITING, FALSE))
			{
				first_try = FALSE;
				switch (open_outfile(&filename[filename_start], &fp,NULL))
				{
					case OUT_OPEN:		/* Successfully opened output file. */
										file_mode = WRITING;
										break;

					case OUT_DECLINED:	/* Don't try curdir, it */
										/* would be confusing.  */
										break;

					default:			if (filename_start == 0)
										{
											switch (open_outfile(&filename[9], &fp,NULL))
											{
												case OUT_OPEN:		file_mode = WRITING;
																	break;

												case OUT_DECLINED:	break;

												default:			selection_box(2,14,0,FALSE,FALSE,FALSE,99,
																		"Failed to open",
																		"cassette file!");
																	break;
											}
										}
										else
											selection_box(2,14,0,FALSE,FALSE,FALSE,99,
												"Failed to open",
												"cassette file!");
										break;
				}
			}
			else
			{
				/* Cancelled - do not ask again. */
				cassette_disabled = TRUE;
				file_mode = CLOSED;
			}
		}
	}
}

/* Write a byte to a virtual cassette. */
void write_byte(void)
{
	/* Open output file if not already open. */
	if (file_mode == CLOSED)
		open_out_file();

	/* Re-open file if file is open in wrong mode. */
	else if (file_mode == READING)
	{
		fclose(fp);
		open_out_file();
	}

	/* Output byte to file if file opened successfully. */
	if (file_mode == WRITING)
		putc(a_reg.u_value, fp);
}

/* Open virtual cassette file for input. */
void open_in_file(void)
{
	boolean first_try = TRUE;

	if (!cassette_disabled)
	{
		file_mode = EDITING;
		while (file_mode == EDITING)
		{
			if (first_try ? set_filename(READING) : get_filename(READING, FALSE))
			{
				first_try = FALSE;
				file_mode = READING;
				if ((fp = fopen(&filename[9],"rb")) == NULL)
				{
					if ((fp = fopen(&filename[filename_start],"rb")) == NULL)
					{
						if ((filename_start == 0) && (strlen(filename) < 22))
						{
							strcpy(&gam_filename[15], &filename[9]);
							strcpy(&arc_filename[17], &filename[9]);
							if ((fp = fopen(gam_filename,"rb")) == NULL)
							{
								if ((fp = fopen(arc_filename,"rb")) == NULL)
									file_mode = EDITING;
							}
						}
						else
							file_mode = EDITING;
					}
				}
				if (file_mode == EDITING)
				{
					selection_box(2,14,0,FALSE,FALSE,FALSE,99,
						"Failed to open",
						"cassette file!");
				}
			}
			else
			{
				/* Cancelled - do not ask again. */
				cassette_disabled = TRUE;
				file_mode = CLOSED;
			}
		}
	}
}

/* Read a bit from a virtual cassette. */
void read_bit(void)
{
	/* Already got bits waiting in buffer? */
	if (got_bits < 1)
	{
		/* Nope, need to read next byte from disk. */
		/* Open input file if not already open. */
		if (file_mode == CLOSED)
			open_in_file();

		/* Re-open file if file is open in wrong mode. */
		else if (file_mode == WRITING)
		{
			fclose(fp);
			open_in_file();
		}

		/* Input byte from file if file opened successfully. */
		if (file_mode == READING)
		{
			/* Reached end of file? */
			if ((buffer = getc(fp)) == EOF)
			{
				/* End of file - close it. */
				fclose(fp);
				file_mode = CLOSED;
			}
			else
				got_bits = 8; /* Got valid byte into buffer; set count. */
		}
	}

	/* Use next available bit of byte held in buffer. */
	c_flag = (buffer & 0x01);
	buffer >>= 1;
	got_bits -= 1;
}

/* Read a byte from a virtual cassette. */
void read_byte(void)
{
	/* Open input file if not already open. */
	if (file_mode == CLOSED)
		open_in_file();

	/* Re-open file if file is open in wrong mode. */
	else if (file_mode == WRITING)
	{
		fclose(fp);
		open_in_file();
	}

	/* Input byte from file if file opened successfully. */
	if (file_mode == READING)
	{
		/* Use up any outstanding read bits before reading next byte. */
		a_reg.u_value = buffer & 0x7f;

		/* Reached end of file? */
		if ((buffer = getc(fp)) == EOF)
		{
			/* End of file - close it. */
			fclose(fp);
			file_mode = CLOSED;
		}

		/* Return read byte to emulator. */
		a_reg.u_value |= ((unsigned char)buffer) << got_bits;
		buffer >>= (8-got_bits);
	}
}
