/*******************  start of original comments  ********************/
/*
 * Written by Douglas Thomson (1989/1990)
 *
 * This source code is released into the public domain.
 */

/*
 * Name:    hardware independent screen IO module
 * Purpose: This file contains the code to interface the rest of the
 *           editor to the display and input hardware.
 * File:    hwind.c
 * Author:  Douglas Thomson
 * System:  this file is intended to be system-independent
 * Date:    October 2, 1989
 * Notes:   This is the only module that is allowed to call the hardware
 *           dependent display IO library.
 *          Typically, functions here check whether any action is
 *           necessary (for example, the cursor may already happen to be
 *           in the required position), call hardware dependent functions
 *           to achieve the required effect, and finally update status
 *           information about the current state of the terminal display.
 *          The idea behind this approach is to keep the hardware
 *           dependent code as small and simple as possible, thus making
 *           porting the code easier.
 */
/*********************  end of original comments   ********************/


/*
 * Some routines were added to display current editor modes in the lite bar
 * at the bottom of the screen. Other routines were rewritten in assembly.
 * I feel the need for speed.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This modification of Douglas Thomson's code is released into the
 * public domain, Frank Davis.  You may distribute it freely.
 */

#include <time.h>

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"
#include "define.h"


/*
 * Name:    format_time
 * Purpose: format the time according to a format string
 * Author:  Jason Hood
 * Date:    May 21, 1998
 * Passed:  format: format string
 *          buffer: output string
 * Notes:   buffer is assumed to be big enough to hold the string.
 */
void format_time( char *format, char *buffer )
{
time_t epoch;
struct tm *t;
int  num;
char **ptr;
static char *formats[] = { "%d", "%02d", "%2d" };
int  f;
int  j, k, len;

   *buffer = 0;
   if ((epoch = time( NULL )) != -1) {
      if ((t = localtime( &epoch )) != NULL) {
         k = 0;
         len = strlen( format );
         for (j = 0; j < len; ++j) {
            if (format[j] != '%')
               buffer[k++] = format[j];
            else {
               if (format[j+1] == '%')
                  buffer[k++] = '%';
               else {
                  num = -1;
                  ptr = NULL;
                  f = 0;
                  ++j;
                  if (format[j] == '0') {
                     f = 1;
                     ++j;
                  } else if (format[j] == '2') {
                     f = 2;
                     ++j;
                  }
                  switch (format[j]) {
                     case TS_DAYMONTH:  num = t->tm_mday;
                                        break;

                     case TS_DAYWEEK:   ptr = days[(f != 0)];
                                        num = t->tm_wday;
                                        break;

                     case TS_ENTER:     buffer[k++] = '\n';
                                        break;

                     case TS_12HOUR:    num = t->tm_hour;
                                             if (num > 12) num -= 12;
                                        else if (num == 0) num  = 12;
                                        break;

                     case TS_24HOUR:    num = t->tm_hour;
                                        break;

                     case TS_MONTHNUM:  num = t->tm_mon + 1;
                                        break;

                     case TS_MONTHWORD: ptr = months[(f != 0)];
                                        num = t->tm_mon;
                                        break;

                     case TS_MINUTES:   num = t->tm_min;
                                        break;

                     case TS_MERIDIAN:  ptr = time_ampm;
                                        num = (t->tm_hour >= 12);
                                        break;

                     case TS_SECONDS:   num = t->tm_sec;
                                        break;

                     case TS_TAB:       buffer[k++] = '\t';
                                        break;

                     case TS_YEAR2:     num = t->tm_year % 100;
                                        f   = 1;
                                        break;

                     case TS_YEAR:      num = t->tm_year + 1900;
                                        break;

                     case TS_ZONE:
                                     #if defined( __DOS16__ )
                                        ptr = (char**)tzname;
                                        num = t->tm_isdst;
                                     #else
                                        ptr = &t->tm_zone;
                                        num = 0;
                                     #endif
                                        break;

                     default:           buffer[k++] = '%';
                                        if (format[j-1] != '%')
                                           buffer[k++] = format[j-1];
                                        buffer[k++] = format[j];
                  }
                  if (ptr != NULL)
                     k += sprintf( buffer+k, "%s", ptr[num] );
                  else if (num != -1)
                     k += sprintf( buffer+k, formats[f], num );
               }
            }
         }
         buffer[k] = '\0';
      }
   }
}


/*
 * Name:    show_modes
 * Purpose: show current editor modes in lite bar at bottom of screen
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen (file_win_mem)
 * jmh 981129: update the "temporary" modes.
 * jmh 981130: split file_win_mem into file_win and mem_eq.
 */
void show_modes( void )
{
char temp[MAX_COLS+2];

   memset( temp, ' ', g_display.ncols );
   temp[g_display.ncols] = '\0';
   s_output( temp, g_display.mode_line, 0, g_display.mode_color );
   s_output( file_win, g_display.mode_line, 1, g_display.mode_color );
   s_output( mem_eq, g_display.mode_line, 12, g_display.mode_color );
   show_window_count( g_status.window_count );
   show_file_count( g_status.file_count );
   show_avail_mem( );
   show_tab_modes( );
   show_indent_mode( );
   show_sync_mode( );
   show_control_z( );
   show_insert_mode( );
   show_search_case( );
   show_wordwrap_mode( );
   show_trailing( );

#if !defined( __UNIX__ )
   if (g_status.graphic_chars > 0)
      show_graphic_chars( );
#endif
   if (g_status.cur_dir)
      show_cur_dir( );
   if (mode.record)
      show_recording( );
}


/*
 * Name:    show_file_count
 * Purpose: show number of open files in lite bar at bottom of screen
 * Passed:  fc:  file count - number of open files
 * Date:    June 5, 1991
 * jmh 981129: do nothing if graphic characters are on
 */
void show_file_count( int fc )
{
char temp[MAX_COLS+2];

   if (g_status.graphic_chars <= 0) {
      s_output( "  ", g_display.mode_line, 3, g_display.mode_color );
      s_output( my_ltoa( fc, temp, 10 ), g_display.mode_line, 3,
                g_display.mode_color );
   }
}


/*
 * Name:    show_window_count
 * Purpose: show total number of windows in lite bar at bottom of screen
 * Passed:  wc:  window count - visible and hidden.
 * Date:    September 13, 1991
 * jmh 981129: do nothing if graphic characters are on
 */
void show_window_count( int wc )
{
char temp[MAX_COLS+2];

   if (g_status.graphic_chars <= 0) {
      s_output( "  ", g_display.mode_line, 8, g_display.mode_color );
      s_output( my_ltoa( wc, temp, 10 ), g_display.mode_line, 8,
                g_display.mode_color );
   }
}


/*
 * Name:    show_avail_mem
 * Purpose: show available free memory in lite bar at bottom of screen
 * Date:    June 5, 1991
 * jmh 981129: do nothing if cursor direction is not normal
 */
void show_avail_mem( void )
{
long avail_mem;
char temp[MAX_COLS+2];

   if (g_status.cur_dir == CUR_RIGHT) {
      memset( temp, ' ', 8 );
      temp[8] = '\0';
      s_output( temp, g_display.mode_line, 14, g_display.mode_color );

      /*
       * reverse the sign if avail_mem is larger than a long.
       */
      avail_mem = my_heapavail( );
      if (avail_mem < 0)
         avail_mem = -avail_mem;

      if (avail_mem < 1048577L)
         my_ltoa( avail_mem, temp, 10 );
      else if (avail_mem < 67108865L) {
         avail_mem /= 1024L;
         my_ltoa( avail_mem, temp, 10 );
         strcat( temp, "k" );
      } else {
         avail_mem /= 1048576L;
         my_ltoa( avail_mem, temp, 10 );
         strcat( temp, "M" );
      }
      s_output( temp, g_display.mode_line, 14, g_display.mode_color );
   }
}


/*
 * Name:    show_tab_modes
 * Purpose: show smart tab mode in lite bar at bottom of screen
 * Date:    October 31, 1992
 */
void show_tab_modes( void )
{
char *blank_tab = "   ";
char ascii_tab[10];

   s_output( tabs, g_display.mode_line, 22, g_display.mode_color );
   s_output( mode.smart_tab ? smart : fixed, g_display.mode_line, 27,
             g_display.mode_color );
   s_output( tab_mode[mode.inflate_tabs], g_display.mode_line, 28,
             g_display.mode_color );
   s_output( blank_tab, g_display.mode_line, 29, g_display.mode_color );
   s_output( my_ltoa( mode.ptab_size, ascii_tab, 10), g_display.mode_line, 29,
             g_display.mode_color );
}


/*
 * Name:    show_indent_mode
 * Purpose: show indent mode in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_indent_mode( void )
{
   s_output( mode.indent ? indent : blank, g_display.mode_line, 32,
             g_display.mode_color );
}


/*
 * Name:    show_search_case
 * Purpose: show search mode in lite bar
 * Date:    June 5, 1991
 */
void show_search_case( void )
{
   s_output( mode.search_case == IGNORE ? ignore : match, g_display.mode_line,
             40, g_display.mode_color );
}


/*
 * Name:    show_sync_mode
 * Purpose: show sync mode in lite bar
 * Date:    January 15, 1992
 */
void show_sync_mode( void )
{
   s_output( mode.sync ? sync_on : sync_off, g_display.mode_line, 48,
             g_display.mode_color );
}


/*
 * Name:    show_wordwrap_mode
 * Purpose: display state of word wrap mode
 * Date:    June 5, 1991
 */
void show_wordwrap_mode( void )
{
   s_output( ww_mode[mode.word_wrap], g_display.mode_line, 54,
             g_display.mode_color );
}


/*
 * Name:     show_trailing
 * Purpose:  show state of trailing flag
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 */
void show_trailing( void )
{
   c_output( mode.trailing ? MODE_TRAILING : ' ', 65, g_display.mode_line,
             g_display.mode_color );
}


/*
 * Name:     show_control_z
 * Purpose:  show state of control z flag
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 */
void show_control_z( void )
{
   c_output( mode.control_z ? MODE_CONTROL_Z : ' ', 77, g_display.mode_line,
             g_display.mode_color );
}


/*
 * Name:     show_insert_mode
 * Purpose:  show insert mode in lite bar
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 */
void show_insert_mode( void )
{
   c_output( mode.insert ? MODE_INSERT : MODE_OVERWRITE,
#if defined( __UNIX__ )
                                                         78,
#else
                                                         79,
#endif
             g_display.mode_line, g_display.mode_color );
}


/*
 * Name:     show_graphic_chars
 * Purpose:  show state of graphic characters
 * Author:   Jason Hood
 * Date:     July 24, 1998
 * jmh 981129: overwrite the file/window count with a flashing message.
 */
void show_graphic_chars( void )
{
#if !defined( __UNIX__ )
   if (g_status.graphic_chars <= 0) {
      s_output( file_win, g_display.mode_line, 1, g_display.mode_color );
      show_window_count( g_status.window_count );
      show_file_count( g_status.file_count );
   } else {
      graphic_mode[GRAPHIC_SLOT] = graphic_char[g_status.graphic_chars-1][5];
      s_output( graphic_mode, g_display.mode_line, 1,
                g_display.mode_color | 0x80 );
   }
#endif
}


/*
 * Name:    show_cur_dir
 * Purpose: show the cursor update direction
 * Author:  Jason Hood
 * Date:    November 29, 1998
 * Notes:   Overwrite the memory display with a flashing direction string.
 */
void show_cur_dir( void )
{
char temp[MAX_COLS+2];

   if (g_status.cur_dir == CUR_RIGHT) {
      s_output( mem_eq, g_display.mode_line, 12, g_display.mode_color );
      show_avail_mem( );
   } else {
      memset( temp, ' ', 10 );
      temp[10] = '\0';
      s_output( temp, g_display.mode_line, 12, g_display.mode_color );
      s_output( cur_dir_mode[g_status.cur_dir], g_display.mode_line, 12,
                g_display.mode_color | 0x80 );
   }
}


/*
 * Name:    show_recording
 * Purpose: indicate a macro is being recorded
 * Author:  Jason Hood
 * Date:    November 29, 1998
 * Notes:   Overwrites the tabs, indent, ingore and sync displays.
 */
void show_recording( void )
{
   s_output( main15, g_display.mode_line, 22, g_display.mode_color | 0x80 );
   show_avail_strokes( 0 );
}


/*
 * Name:    my_scroll_down
 * Purpose: display a portion of a window
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Using the bios scroll functions causes a slightly noticable
 *            "flicker", even on fast VGA monitors.  This is caused by changing
 *            the high-lited cursor line to text color then calling the bios
 *            function to scroll.  Then, the current line must then be
 *            hilited.
 *          This function assumes that win->cline is the current line.
 */
void my_scroll_down( TDE_WIN *window )
{
int  i;
int  curl;
int  eof;
TDE_WIN w;              /* scratch window struct for dirty work */

   if (!window->visible  ||  !g_status.screen_display)
      return;
   dup_window_info( &w, window );
   curl = i = window->bottom_line + 1 - window->cline;
   eof =  w.ll->prev != NULL;
   for (; i>0; i--) {
      if (w.ll->len != EOF) {
         /*
          * if this is window->cline, do not show the line because we
          * show the curl at the end of this function.  don't show it twice
          */
         if (i != curl)
            update_line( &w );
      } else if (eof < 2) {
         show_eof( &w );
         ++eof;
      } else
         window_eol_clear( &w, g_display.text_color );
      if (w.ll->next != NULL)
         w.ll = w.ll->next;
      ++w.cline;
      ++w.rline;
   }
   show_curl_line( window );
}


/*
 * Name:    combine_strings
 * Purpose: stick 3 strings together
 * Date:    June 5, 1991
 * Passed:  buff:    buffer to hold concatenation of 3 strings
 *          s1:  pointer to string 1
 *          s2:  pointer to string 2
 *          s3:  pointer to string 3
 */
void combine_strings( char *buff, char *s1, char *s2, char *s3 )
{
   assert( strlen( s1 ) + strlen( s2 ) + strlen( s3 ) < g_display.ncols );
   strcpy( buff, s1 );
   strcat( buff, s2 );
   strcat( buff, s3 );
}


/*
 * Name:    make_ruler
 * Purpose: make ruler with tabs, tens, margins, etc...
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void make_ruler( TDE_WIN *window )
{
register TDE_WIN *win;
char num[20];
register unsigned char *p;
int  len;
int  col;
int  i;
int  mod;

   win = window;

   /*
    * need to have a least two lines in a window when we display a ruler.
    */
   if (win->bottom_line - win->top_line < 1)
      win->ruler = FALSE;
   if (win->ruler) {

      /*
       * find the width of the window and fill the ruler with dots.
       */
      len = win->end_col + 1 - win->start_col;

      assert( len >= 0 );
      assert( len <= g_display.ncols );

      memset( win->ruler_line, RULER_FILL, len );
      win->ruler_line[len] = '\0';
      col = win->bcol+1;

      assert( col >= 1 );
      assert( col <= MAX_LINE_LENGTH );

      for (p=(unsigned char *)win->ruler_line; *p; col++, p++) {

         /*
          * put a tens digit in the tens column
          */
         mod = col % 10;
         if (mod == 0) {
            my_ltoa( col/10, num, 10 );

            /*
             * let the margin chars have precedence over tens digit
             */
            for (i=0; num[i] && *p; col++, i++) {
               if (col == mode.left_margin+1)
                  *p = LM_CHAR;
               else if (col == mode.right_margin+1) {
                  if (mode.right_justify == TRUE)
                     *p = RM_CHAR_JUS;
                  else
                     *p = RM_CHAR_RAG;
               } else if (col == mode.parg_margin+1)
                  *p = PGR_CHAR;
               else
                  *p = num[i];
               p++;
            }

            /*
             * we may have come to the end of the ruler in the for loop.
             */
            if (*p == '\0')
               break;
         } else if (mod == 5)
            *p = RULER_TICK;
         if (col == mode.parg_margin+1)
            *p = PGR_CHAR;
         if (col == mode.left_margin+1)
            *p = LM_CHAR;
         else if (col == mode.right_margin+1) {
            if (mode.right_justify == TRUE)
               *p = RM_CHAR_JUS;
            else
               *p = RM_CHAR_RAG;
         }
      }
   }
}


/*
 * Name:    show_ruler
 * Purpose: show ruler with tens, margins, etc...
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_ruler( TDE_WIN *window )
{
   if (window->ruler && window->visible) {
      s_output( window->ruler_line, window->top_line, window->start_col,
                g_display.ruler_color );
#if defined( __UNIX__ )
      refresh( );
#endif
   }
}


/*
 * Name:    show_ruler_char
 * Purpose: show ruler character under ruler pointer
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_ruler_char( TDE_WIN *window )
{
register TDE_WIN *win;
char c;

   win = window;
   if (win->ruler && win->visible) {
      c = win->ruler_line[win->ccol - win->start_col];
      c_output( c, win->ccol, win->top_line, g_display.ruler_color );
   }
#if defined( __UNIX__ )
   refresh( );
#endif
}


/*
 * Name:    show_ruler_pointer
 * Purpose: show ruler pointer
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_ruler_pointer( TDE_WIN *window )
{
   if (window->ruler && window->visible) {
      c_output( RULER_PTR, window->ccol, window->top_line,
                g_display.ruler_pointer );
#if defined( __UNIX__ )
      refresh( );
#endif
   }
}


/*
 * Name:    show_all_rulers
 * Purpose: make and show all rulers in all visible windows
 * Date:    June 5, 1991
 */
void show_all_rulers( void )
{
register TDE_WIN *wp;

   wp = g_status.window_list;
   while (wp != NULL) {
      make_ruler( wp );
      if (wp->visible) {
         show_ruler( wp );
         show_ruler_pointer( wp );
      }
      wp = wp->next;
   }
}


/*
 * Name:    show_help
 * Purpose: display help screen for certain functions
 * Author:  Jason Hood
 * Date:    August 9, 1998
 * Passed:  nothing (uses g_status.command)
 * Returns: OK if screen was restored;
 *          ERROR if screen was redrawn.
 * Notes:   center the help display horizontally (assuming all strings are
 *           equal length), but above center vertically.
 *          if there's enough memory, save and restore screen contents,
 *           otherwise just redraw the screen.
 */
int  show_help( void )
{
char **pp;       
int  row;
int  col;
Char *buffer;
int  wid;
int  len;
int  rc = OK;

   switch (g_status.command) {
      case FindRegX:
      case RepeatFindRegX:
      case DefineRegXGrep:
      case RepeatGrep:
         pp = regx_help;
         break;

      case StampFormat:
         pp = stamp_help;
         break;

      case BorderBlock:
         pp = border_help;
         break;

      default:
         pp = NULL;
         break;
   }
   if (pp == NULL)
      return( OK );

   for (len = 0; pp[len] != NULL; ++len) ;
   wid = strlen( pp[0] );
   row = (g_display.mode_line - len) / 2;
   col = (g_display.ncols - wid) / 2;
   if (row < 0)
      row = 0;
   if (col < 0)
      col = 0;

   buffer = (Char *)malloc( wid * len * sizeof(Char) );
   if (buffer != NULL)
      save_area( buffer, wid, len, row, col );

   xygoto( -1, -1 );
   show_strings( pp, len, row, col );
   getkey( );

   if (buffer != NULL) {
      restore_area( buffer, wid, len, row, col );
      free( buffer );
   } else if (g_status.current_window != NULL) {
      redraw_screen( g_status.current_window );
      rc = ERROR;
   }
   return( rc );
}


/*
 * Name:    show_strings
 * Purpose: display strings from an array
 * Author:  Jason Hood
 * Date:    August 9, 1998
 * Passed:  strings: the array to display
 *          num:     number of strings to display
 *          row:     starting line position
 *          col:     starting column position
 * Returns: nothing
 * Notes:   Assumes there are at least num strings in the array.
 *          Will stop should the bottom line be reached.
 *          Uses the help color.
 */
void show_strings( char **strings, int num, int row, int col )
{
int max_row = g_display.mode_line + 1;

   if (max_row > row + num)
      max_row = row + num;
   for (; row < max_row; strings++, row++)
      s_output( *strings, row, col, g_display.help_color );
}
