/*  CDtest 1.3

	Tests if simultaneous I/O corrupts CD-ROM reading.  If so, likely causes are the
	PCI RZ-1000 EIDE chip, the CMD-640 EIDE chip or a faulty DMA controller.
	Runs under DOS, DESQview, Windows, NT and OS/2 in DOS emulation mode.

	This program may be copied freely and used for any purpose except military.
	(c) copyright 1995,1996 by
	Roedy Green
	Canadian Mind Products
	POB 707 Quathiaski Cove
	Quadra Island BC Canada
	V0P 1N0
	(250) 285-2954
	Internet Roedy@Bix.Com

	compiled under SMALL model for DOS under Borland C++ 4.5

	It works by reading a set of files repeatedly from the CD-ROM drive
	and computing a checksum on each file.	It reads the same set of
	files multiple times with as much I/o as possible
	interfering.  It ensures the same checksum is computed each time.

	Version history
	1.3 - embed Quathiaski address.
	1.2 - consistent use of term CMD-640.
	1.1 - add warning about turnin on background execution.
	1.0 - first version, cloned and modified from EIDEtest
	*/

/* D E F I N E S */

#define BufferSizeInBytes 31744
/* size of buffer for reading the files.
	This size should generate lots of physical I/O.
	Smaller sizes might come out of cache.
	Bigger sizes do only a few big I/Os */

#define TestSizeInBytes 7616560L
/* aggregate bytes to read for the test per pass, spread over many files. */

#define MaxFiles 2000
/* maximum number of files on CDROM that will be processed.
	Up to this many will have checksums computed.
	Usually TestSizeInBytes will be the limiting factor. */

#define StartDecorate "\xb0\xb1\xb2\xdb"
/* line of pretty blobs in oemfont to attract attention */
#define EndDecorate   "\xdb\xb2\xb1\xb0"

#define Esc '\x1b'

/* T Y P E D E F S */

typedef enum TESTRESULT
{
   PASS, FAIL, INCOMPLETE
} TESTRESULT;

/* I N C L U D E S */

#include <stdlib.h>
// e.g. exit, min, max

#include <stdio.h>
// e.g. FILE, printf, fopen

#include <conio.h>
// e.g. getch kbhit crlscr

#include <alloc.h>
// e.g. malloc free

// #include <dos.h>
// _dos_findfirst _dos_findnext

#include <string.h>
// strcpy

#include "typedef.hpp"
// WORD BYTE BOOL

#include "filetree.hpp"
// class to process a tree of files

#include "CRC.hpp"
// CRC polynomical calculations.

/* G L O B A L S */

char WildCard[] = "E:\\*.*";
/*  Wildcard to describe all files on the EIDE CD-ROM */

/* P R O T O T Y P E S */

int main( int argc,  char *argv[] );
void Banner ( void );
void beep (void );
int pause ( void );
int DoesOSDoSimultaneousIO ( void );
int CheckForEsc ( void );

/* ***************************************** */

/* C L A S S  D E C L A R A T I O N */

/* ***************************************** */

class VERIFY : public FILETREE
    {
    /* class to repeatedly read files off the CD-ROM and make sure they are
       the same each time.  Computes and compares checksums */
 public:

    VERIFY(void);

    ~VERIFY() {};

    void VERIFY::Phase1 ( void );

    void VERIFY::Phase2 ( void );

    void DisplayResults( void );

    BOOL ProcessOneFile ( char * QualFileName, long FileSize );
    /* returns true to abort entire process.
       Computes checksum for one file.
		 It gets passed a fully qualified FileName and its size */

    CRC OneFileCRC;
    /* CRC object to compute CRC for current file. */

    int Phase; /* 1=compute CRC or 2=verify CRC */

    TESTRESULT TestResult;
    /* result of the entire test */

    BOOL TestDone;
    /* true if no more processing to do */

    WORD ExpectedCRC [MaxFiles];
    /* CRC expected for each file */

    int  FileIndex; /* which file we are processing on CD ROM */

    long ToGo;
    /* How many more bytes we need to read for the pass.
       We don't read the entire CD-ROM, just the first 7 MB or so */

};

/* ***************************************** */

/* F U N C T I O N   D E F I N I T I O N S */

int main( int argc,  char *argv[] )

{
    Banner();

    if ( argc != 2 /*  0=CDtest.Exe 1=E: */ )
	{
	printf("Oops!  usage:  CDtest  E:"
	    "\nwhere E: is the EIDE CD-ROM to test.");
	exit (2);
	}

    /* replace first letter of WildCard with Drive Letter */
    /* e.g. WildCard might now read:  E:\*.* */
    WildCard[0] = *argv[1];

    VERIFY Verify; /* object to do all the CD-ROM reading tests */

    printf("\nYou must run CDtest with background execution configured on."
	"\nAbort now if you have not done so."
	"\n");

    if ( pause() == 2 )  Verify.TestDone = TRUE; /* user ESC abort */
    else
	{
	printf("\nInsert any CD-ROM in drive %s", argv[1]);
	if ( pause() == 2 )  Verify.TestDone = TRUE; /* user ESC abort */
	}

    /* do Phase 1 */
    Verify.Phase1();
    beep(); /* announce Phase1 over */

    /* do Phase 2 */
    Verify.Phase2();
    beep(); /* announce Phase2 over */

    Verify.DisplayResults(); /* report good or bad news */

    return (int) Verify.TestResult ; /* 0 = pass, 1 = fail, 2 = incomplete */

} /* end main */

/*****************/

void Banner(void)
{
    /* display copyright banner. */

    clrscr(); /* clear screen */

    printf("\n" StartDecorate " CDtest 1.3 " EndDecorate
	"\n"
	"\nFreeware to test for the flawed EIDE controllers such as the"
	"\nPC-Tech RZ-1000 and the CMD-640."
	"\nRuns under DESQview, Windows, Windows For WorkGroups, Windows-95, NT"
	"\nand OS/2 all in DOS emulation mode."
	"\n"
	"\nCopyright (c) 1995,1996 Canadian Mind Products"
	"\nPOB 707 Quathiaski Cove, Quadra Island, BC Canada V0P 1N0"
	"\n(250) 285-2954                              roedy@bix.com"
	"\nMay be used freely for non-military use only."
	"\n\n");

} /* end Banner */

/*****************/

VERIFY::VERIFY(void)
{
    /* constructor */
    TestResult = INCOMPLETE;
    TestDone = FALSE;
    FileIndex = 0;
    ToGo =TestSizeInBytes;

}
/*****************/

void VERIFY::Phase1 ( void )
{
    Phase = 1;

    if (TestDone) return;

    clrscr(); /* clear screen */

    printf("\n" StartDecorate " PHASE 1 " EndDecorate
	"\nDuring phase 1, do not run any other programs."
	"\nEspecially avoid using the tape or floppy drive."
	"\nShutdown all background programs BEFORE you proceed."
	"\nIf you can't get them all, not to worry;"
	"\nthe test results will still be valid."
	"\n");

    if ( pause() == 2 )
	{
	TestDone = TRUE;
	return; /* user abort */
	}
    printf("\nPhase 1 in progress.  Please stand by...\n");

    /* read about 7 Mb of files, computing checksums. User may abort by hitting ESC */
    FileIndex = 0;
    ToGo =TestSizeInBytes;
    ForEachFile ( WildCard,/* ADVSHR */  "?000??", TRUE /* recurse */ );
    return;
}	/* end Phase1 */

/*****************/

void VERIFY::Phase2 ( void )

{
    if (TestDone) return;

    clrscr(); /* clear screen */

    Phase = 2;

    switch (DoesOSDoSimultaneousIO())
	{

	case 0: /* DOS */
	    clrscr(); /* clear screen */
	    printf( "\n" StartDecorate " PHASE 2 " EndDecorate
		"\n"
		"\nDURING phase 2, do as many of these tests with as you can"
		"\neach with background execution configured ON,"
		"\nto generate simultaneous I/O:"
		"\n"
		"\n- Backup some files to floppy with MWBackup, MSBackup, Norton Backup"
		"\n  or other high performance floppy backup program with fast-mode"
		"\n  simultaneous disk I/O configured."
		"\n- Backup a file using your mag tape backup software with fast-mode"
		"\n  simultaneous disk I/O configured."
		"\n- Exercise any simultaneous I/O devices e.g. modem, mouse or sound card."
		"\n");
	    break;

	case 1: /* OS */
	    clrscr(); /* clear screen */
	    printf( "\n" StartDecorate " PHASE 2 " EndDecorate
		"\n"
		"\nDURING phase 2, do as many of these tests with as you can"
		"\neach with background execution configured ON,"
		"\nto generate simultaneous I/O:"
		"\n"
		"\n- Format a floppy."
		"\n- Backup some files to floppy with MWBackup, MSBackup, Norton Backup"
		"\n  or other high performance floppy backup program with fast-mode"
		"\n  simultaneous disk I/O configured."
		"\n- Backup a file using your mag tape backup software with fast-mode"
		"\n  simultaneous disk I/O configured."
		"\n- Scan a document with your scanner."
		"\n- Browse some files on other hard disks and CD-ROM."
		"\n- Exercise any simultaneous I/O devices e.g. modem, mouse or sound card."
		"\n");
	    break;

	case 2: /* esc */
	    TestDone = TRUE;
	    return;
	}

    printf("\nYour experiments may run slowly because of CDtest's I/O overhead."
	"\n"
	"\nBEGIN YOUR EXPERIMENTS NOW!"
	"\n"
	"\nTake all the time you want -- days even."
	"\nWhen you have finished all your tests, come back and hit the space bar."
	"\n"
	"\nPhase 2 in progress..."
	"\n");

    while ( ! TestDone )
	{
	/* keep repeating the test till user hits ESC or space bar */
	FileIndex = 0;
	ToGo = TestSizeInBytes;
	ForEachFile ( WildCard,/* ADVSHR */  "?000??", TRUE /* recurse */ );
	}
    return;
} /* Phase2 */

/*****************/

void VERIFY::DisplayResults( void )
{

    clrscr(); /* clear screen */

    printf( "\n" StartDecorate " TEST RESULTS " EndDecorate
	"\n");

    switch (TestResult)
	{
	case PASS:
	    printf("\nCDtest passed.  No flaws found."
		"\nYou may still have a flawed EIDE controller chip with a"
		"\nsuccessful software bypass."
		"\nKeep in mind, that unless you carefully followed the"
		" instructions on the,"
		"\nscreen, especially running all tests with background execution"
		"\nconfigured on, the results of this test are meaningless."
		"\n");
	    break;

	case FAIL:
	    printf("\nCDtest failed."
		"\nIf you have a PCI motherboard the failure is probably due to a faultly"
		"\nPC-Tech RZ-1000 or CMD-640 EIDE controller chip."
		"\nIt could also be due to overheating, a faulty DMA controller,"
		"\nor software bugs.");
	    break;

	case INCOMPLETE:
	default:
	    printf("\nCDtest not completed."
		"\nPossibly a user abort or problems reading the CD-ROM drive."
		"\n");
	    break;

	}
    printf("\nPlease admire the test results.\n");
    pause();

} /* end DisplayResults */

/* ***************************************** */

BOOL VERIFY::ProcessOneFile ( char * QualFileName, long FileSize )
{
    /* Gets called once for each file on the CD-ROM matching
       WildCard and attribute filter.
       It opens the file, and computes the checksum.
       In Phase 1, computes checksums.	In Phase 2, verifies them */

    FILE *TempFile;

    BYTE *Buffer = new BYTE [BufferSizeInBytes];

    if (FileSize)
	{
	TempFile = fopen(QualFileName,"rb"); /* read binary mode */
	if (TempFile )
	    {
	    /* got an acceptable file, start a fresh checksum */
	    OneFileCRC.Reset();

	    size_t BytesActuallyRead;
	    size_t BytesToRead;
	    long   ToGoInThisFile = min ( FileSize, ToGo );

	    /* read test file, till eof, or we have enough bytes, failure or user abort */
	    while (ToGoInThisFile && !TestDone)
		{
		BytesToRead = (size_t) min ( (long) BufferSizeInBytes ,ToGoInThisFile );
		BytesActuallyRead = fread(Buffer, 1, BytesToRead,  TempFile);
		if (BytesActuallyRead == BytesToRead)
		    {
		    OneFileCRC.Append(Buffer,BytesActuallyRead);
		    ToGo -= BytesActuallyRead;
		    ToGoInThisFile -= BytesActuallyRead;
		    }
		else
		    {
		    TestResult = FAIL;
		    TestDone = TRUE;
		    break;
		    }

		if (CheckForEsc()) TestDone = TRUE;
		/* check for user abort every block read */
		}
	    fclose(TempFile);
	    }
	else
	    {
	    /* file would not open */
	    TestResult = FAIL;
	    TestDone = TRUE;
	    }
	}
    else /* NOP */; /* ignore 0-length files */

    if (!TestDone) switch (Phase)
	{
	case 1:
	    /* save computed checksum */
	    ExpectedCRC [FileIndex++] = OneFileCRC.Result();
	    break;

	case 2:
	    /* ensure checksum is same as last time */
	    if (ExpectedCRC [FileIndex++] == OneFileCRC.Result())
		TestResult = PASS;
	    else
		{
		/* crc check failed */
		TestResult = FAIL;
		TestDone = TRUE;
		}
	    break;
	}

    delete Buffer;

    return (BOOL) ((ToGo <= 0) || (FileIndex >= MaxFiles) || TestDone);
    /* quit calling this function as soon as enough bytes read,
       enough files read,
       or test aborted by user or  test failure. */

} /* end ProcessOneFile */

/* ***************************************** */

void beep (void)
{
    /* long beep to attract attention */
    printf("\x07\x07\x07\x07");
} /* end beep */

/*****************/

int pause (void)
{
    /*
	 returns 1=user hit space
    returns 2=user hit ESC
    */
    int Result = 0;
    /* pause until the user hits the space bar */
    printf ("\nWhen you are ready to proceed, hit the space bar; hit ESC to abort");

    while (!Result)
	{
	switch (getch())
	    {
	    case ' ':
		Result = 1; /*	space we are done */
		break;
	    case Esc:
		Result = 2; /* Esc, Abort */
		break;
	    default:
		printf("\x07" );  /* beep to indicate char ignored. */
		break;
	    }
	}
    printf("\r                                                                  \n\n");
    /* erase the  prompt line, to ack the space bar */
    return Result;

} /* end pause */

/*****************/

int DoesOSDoSimultaneousIO (void)
{

    /*
      returns 0= no, DESQiew, Windows,...
      returns 1= yes. OS/2, NT...
      returns 2=user hit ESC
	   */
    int Result = -1;

    printf( "\nDoes your operating system do more than one I/O at a time?"
	"\nAnswer N for DOS, DESQview, Windows, Windows for Workgroups or Windows-95."
	"\nAnswer Y for OS/2, Warp, NT, Linux, SCO XENIX or UNIX.");
    /* don't advertise the ESC possibility */

    while (Result == -1)
	{
	switch (getch())
	    {
	    case 'N':
	    case 'n':
		Result = 0; /*	No */
		break;
	    case 'Y':
	    case 'y':
		Result = 1; /* Yes */
		break;
	    case Esc:
		Result = 2; /* Esc, Abort */
		break;
	    default:
		printf("\x07" );  /* beep to indicate char ignored. */
		break;
	    }
	}

    return Result;

} /* end DoesOSDoSimultaneousIO */

/* ***************************************** */

int CheckForEsc (void)
{

    /*
 returns 0=user did nothing.  We don't wait for him to hit a key.
 returns 1=user hit space
 returns 2=user hit ESC
 */
    int Result = 0;

    if (kbhit())
	{
	switch (getch())
	    {
	    case ' ':
		Result = 1; /*	space we are done */
		break;
	    case Esc:
		Result = 2; /* Esc, Abort */
		break;
	    default:
		printf("\x07" );  /* beep to indicate char ignored. */
		Result = 0;
		break;
	    }
	}
    else
	Result = 0; /* no key hit */
    return Result;

} /* end CheckForEsc */

/* ***************************************** */

/* -30- */
