/*

Micro BASIC Interpeter (lib.c)

This is the libraries module. Keeps all the MBASIC internal functions.

*** Copyleft - Andre Murta - August 1992/December 2002 ***

Adapted from Herbert Schildt Little C (Dr. Dobbs Journal, August 1989)

*/

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include "mbasic.h"
#include "structures.h"

extern char *prog;	/* Expression to be analyzed */
extern char *p_buf;
extern jmp_buf e_buf;

/* Variables structure */
extern struct variables vars[NUM_VARS];

/* data files control structure */
extern struct dtfiles data_files[NUM_FILES];

/* commands table */
extern struct commands table[];

/* user defined functions */
extern struct functions func_table[];

/* this variable is the seed which will be used by RANDOMIZE and RND */
extern unsigned int random_seed;
double last_rand_number=0.0; /* used by rnd */

extern char token[];	/* Holds string represtation of token */
extern char token_type;
extern char tok;

extern int get_token();
extern void serror(), get_exp(), putback();
extern char *find_var_string();

/* MBasic chr() function */
char *func_chr()
{
    struct variables val;
    static char ret[2];
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    if ((val.value.dv < 0) || (val.value.dv > 255)) serror(21);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    sprintf(ret, "%c", (int)val.value.dv);
    
    return ret;
}

/* MBasic asc() function */
double func_asc()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)(val.value.sv[0]);
}

/* MBasic len() function */
double func_len()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)(strlen(val.value.sv));
}

/* MBasic inkey() function */
double func_inkey()
{
    char c;

    c = getch();
    
    return (double)(c);
}

/* MBasic left() function */
char *func_left()
{
    struct variables str,amount;
    static char s[STRING_LEN];
    int i;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&str);
    if(str.type != STRING_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&amount);
    if(amount.type != NUMERIC_VAR) serror(18);
    if ((amount.value.dv < 0) || (amount.value.dv > 255)) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    /* printf("(left) Value: %d\n", (int)val.value.dv); */
    for(i=0; i<(int)(amount.value.dv); i++) {
        s[i] = str.value.sv[i];
    }
    s[i] = '\0';

    return s;
}

/* MBasic mid() function */
char *func_mid()
{
    struct variables str,start,amount;
    static char s[STRING_LEN];
    int i;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&str);
    if(str.type != STRING_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&start);
    if(start.type != NUMERIC_VAR) serror(18);
    if ((start.value.dv < 0) || (start.value.dv > 255)) serror(21);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&amount);
    if(amount.type != NUMERIC_VAR) serror(18);
    if ((amount.value.dv < 0) || (amount.value.dv > 255)) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    /* printf("(left) Value: %d\n", (int)val.value.dv); */
    for(i=(int)(start.value.dv-1); i<(int)((start.value.dv-1) + amount.value.dv); i++) {
        s[i-(int)(start.value.dv-1)] = str.value.sv[i];
    }
    s[i-(int)(start.value.dv-1)] = '\0';

    return s;
}

/* MBasic right() function */
char *func_right()
{
    struct variables str,amount;
    static char s[STRING_LEN];
    int i,count;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&str);
    if(str.type != STRING_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&amount);
    if(amount.type != NUMERIC_VAR) serror(18);
    if ((amount.value.dv < 0) || (amount.value.dv > 255)) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    count = 0;

    /* printf("(left) Value: %d\n", (int)val.value.dv); */
    for(i=strlen(str.value.sv)-(int)amount.value.dv; i<strlen(str.value.sv); i++) {
        s[count] = str.value.sv[i];
        count++;
    }
    s[count] = '\0';

    return s;
}

/* MBasic sqr() function */
double func_sqr()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return sqrt(val.value.dv);
}

/* MBasic log() function */
double func_log()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return log(val.value.dv);
}

/* MBasic exp() function */
double func_exp()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return exp(val.value.dv);
}

/* MBasic sin() function */
double func_sin()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return sin(val.value.dv);
}

/* MBasic cos() function */
double func_cos()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return cos(val.value.dv);
}

/* MBasic tan() function */
double func_tan()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return tan(val.value.dv);
}

/* MBasic atn() function */
double func_atn()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return atan(val.value.dv);
}

/* MBasic pi() function */
double func_pi()
{
    return 3.14159;
}

/* MBasic abs() function */
double func_abs()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return fabs(val.value.dv);
}

/* MBasic sgn() function */
double func_sgn()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    if(val.value.dv < 0) return -1;
    else if(val.value.dv == 0) return 0;
    else if(val.value.dv > 0) return 1;
}

/* MBasic int() function */
double func_int()
{
    struct variables val;
    int ival;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    ival = (int)val.value.dv;

    return (double)ival;
}

/* MBasic timer() function */
double func_timer()
{
    return (double)rawclock();
}

/* MBasic val() function */
double func_val()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return atof(val.value.sv);
}

/* MBasic str() function */
char *func_str()
{
    struct variables val;
    static char ret[STRING_LEN];
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    if ((val.value.dv < 0) || (val.value.dv > 255)) serror(21);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    sprintf(ret, "%g", val.value.dv);
    
    return ret;
}

/* MBasic lcase() function */
char *func_lcase()
{
    struct variables str;
    static char s[STRING_LEN];
    int i;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&str);
    if(str.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    for(i=0; i<strlen(str.value.sv); i++) {
        s[i] = tolower(str.value.sv[i]);
    }
    s[i] = '\0';

    return s;
}

/* MBasic ucase() function */
char *func_ucase()
{
    struct variables str;
    static char s[STRING_LEN];
    int i;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&str);
    if(str.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    for(i=0; i<strlen(str.value.sv); i++) {
        s[i] = toupper(str.value.sv[i]);
    }
    s[i] = '\0';

    return s;
}

/* MBasic time() function */
char *func_time()
{
    time_t now;
    char ttime[30];
    static char ctime[12];

    time(&now);

    strcpy(ttime, asctime(localtime(&now)));

    ctime[0] = ttime[11];
    ctime[1] = ttime[12];
    ctime[2] = ttime[13];
    ctime[3] = ttime[14];
    ctime[4] = ttime[15];
    ctime[5] = ttime[16];
    ctime[6] = ttime[17];
    ctime[7] = ttime[18];
    ctime[8] = '\0';

    return ctime;
}

/* MBasic date() function */
char *func_date()
{
    time_t now;
    char month[4];
    char tdate[30];
    static char cdate[10];

    time(&now);

    strcpy(tdate, asctime(localtime(&now)));

    month[0]=tdate[4]; month[1]=tdate[5]; month[2]=tdate[6]; month[3]='\0';

    /* printf("\nasctime:[%s]month:[%s]\n",asctime(localtime(&now)), month); */
    

    if(!strcmp(month, "Jan")) {
        cdate[0] = '0'; cdate[1] = '1';
    } else if(!strcmp(month, "Feb")) {
        cdate[0] = '0'; cdate[1] = '2';
    } else if(!strcmp(month, "Mar")) {
        cdate[0] = '0'; cdate[1] = '3';
    } else if(!strcmp(month, "Abr")) {
        cdate[0] = '0'; cdate[1] = '4';
    } else if(!strcmp(month, "Mai")) {
        cdate[0] = '0'; cdate[1] = '5';
    } else if(!strcmp(month, "Jun")) {
        cdate[0] = '0'; cdate[1] = '6';
    } else if(!strcmp(month, "Jul")) {
        cdate[0] = '0'; cdate[1] = '7';
    } else if(!strcmp(month, "Aug")) {
        cdate[0] = '0'; cdate[1] = '8';
    } else if(!strcmp(month, "Sep")) {
        cdate[0] = '0'; cdate[1] = '9';
    } else if(!strcmp(month, "Oct")) {
        cdate[0] = '1'; cdate[1] = '0';
    } else if(!strcmp(month, "Nov")) {
        cdate[0] = '1'; cdate[1] = '1';
    } else if(!strcmp(month, "Dez")) {
        cdate[0] = '1'; cdate[1] = '2';
    } else {
        printf("Invalid date range (internal error)\n");
        exit(1);
    }

    cdate[2] = '-';
    cdate[3] = tdate[8];
    cdate[4] = tdate[9];
    cdate[5] = '-';
    cdate[6] = tdate[20];
    cdate[7] = tdate[21];
    cdate[8] = tdate[22];
    cdate[9] = tdate[23];
    cdate[10] = '\0';

    return cdate;
}

/* MBasic asin() function */
double func_asin()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return asin(val.value.dv);
}

/* MBasic acos() function */
double func_acos()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return acos(val.value.dv);
}

/* MBasic sinh() function */
double func_sinh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return sinh(val.value.dv);
}

/* MBasic cosh() function */
double func_cosh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return cos(val.value.dv);
}

/* MBasic tanh() function */
double func_tanh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return tanh(val.value.dv);
}


/* MBasic log10() function */
double func_log10()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return log10(val.value.dv);
}

/* MBasic floor() function */
double func_floor()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return floor(val.value.dv);
}

/* MBasic rnd() function */
double func_rnd()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    if(val.value.dv < 0) {
        srand(random_seed);
        last_rand_number = (double)rand()/(double)(RAND_MAX+1);
        return -(last_rand_number);
    } else  if(val.value.dv == 0) {
        return -(last_rand_number);
    } else if(val.value.dv > 0) {
        last_rand_number = (double)rand()/(double)(RAND_MAX+1);
        return -(last_rand_number);
    }
}

/* MBasic etype() function */
double func_etype()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)val.type;
}

/* MBasic chdir() function */
double func_chdir()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)chdir(val.value.sv);
}


/* MBasic rmdir() function */
double func_rmdir()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)rmdir(val.value.sv);
}

/* MBasic mkdir() function */
double func_mkdir()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)mkdir(val.value.sv);
}

/* MBasic col() function */
double func_col()
{
    return (double)wherex();
}

/* MBasic row() function */
double func_row()
{
    return (double)wherey();
}

/* MBasic cols() function */
double func_cols()
{
    return (double)ScreenCols();
}

/* MBasic rows() function */
double func_rows()
{
    return (double)ScreenRows();
}

/* MBasic stack() function */
double func_stack()
{
    return (double)stackavail();
}

/* MBasic movetext() function */
double func_copytext()
{
    struct variables left, top, right, bottom, destleft, desttop;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&left);
    if(left.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&top);
    if(top.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&right);
    if(right.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&bottom);
    if(bottom.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&destleft);
    if(destleft.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&desttop);
    if(desttop.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return (double)movetext((int)left.value.dv, (int)top.value.dv,
                            (int)right.value.dv, (int)bottom.value.dv,
                            (int)destleft.value.dv, (int)desttop.value.dv);
}

/* MBasic rename() function */
double func_rename()
{
    struct variables oldname, newname;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&oldname);
    if(oldname.type != STRING_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&newname);
    if(newname.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return (double)rename(oldname.value.sv, newname.value.sv);
}

/* MBasic kill() function */
double func_kill()
{
    struct variables filename;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&filename);
    if(filename.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return (double)unlink(filename.value.sv);
}

/* MBasic loc() function */
double func_loc()
{
    struct variables filepointer;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(18);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    if(data_files[(int)filepointer.value.dv].recsize && data_files[(int)filepointer.value.dv].method == RANDOM)
        return (double)ftell(data_files[(int)filepointer.value.dv].fp) / data_files[(int)filepointer.value.dv].recsize;
    else
        return (double)ftell(data_files[(int)filepointer.value.dv].fp);
}

/* MBasic lof() function */
double func_lof()
{
    struct variables filepointer;
    long current_point, total_value;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(18);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    /* to keep the current file pointer */
    current_point = ftell(data_files[(int)filepointer.value.dv].fp);

    if(data_files[(int)filepointer.value.dv].recsize && data_files[(int)filepointer.value.dv].method == RANDOM) {
        /* get ammount */
        fseek(data_files[(int)filepointer.value.dv].fp, 0L, SEEK_END);
        total_value = ftell(data_files[(int)filepointer.value.dv].fp) / data_files[(int)filepointer.value.dv].recsize;
        /* restore file pointer */
        fseek(data_files[(int)filepointer.value.dv].fp, current_point, SEEK_SET);
        return (double)total_value;
    } else {
        /* get ammount */
        fseek(data_files[(int)filepointer.value.dv].fp, 0L, SEEK_END);
        total_value = ftell(data_files[(int)filepointer.value.dv].fp);
        /* restore file pointer */
        fseek(data_files[(int)filepointer.value.dv].fp, current_point, SEEK_SET);
        return (double)total_value;
    }
}

/* MBasic eof() function */
double func_eof()
{
    struct variables filepointer;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(18);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    return (double)feof(data_files[(int)filepointer.value.dv].fp);
}

/* MBasic instr() function */
double func_instr()
{
    struct variables start, sourcestr, searchstr;
    register int i,j;
    char buffer1[STRING_LEN], buffer2[STRING_LEN];

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&start);
    if(start.type != NUMERIC_VAR) serror(18);
    if(start.value.dv < 1 && start.value.dv > STRING_LEN) serror(21);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&sourcestr);
    if(sourcestr.type != STRING_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&searchstr);
    if(searchstr.type != STRING_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    /* printf("Strlen: (%d)\n", strlen(sourcestr.value.sv)); */

    for(i=(int)start.value.dv - 1; i<strlen(sourcestr.value.sv); ++i) {
        memset(buffer1, '\0', strlen(buffer1));
        memset(buffer2, '\0', strlen(buffer2));

        for(j=i; j<strlen(searchstr.value.sv)+i; ++j)
            if(j<strlen(sourcestr.value.sv))
                sprintf(buffer1, "%s%c", buffer1, sourcestr.value.sv[j]);
            else return 0;

        if(!strcmp(buffer1, searchstr.value.sv)) return i+1;
        /* printf("<%s> - <%s> - (%d)\n", buffer1, searchstr.value.sv, i); */
    }

    return 0;
}

/* MBasic spc() function */
char *func_spc()
{
    struct variables numbers;
    char buffer[STRING_LEN];

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&numbers);
    if(numbers.type != NUMERIC_VAR) serror(18);
    if(numbers.value.dv < 1 && numbers.value.dv > STRING_LEN) serror(21);

    get_token();
    if(tok != RPAREN) serror(20);

    memset(buffer, '\0', strlen(buffer));
    memset(buffer, ' ', (int)numbers.value.dv);
}

/* MBasic tab() function */
char *func_tab()
{
    struct variables val;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    gotoxy((int)val.value.dv, wherey());

    return "";
}

/* MBasic asinh() function */
double func_asinh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return asinh(val.value.dv);
}

/* MBasic acosh() function */
double func_acosh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return acosh(val.value.dv);
}

/* MBasic atn2() function */
double func_atn2()
{
    struct variables x,y;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&y);
    if(y.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&x);
    if(x.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return atan2(y.value.dv, x.value.dv);
}

/* MBasic atnh() function */
double func_atnh()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return atanh(val.value.dv);
}

/* MBasic log2() function */
double func_log2()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return log2(val.value.dv);
}

/* MBasic ceil() function */
double func_ceil()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return ceil(val.value.dv);
}

/* MBasic fmod() function */
double func_fmod()
{
    struct variables x,y;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&x);
    if(x.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&y);
    if(y.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);
    
    return fmod(x.value.dv, y.value.dv);
}

/* MBasic hypot() function */
double func_hypot()
{
    struct variables x,y;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&x);
    if(x.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&y);
    if(y.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return hypot(x.value.dv, y.value.dv);
}

/* MBasic pow() function */
double func_pow()
{
    struct variables x,y;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&x);
    if(x.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&y);
    if(y.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    return pow(x.value.dv, y.value.dv);
}

/* MBasic pow10() function */
double func_pow10()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return pow10(val.value.dv);
}

/* MBasic pow2() function */
double func_pow2()
{
    struct variables val;
    
    get_token();
    if(tok != LPAREN) serror(19);
    
    get_exp(&val);
    if(val.type != NUMERIC_VAR) serror(18);
    
    get_token();
    if(tok != RPAREN) serror(20);
    
    return pow2(val.value.dv);
}

/* MBasic gettext() function */
char *func_gettext()
{
    struct variables x1,y1,x2,y2;
    static char temp_buffer[STRING_LEN], buffer[STRING_LEN];
    register int i,count=0;

    memset(buffer, '\0', STRING_LEN);
    memset(temp_buffer, '\0', STRING_LEN);

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&x1);
    if(x1.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&y1);
    if(y1.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&x2);
    if(x2.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&y2);
    if(y2.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok != RPAREN) serror(20);

    gettext((int)x1.value.dv, (int)y1.value.dv, (int)x2.value.dv, (int)y2.value.dv, temp_buffer);

    for(i=0; i<strlen(temp_buffer); ++i)
    {
        if(temp_buffer[i] != 7) {
            buffer[count] = temp_buffer[i];
            count++;
        }
    }

    return buffer;
}
