; This program is made by Daniel Horchner.
; email: dbjh@gmx.net
;
; The check for a PS/2 mouse is made, because I have seen int 33h
; implementations that do return the right IRQ for a PS/2 mouse and others
; that don't (e.g. something like IRQ 255).
; This program works under DOS/Win 3.1/Win 9x/Win NT. Under DOS 'mouseIRQ' is
; called on a mouse event, under Windows 'mouse' is called.
; All Windows versions provide a PM int 33h interface, but in true Micro$hit
; fashion all are somewhat different. Under Win 3.1 the PM interface is only
; available unless a RM mouse driver is present. Under Win NT you have to
; force a 16-bit return... So, in order for this program to work under
; Win NT, remove the semicolon in front of the line with 'db 66h'.

        .386p
        locals

code32  segment para public use32
        assume cs:code32, ds:code32, ss:code32

NO_EXTERN       =       0
include ..\raw32.inc
include raw32mod.inc

;32-bit data
palette         dd      0               ; pointer to palette buffer
font            dd      0               ; pointer to font table
mouseX          dw      0
mouseY          dw      0
button          dw      0               ; mouse button status
mouseXptr       dd      0
mouseYptr       dd      0
buttonptr       dd      0
activity        db      0               ; flag that indicates mouse action
Xmsg            db      'x coordinate=',0
Ymsg            db      'y coordinate=',0
msg             db      'Press right mouse button to exit.',0
nomouse_msg     db      'No mouse (driver) installed.$'
bgforx          db      3*8 * 8 dup(0)  ; coordinate # has 3 digits, 8x8 font
bgfory          db      3*8 * 8 dup(0)
cursor  label                           ; mouse cursor
db      4, 4, 4, 4, 4, 4, 0, 0
db      4, 4, 4, 4, 4, 0, 0, 0
db      4, 4, 4, 4, 0, 0, 0, 0
db      4, 4, 4, 4, 4, 0, 0, 0 
db      4, 4, 0, 4, 4, 4, 0, 0
db      4, 0, 0, 0, 4, 4, 4, 0
db      0, 0, 0, 0, 0, 4, 4, 4
db      0, 0, 0, 0, 0, 0, 4, 4
bgforcursor     db      8*8 dup(0)
mouseIRQvect    db      0               ; int called on mouse IRQ
org_mouseIRQ    dd      0               ; offset and selector of original IRQ
                dw      0               ;  handler
RMsegment       dw      0
RMsegsize       dd      RMsegend

;32-bit code
main:
        mov v86r_ax,0                   ; ax=0 -> Mouse reset/Mouse installed
        mov al,33h                      ;  check
        int RMCALL_VECT
        cmp v86r_ax,0ffffh              ; ax=ffffh -> mouse installed
        je short @@hide
        mov edx,offset nomouse_msg
        call dosprintx
        jmp exit

@@hide:
        mov v86r_ax,2                   ; ax=2 -> Hide mouse cursor
        int RMCALL_VECT

        cmp windows,0
        jne @@setptrs

        mov v86r_ax,24h                 ; ax=24h -> Get driver version, mouse
        int RMCALL_VECT                 ;  type & IRQ number
        cmp v86r_ch,4                   ; ch=mouse type; 4=PS/2 mouse
        jne short @@IRQok
        mov bl,4                        ; PS/2 mouse at IRQ 12 (slave IRQ 4)
        add bl,IRQ8_vect                ; bl=int called on IRQ 12
        jmp short @@IRQdone
@@IRQok:
        mov bl,v86r_cl                  ; cl=mouse IRQ
        add bl,IRQ0_vect                ; bl=int called on mouse IRQ
@@IRQdone:
        mov mouseIRQvect,bl
        call getvect                    ; Get original IRQ handler address
        mov word ptr org_mouseIRQ[4],cx
        mov org_mouseIRQ[0],edx

                                        ; Copy RMseg to low memory
        mov eax,RMsegsize
        add eax,15                      ; extra mem for aligning on paragraph
        call getlomem
        jc @exit
        add eax,0fh                     ; code32 is aligned on paragraph
        and al,0f0h                     ; Ensure RMseg aligned on paragraph
        add eax,code32a
        mov edi,eax
        mov es,zerosel                  ; lomem via zerosel under dosemu/NT
        mov esi,RMseg                   ; esi=32-bit segment address!
        shl esi,4
        sub esi,code32a
        mov ecx,RMsegsize
        cld
        rep movsb
                                        ; Set ptrs to variables in low mem
        mov mouseXptr,offset RMmouseX
        add mouseXptr,eax
        mov mouseYptr,offset RMmouseY
        add mouseYptr,eax
        mov buttonptr,offset RMbutton
        add buttonptr,eax

        shr eax,4
        mov RMsegment,ax
        jmp short @@go_set_font

@@setptrs:
        mov eax,code32a
        mov mouseXptr,offset mouseX
        add mouseXptr,eax
        mov mouseYptr,offset mouseY
        add mouseYptr,eax
        mov buttonptr,offset button
        add buttonptr,eax
        
@@go_set_font:
        call setfont                    ; Set font variable (for put funcs)

        mov v86r_ax,13h
        mov al,10h
        int RMCALL_VECT

        mov eax,256*3                   ; 256 palette entries; for each entry
        call gethimem                   ;  3 bytes: red, green and blue
        jc @exit                        ;  intensity of color
        mov palette,eax

        mov es,data32sel
        mov edi,palette
        add edi,(256-64)*3              ; Use last 64 palette entries
        mov ecx,64*3/4                  ; 64 red, green and blue byte values
        xor eax,eax
        rep stosd                       ; Clear palette entries (set to black)

        mov edi,palette
        add edi,(256-64)*3 + 2          ; blue component of color is 3rd byte
        mov ecx,63                      ; cl=intensity of blue (63 at start;
@@next_entry:                           ;  standard VGA has a 6-bit DAC -> 64
        mov [edi],cl                    ;  intensities possible for red, green
        add edi,3                       ;  and blue each)
        loop @@next_entry               ; Decrease blue intensity one step
                                        ; 64th intensity is omitted; already
                                        ;  set to black when clearing palette
                                        ;  entries
        cli
        mov al,256-64                   ; al=base pal entry that rgb values
        mov edx,3c8h                    ;  are written for; last 64 entries
        out dx,al
        mov esi,palette
        add esi,(256-64)*3
        mov ecx,64*3
        mov edx,3c9h
        rep outsb                       ; Output byte at ds:esi to port dx
        sti

        mov es,zerosel
        mov edi,0a0000h
        mov al,256-64                   ; al=palette entry
        mov ah,al
        mov dx,ax
        shl eax,16
        mov ax,dx                       ; eax=4 bytes with palette entry
        xor dl,dl                       ;  (draw 4 pixels at once; stos_d_)
@@next_color:
        mov ecx,320/4                   ; 320 pixels per line
        rep stosd                       ; Draw line
        mov ecx,320/4 
        rep stosd                       ; Draw line
        mov ecx,320/4
        rep stosd                       ; Draw line
        add eax,01010101h               ; 1 dword = 4 pixels; next pal entry
        inc dl
        cmp dl,64                       ; All intensities drawn?
        jb short @@next_color
        mov ecx,(200-192)*320/4
        mov eax,0ffffffffh
        rep stosd                       ; Draw remaining lines the same color
;
        mov esi,offset Xmsg
        mov ecx,0
        mov edx,0
        mov bl,0fh
        call gputstr
        mov esi,offset Ymsg
        add edx,8
        call gputstr
        mov esi,offset msg
        add edx,16
        call gputstr

        mov edi,offset bgforx           ; Save background for x coordinate #
        mov ecx,3*8 shl 16 + 13*8       ; 3 chars, strlen("x coordinate=")=13
        mov edx,8 shl 16
        call getbitmap
        mov edi,offset bgfory           ; Save background for y coordinate #
        add edx,8                       ; y message 8 pixels below x message
        call getbitmap
                                        ; Init vars used by mouse 'handler'
        mov v86r_ax,3                   ; ax=3 -> Get mouse pos and but stat
        mov al,33h
        int RMCALL_VECT
        mov edi,offset bgforcursor
        mov ecx,8 shl 16
        mov cx,v86r_cx
        mov ebx,mouseXptr
        mov gs:[ebx],cx
        shr cx,1                        ; mouse x range=640 video x range=320
        mov edx,8 shl 16
        mov dx,v86r_dx
        mov ebx,mouseYptr
        mov gs:[ebx],dx
        call getbitmap

        cmp windows,0
        jne short @@winhandler

        mov v86r_ax,0ch                 ; ax=0ch -> Set mouse user subroutine
        mov ax,RMsegment
        mov v86r_es,ax
        mov v86r_dx,offset mouse_sr     ; es:dx=pointer to routine
        mov v86r_cx,0ffh                ; cx=user interrupt mask
        mov al,33h
        int RMCALL_VECT

        mov bl,mouseIRQvect
        mov cx,cs
        mov edx,offset mouseIRQ         ; Install new 'handler' to avoid poll
        call setvect
        jmp short main_loop

@@winhandler:
        mov eax,0ch                     ; ax=0ch -> Set mouse user subroutine
        mov es,code32sel
        mov edx,offset mouse            ; es:edx=pointer to routine
        mov ecx,0ffh                    ; cx=user interrupt mask
        int 33h                         ; Yep, Windows has a PM int 33h

main_loop:
        mov edi,buttonptr
        cmp word ptr gs:[edi],2         ; Right mouse button pressed?
        je @exit

        cmp activity,0                  ; Mouse activity?
        je short @@updatedone
        and eax,0ffffh                  ; High word eax will be undefined ->
        mov edi,mouseXptr               ;  clear it (for gputnumdec)
        mov ax,gs:[edi]
        call update_x
        mov edi,mouseYptr
        mov ax,gs:[edi]
        call update_y
        mov activity,0
@@updatedone:
        jmp short main_loop

@exit:                                  ; Restore original mouse handler
        cmp windows,0
        jne short @@set_text
        mov bl,mouseIRQvect
        mov cx,word ptr org_mouseIRQ[4]
        mov edx,org_mouseIRQ[0]
        call setvect

@@set_text:
        mov v86r_ax,3                   ; Set video mode 3 (text 80x25x16)
        mov al,10h
        int RMCALL_VECT
        mov v86r_ax,0                   ; ax=0 -> Mouse reset
        mov al,33h
        int RMCALL_VECT
        jmp exit                        ; Return to real/V86 mode

;
update_x:
        push ecx edx esi

        mov esi,offset bgforx
        mov ecx,3*8 shl 16 + 13*8       ; 3 chars, strlen("x coordinate=")=13
        mov edx,8 shl 16
        cli                             ; ?, necessary on very fast PC's
        call putbitmap

        mov ebx,30fh
        call gputnumdec
        pop esi edx ecx
        sti
        ret

;
update_y:
        push ecx edx esi

        mov esi,offset bgfory
        mov ecx,3*8 shl 16 + 13*8       ; 3 chars, strlen("y coordinate=")=13
        mov edx,8 shl 16 + 8
        cli                             ; ?, necessary on very fast PC's
        call putbitmap

        mov ebx,30fh
        call gputnumdec
        pop esi edx ecx
        sti
        ret

;
mouseIRQ:
        push eax ecx edx esi edi ds gs
        mov ds,cs:data32sel
        mov gs,zerosel

        mov activity,1                  ; Indicate activity (mouse IRQ raised
                                        ;  by movement and button press)
        mov ecx,8 shl 16                ; mouse cursor is an 8x8 bitmap
        mov edx,8 shl 16
        mov edi,mouseXptr
        mov cx,gs:[edi]
        mov edi,mouseYptr
        mov dx,gs:[edi]
        shr cx,1                        ; mouse x range=640 video x range=320
        mov esi,offset bgforcursor      ; esi=address of saved background

        push 3002h
        call fword ptr org_mouseIRQ     ; Call original PM IRQ handler which
                                        ;  calls RM IRQ handler which in turn
                                        ;  calls 'mouse_sr' which updates
                                        ;  button, mouseX and mouseY

        call putbitmap                  ; Restore background (after calling
        mov edi,mouseXptr               ;  RM handler to minimize flicker)
        mov cx,gs:[edi]
        mov edi,mouseYptr
        mov dx,gs:[edi]
        shr cx,1
        mov edi,esi
        call getbitmap                  ; Save background
        mov esi,offset cursor
        call putbitmap                  ; Draw mouse cursor

@@exit:
        pop gs ds edi esi edx ecx eax
        iretd

;
mouse:
        mov ds,cs:data32sel
        mov gs,zerosel

        mov activity,1

        push ebx ecx edx                ; Save mouse status info
        mov ecx,8 shl 16
        mov edx,8 shl 16
        mov cx,mouseX
        mov dx,mouseY
        shr cx,1
        mov esi,offset bgforcursor
        call putbitmap

        pop edx ecx ebx
        mov button,bx
        mov mouseX,cx
        mov mouseY,dx
        shr cx,1
        and ecx,0ffffh
        or ecx,8 shl 16
        and edx,0ffffh
        or edx,8 shl 16
        mov edi,esi
        call getbitmap
        mov esi,offset cursor
        call putbitmap

;       db 66h                          ; Win NT needs a 16-bit return...
        retf

;
; Draw a picture element on the output screen ;) (macro)
; In:
;   gs = zerosel
;
;   x = x coordinate (dword)
;   y = y coordinate (dword)
;   color = palette entry (byte)
;   reg1 = 1st temporary register (dword)
;   reg2 = 2nd temporary register (dword)
; Out:
;   reg1 = ?
;   reg2 = ?
;
putpixel        macro   x, y, color, reg1, reg2
        mov reg1,y
        mov reg2,y
        shl reg1,8                      ; y * 256
        shl reg2,6                      ; y * 64
        add reg1,reg2                   ; y * 320
        add reg1,x                      ; reg1 = y * 320 + x
        mov byte ptr gs:[0a0000h+reg1],color
endm

;
; Set font variable to linear address of 8x8 font in RAM
; In:
;   ds = data32sel
; Out:
;   font = linear address of font
;   eax = ?
;   ecx = ?
;   esi = ?
;   edi = ?
;
setfont:
        push es
        mov v86r_ax,1130h               ; al=30 -> Get cur char table info
        mov v86r_bh,3                   ; ROM 8x8 character table pointer
        mov al,10h
        int RMCALL_VECT
        movzx eax,v86r_es               ; es:bp=pointer to table
        shl eax,4
        and v86r_ebp,0ffffh             ; Clear high word for "add eax,..."
        add eax,v86r_ebp
                                        ; Copy char table to RAM -> faster
        mov esi,eax                     ;  access -> faster writing of text
        mov eax,8*256                   ; 256 diff chars; 1 byte per scanline
        mov ecx,8*256/4
        call gethimem
        mov es,data32sel
        mov ds,zerosel
        mov edi,eax
        rep movsd
        mov ds,cs:data32sel
        add eax,code32a
        mov font,eax                    ; Linear address of font in RAM
        pop es
        ret

;
; Graphically put 0 terminated string to screen
; In:
;   ds:esi = address of 0 terminated string
;   gs = zerosel
;   bl = character color
;   cx = x coordinate to put top left corner of string
;   dx = y coordinate to ,,  ,,   ,,    ,,   ,,   ,,
;   font = linear address of 8x8 font
;   video mode = 8 bits per pixel
;
gputstr:
        pushad
        shl ecx,16
        mov cx,dx                       ; ecx=x:y coordinates to put string
        push ecx
        push es
        mov ax,ds                       ; First, get string length
        mov es,ax                       ; es=ds for scasb
        mov edi,esi
        mov ecx,0ffffffffh              ; Search max 4GB
        mov al,0                        ; Scan for 0 (=end of string)
        cld
        repne scasb                     ; Compare al with es:edi
        not ecx
        dec ecx                         ; When scasb stops, edi points 1 byte
                                        ;  past the 0; that's 1 byte too far
        pop es                          ; ecx=string length

        pop edx                         ; edx=x:y coordinates to put string
        mov ah,bl                       ; bl=character color

        push esp                        ; Minor adjustment for 'gputnumdec'
        push esi
putchar:
        pop esi
        mov al,[esi]                    ; al=ASCII # of character to print
        mov ebp,0                       ; ebp=scanline in character to print
        inc esi
        push esi                        ; Save pointer to string
        push ecx                        ; Save counter of chars left to print
@@next_scanline:
        movzx edi,al
        shl edi,3
        add edi,cs:font                 ; edi=font address + ASCII # * 8
        mov cl,gs:[edi+ebp]             ; cl=1 scanline of char (_8_x8...)
        mov ch,0                        ; ch=pixel # in scanline of char to
@@next_pixel:                           ;  print
        test cl,80h                     ; The font is a character bit mask ->
        jz short @@pixel_done           ;  Draw only the character itself
        mov edi,edx                     ; edx=x:y coordinates
        and edi,0ffffh                  ; edi=y coordinate
        mov esi,edx
        shr esi,16                      ; esi=x coordinate
        add edi,ebp                     ; edi=y + scanline # in character
        movzx ebx,ch
        add esi,ebx                     ; esi=x + pixel # in scanline of char
        putpixel esi,edi,ah, ebx,edi    ; edi= y==temp reg2 -> Is allowed
@@pixel_done:
        shl cl,1                        ; Shift next bit to test position
        inc ch
        cmp ch,8                        ; Last pixel in char scanline printed?
        jb short @@next_pixel
        inc ebp
        cmp ebp,8                       ; Last scanline printed?
        jb short @@next_scanline
        add edx,80000h                  ; x=x+8 -> following char will be
        pop ecx                         ;  printed next to last one
        dec ecx
        jnz short putchar               ; Print next character

        pop esi
        pop esp                         ; Minor adjustment for 'gputnumdec'
        popad
        ret

;
; Graphically put number in eax to screen in decimal
; In:
;   eax = number
;   ds = data32sel; Add ds=ss code and ds-restore code if ds!=data32sel
;   gs = zerosel
;   bl = character color
;   bh = minimal number of characters to write (1 dword on stack per char)
;   cx = x coordinate to put top left corner of string
;   dx = y coordinate to ,,  ,,   ,,    ,,   ,,   ,,
;   font = linear address of font
;   video mode = 8 bits per pixel
;
gputnumdec:
        pushad
        mov ebp,esp                     ; Save stack pointer
        mov esi,ebx                     ; Save bl (color)
        shl ecx,16
        mov cx,dx                       ; ecx=x:y coordinates to put string
        mov edi,ecx                     ; Save coordinates

        mov cl,bh
        mov ebx,10                      ; Divide by 10
        xor ch,ch                       ; count of numbers pushed on stack
@@push_digit:
        xor edx,edx                     ; Reset edx: eax = _edx_:eax / ebx
        div ebx
        push edx                        ; remains in edx
        inc ch
        cmp eax,0                       ; Are there any digits left?
        jne short @@push_digit
        sub cl,ch                       ; If cl > ch add zero's
        jbe short @@start_pop_digit
@@extra_digit:
        push 0
        inc ch
        dec cl
        jnz short @@extra_digit
@@start_pop_digit:
        mov edx,esp                     ; Save esp
        mov ebx,esp                     ; String has to 'grow' up in mem
        movzx eax,ch
        push eax                        ; Save string length
@@pop_digit:
        mov eax,ss:[edx]                ; Get next digit, but don't give up
        add edx,4                       ;  stack space (POP would)
        add al,'0'                      ; Convert to ASCII
        mov ss:[ebx],al                 ; Store character on stack
        inc ebx
        dec ch
        jnz short @@pop_digit           ; ch=count of numbers pushed on stack

        mov eax,esi
        mov ah,al                       ; ah=color
        mov edx,edi                     ; edi=x:y coordinates
        pop ecx                         ; ecx=string length
        mov esi,esp                     ; esp=start of string
;       mov bx,ss                       ; Already done in extender (ss=ds);
;       mov ds,bx                       ;  str ptr at [esp] must point in ds
                                        ; Avoid redundant POPAD (see note)
        push ebp                        ; Provide ptr to PUSHAD stack frame
        push esi                        ; 'putchar' expects a ptr to the
        jmp putchar                     ;  string at [esp]
                                        ; The RET of gputstr returns to the
                                        ;  calling code

;
; Draw a pixel from a buffer to the screen (macro)
; In:
;   gs = zerosel
;   ds:source = address of buffer to get pixel from ('source' must be a reg)
;
;   x = x coordinate (dword)
;   y = y coordinate (dword)
;   reg1 = 1st temporary register (dword)
;   reg2 = 2nd temporary register (dword)
;   reg3 = 3rd temporary register (byte); can be the same as reg2
; Out:
;   reg1 = ?
;   reg2 = ?
;   reg3 = ?
;
; Note:
;   Multiple statements of this macro have to be separated by a global label,
;   because this macro contains a local label.
;
putpixelfrombuf macro   x, y, source, reg1, reg2, reg3
        cmp x,320                       ; Clip x coordinate
        jae short @@exitm
        cmp y,200                       ; Clip y coordinate
        jae short @@exitm
        mov reg1,y
        mov reg2,y
        shl reg1,8                      ; y * 256
        shl reg2,6                      ; y * 64
        add reg1,reg2                   ; y * 320
        add reg1,x                      ; reg1 = y * 320 + x
        mov reg3,byte ptr [source]
        mov byte ptr gs:[0a0000h+reg1],reg3
@@exitm:
endm

;
; Put a bitmap to the screen
; In:
;   gs = zerosel (for putpixelfrombuf)
;   ecx = bitmap width:x coordinate of top left (high word:low word)
;   edx = bitmap height:y coordinate of top left
;   ds:esi = address of bitmap
;
; Note:
;   This routine doesn't draw pixels in the buffer that have a value of 0.
;
putbitmap:
        pushad
        mov ebp,edx
        shr ebp,16
        and edx,0ffffh                  ; edx=y coordinate to start with
        add ebp,edx                     ; ebp=y maximum

        mov edi,ecx
        shr edi,16
        and ecx,0ffffh
        add edi,ecx                     ; edi=x maximum
        push ecx                        ; [esp]=x coordinate to start with
@@next_y:
        cmp edx,ebp
        jge short @@exit
        mov ecx,[esp]
@@next_x:
        cmp ecx,edi
        jge short @@linedone
        cmp byte ptr [esi],0            ; Enable 'transparent' bitmaps
        je short @@putpixeldone
        putpixelfrombuf ecx,edx,esi, ebx,eax,al
@@putpixeldone:
        inc esi
        inc ecx
        jmp short @@next_x
@@linedone:
        inc edx
        jmp short @@next_y

@@exit:
        add esp,4
        popad
        ret

;
; Get a pixel from the screen and put it in a buffer (macro)
; In:
;   gs = zerosel
;   ds:dest = address of buffer to put pixel to ('dest' must be a reg)
;
;   x = x coordinate (dword)
;   y = y coordinate (dword)
;   reg1 = 1st temporary register (dword)
;   reg2 = 2nd temporary register (dword)
;   reg3 = 3rd temporary register (byte); can be the same as reg2
; Out:
;   reg1 = ?
;   reg2 = ?
;   reg3 = ?
;
; Note:
;   Multiple statements of this macro have to be separated by a global label,
;   because this macro contains a local label.
;
getpixeltobuf   macro   x, y, dest, reg1, reg2, reg3
        cmp x,320                       ; Clip x coordinate
        jae short @@exitm
        cmp y,200                       ; Clip y coordinate
        jae short @@exitm
        mov reg1,y
        mov reg2,y
        shl reg1,8                      ; y * 256
        shl reg2,6                      ; y * 64
        add reg1,reg2                   ; y * 320
        add reg1,x                      ; reg1 = y * 320 + x
        mov reg3,byte ptr gs:[0a0000h+reg1]
        mov byte ptr [dest],reg3
@@exitm:
endm

;
; Get a bitmap from the screen
; In:
;   gs = zerosel (for getpixeltobuf)
;   ecx = bitmap width:x coordinate of top left (high word:low word)
;   edx = bitmap height:y coordinate of top left
;   ds:edi = address of buffer to store bitmap
;
getbitmap:
        pushad
        mov ebp,edx
        shr ebp,16
        and edx,0ffffh                  ; edx=y coordinate to start with
        add ebp,edx                     ; ebp=y maximum

        mov esi,ecx
        shr esi,16
        and ecx,0ffffh
        add esi,ecx                     ; esi=x maximum
        push ecx                        ; [esp]=x coordinate to start with
@@next_y:
        cmp edx,ebp
        jge short @@exit
        mov ecx,[esp]
@@next_x:
        cmp ecx,esi
        jge short @@linedone
        getpixeltobuf ecx,edx,edi, ebx,eax,al
@@getpixeldone:
        inc edi
        inc ecx
        jmp short @@next_x
@@linedone:
        inc edx
        jmp short @@next_y

@@exit:
        add esp,4
        popad
        ret

;
; Print '$' terminated string via DOS
; In:
;   ds = data32sel
;   edx = offset of '$' terminated string
;
dosprintx:
        push eax ecx esi edi es
        mov ax,ds                       ; First, get string length
        mov es,ax                       ; es=ds for scasb
        mov edi,edx
        mov ecx,0ffffffffh              ; Search max 4GB
        mov al,'$'                      ; Scan for '$' (=end of DOS string)
        cld
        repne scasb                     ; Compare al with es:edi
        not ecx
        mov eax,ecx                     ; ecx=string length (including '$')
        push lomembase
        call getlomem
        mov esi,edx
        mov edi,eax
        add edi,code32a
        mov es,zerosel
        cld
        rep movsb
        mov edx,eax
        call dosprint
        pop lomembase
        pop es edi esi ecx eax
        ret

code32  ends

RMseg   segment para public use16
        assume cs:RMseg

RMmouseX        dw      0
RMmouseY        dw      0
RMbutton        dw      0               ; mouse button status

mouse_sr:
        mov cs:RMbutton,bx              ; bx=button status
        mov cs:RMmouseX,cx              ; cx=x coordinate
        mov cs:RMmouseY,dx              ; dx=y coordinate
        retf

RMsegend        =       $
RMseg   ends

        end     main
