/* --------------------------------- debug.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Routines that get used only for debug and portability.
*/

#include <stdio.h>
#include <stdlib.h>

#ifndef	DEBUG_MULDIV
#define	DEBUG_MULDIV
#endif

#include "fly.h"

#ifdef	DEBUG_MULDIV

static int	muldiv_flag = 0;

extern long FAR
lcheck (long x, char *name, char *file, int line)
{
	if (x > (long)0x00007fffL) {
		muldiv_flag = 1;
		LogPrintf ("check> %s(%u) %s>%ld\n", file, line, name, x);
		++st.stats[42];
		return (0x00007fffL);
	}
	if (x < (long)0xffff8000L) {
		muldiv_flag = 2;
		LogPrintf ("check> %s(%u) %s<%ld\n", file, line, name, x);
		++st.stats[42];
		return (0xffff8000L);
	}
	return (x);
}

extern int FAR
muldivchk (long x, long y, long z, char *file, int line)
{
	long	r, tx, ty, tz, tr;
	int	t;

	t = muldiv_flag;
	muldiv_flag = 0;

	if (z == 0L) {
		muldiv_flag = 3;
		LogPrintf ("check> %s(%u) z=%ld\n", file, line, z);
		++st.stats[42];
		return (0x7fff);
	}
	tr=r = ((tx=x) * (ty=y)) / (tz=z);
	x = lcheck (x, "x", file, line);
	y = lcheck (y, "y", file, line);
	z = lcheck (z, "z", file, line);
	r = lcheck (r, "r", file, line);
	if (muldiv_flag)
		LogPrintf ("x %ld y %ld z %ld md %ld\n", tx, ty, tz, tr);
	else
		muldiv_flag = t;
	return ((int)r);
}

extern int FAR
fmulchk (long x, long y, char *file, int line)
{
	long	r, tx, ty, tr;
	int	t;

	t = muldiv_flag;
	muldiv_flag = 0;

	tr=r = ((tx=x) * (ty=y)) >> FSCALE;
	x = lcheck (x, "x", file, line);
	y = lcheck (y, "y", file, line);
	r = lcheck (r, "r", file, line);
	if (muldiv_flag)
		LogPrintf ("x %ld y %ld fm %ld\n", tx, ty, tr);
	else
		muldiv_flag = t;
	return ((int)r);
}
#endif	/*DEBUG_MULDIV*/

/*
 *	cy*cz		cy*sz		-sy
 *	sx*sy*cz-cx*sz	sx*sy*sz+cx*cz	sx*cy
 *	cx*sy*cz+sx*sz	cx*sy*sz-sx*cz	cx*cy
 *
 *	tt1 = cx*sz
 *	tt2 = sx*cz
 *	tt3 = cx*cz
 *	tt4 = sx*sz
 *
 *	cy*cz		cy*sz		-sy
 *	sy*tt2-tt1	sy*tt4+tt3	sx*cy
 *	sy*tt3+tt4	sy*tt1-tt2	cx*cy
 *
 */

extern void FAR
cbuild_matxyz (MAT T, int spitch, int cpitch, int sroll, int croll,
	int syaw, int cyaw)
{
	int	tt1, tt2, tt3, tt4;
						/* x = pitch	*/
						/* y = roll	*/
						/* z = yaw	*/

	tt1 = fmul (syaw, cpitch);		/* sz*cx	*/
	tt2 = fmul (cyaw, spitch);		/* cz*sx	*/
	tt3 = fmul (cyaw, cpitch);		/* cz*cx	*/
	tt4 = fmul (syaw, spitch);		/* sz*sx	*/

	T[0][0] = fmul (cyaw, croll);		/* cz*cy	*/
	T[0][1] = fmul (syaw, croll);		/* sz*cy	*/
	T[0][2] = -sroll;			/* -sy		*/

	T[1][0] = fmul (tt2, sroll) - tt1;	/* sy*tt2-tt1	*/
	T[1][1] = fmul (tt4, sroll) + tt3;	/* sy*tt4+tt3	*/
	T[1][2] = fmul (croll, spitch);		/* cy*sx	*/

	T[2][0] = fmul (tt3, sroll) + tt4;	/* sy*tt3+tt4	*/
	T[2][1] = fmul (tt1, sroll) - tt2;	/* sy*tt1-tt2	*/
	T[2][2] = fmul (cpitch, croll);		/* cx*cy	*/
}

/*
 *	cy*cz-sy*sx*sz	cy*sz+sy*sx*cz	-sy*cx
 *	-cx*sz		cx*cz		sx
 *	sy*cz+cy*sx*sz	sy*sz-cy*sx*cz	cy*cx
 *
 *	tt1 = cy*sz
 *	tt2 = sy*cz
 *	tt3 = cy*cz
 *	tt4 = sy*sz
 *
 *	tt3-tt4*sx	tt1+tt2*sx	-sy*cx
 *	-cx*sz		cx*cz		sx
 *	tt2+tt1*sx	tt4-tt3*sx	cy*cx
 */

extern void FAR
cbuild_matyxz (MAT T, int sx, int cx, int sy, int cy, int sz, int cz)
{
	int	tt1, tt2, tt3, tt4;
						/* x = pitch	*/
						/* y = roll	*/
						/* z = yaw	*/

	tt1 = fmul (cy, sz);
	tt2 = fmul (sy, cz);
	tt3 = fmul (cy, cz);
	tt4 = fmul (sy, sz);

	T[0][0] = tt3 - fmul (tt4, sx);
	T[0][1] = tt1 + fmul (tt2, sx);
	T[0][2] = -fmul (sy, cx);

	T[1][0] = -fmul (cx, sz);
	T[1][1] = fmul (cx, cz);
	T[1][2] = sx;

	T[2][0] = tt2 + fmul (tt1, sx);
	T[2][1] = tt4 - fmul (tt3, sx);
	T[2][2] = fmul (cy, cx);
}

